/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.blockmanagement;

import com.google.common.annotations.VisibleForTesting;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.UnderReplicatedBlockDataAccess;
import io.hops.metadata.hdfs.entity.BlockInfoProjected;
import io.hops.metadata.hdfs.entity.UnderReplicatedBlock;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;

public class HopsFSRestore {
    public static final Log LOG = LogFactory.getLog(HopsFSRestore.class);
    private CloudPersistenceProvider cloudConnector;
    private ExecutorService executorService;
    private Map<BlockIDAndGSTuple, BlockInfoProjected> dbBlocks;
    private Map<BlockIDAndGSTuple, CloudBlock> cloudBlocks;
    private final List<String> buckets;
    private final int prefixSize;

    public HopsFSRestore(Configuration conf) throws IOException {
        this.cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
        this.executorService = Executors.newFixedThreadPool(conf.getInt("dfs.cloud.backup.restore.threads", 30));
        this.buckets = CloudHelper.getBucketsFromConf(conf);
        if (this.buckets.size() != 1) {
            throw new IllegalStateException("There should be exactly one bucket/container configured in hdfs-site.xml");
        }
        this.prefixSize = conf.getInt("dfs.cloud.prefix.size", 500);
    }

    public RestoreStats compareAndFixState() throws IOException, ExecutionException, InterruptedException {
        ArrayList<Future<Object>> futures;
        if (!this.cloudConnector.isVersioningSupported(this.buckets.get(0))) {
            throw new UnsupportedOperationException("HopsFS can not restore data as the bucket/container does not support Versioning.");
        }
        RestoreStats stats = new RestoreStats();
        this.cloudBlocks = this.getExistingBlocksFromCloud();
        this.dbBlocks = this.getExistingBlocksFromDB();
        long startTime = System.currentTimeMillis();
        long matching = 0L;
        Iterator<BlockIDAndGSTuple> iter = this.dbBlocks.keySet().iterator();
        while (iter.hasNext()) {
            BlockIDAndGSTuple blkID = iter.next();
            if (!this.cloudBlocks.containsKey(blkID)) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("HopsFS-Cloud. Block ID: " + blkID + " exists in both DB and Cloud "));
            }
            iter.remove();
            this.cloudBlocks.remove(blkID);
            ++matching;
        }
        stats.setMatching(matching);
        LOG.info((Object)("HopsFS-Cloud. " + matching + " blocks exists in both the DB and Cloud. Time taken: " + (System.currentTimeMillis() - startTime) + " ms."));
        if (this.dbBlocks.size() > 0) {
            LOG.info((Object)("HopsFS-Cloud. " + this.dbBlocks.size() + " blocks left unaccounted for. Recovering blocks ..."));
            startTime = System.currentTimeMillis();
            futures = new ArrayList(this.dbBlocks.size());
            for (BlockIDAndGSTuple id : this.dbBlocks.keySet()) {
                futures.add(this.executorService.submit(new BlockRecoverer(id)));
            }
            long recovered = 0L;
            for (Future future : futures) {
                BlockIDAndGSTuple blkID = (BlockIDAndGSTuple)future.get();
                if (blkID.getBlockID() == Block.NON_EXISTING_BLK_ID) continue;
                this.dbBlocks.remove(blkID);
                this.cloudBlocks.remove(blkID);
                ++recovered;
            }
            stats.setRecovered(recovered);
            LOG.info((Object)("HopsFS-Cloud. " + recovered + " blocks successfully recovered. Time taken: " + (System.currentTimeMillis() - startTime) + " ms."));
            stats.setMissingFromCloud(this.dbBlocks.size());
            if (this.dbBlocks.size() > 0) {
                LOG.warn((Object)("HopsFS-Cloud. " + this.dbBlocks.size() + " blocks are missing from cloud."));
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("HopsFS-Cloud. Missing blocks IDs : " + Arrays.toString(this.dbBlocks.keySet().toArray())));
                }
                this.handleMissingBlocks(this.dbBlocks);
            }
        }
        if (this.cloudBlocks.size() > 0) {
            futures = new ArrayList<Future<Object>>(this.cloudBlocks.size());
            startTime = System.currentTimeMillis();
            long deleted = 0L;
            for (BlockIDAndGSTuple blockIDAndGSTuple : this.cloudBlocks.keySet()) {
                CloudBlock block = this.cloudBlocks.get(blockIDAndGSTuple);
                futures.add(this.executorService.submit(new BlockDeleter(block.getBlock())));
            }
            for (Future future : futures) {
                future.get();
                ++deleted;
            }
            stats.setExcessCloudBlocks(deleted);
            LOG.info((Object)("HopsFS-Cloud. " + deleted + " blocks successfully deleted from Cloud. Time taken: " + (System.currentTimeMillis() - startTime) + " ms."));
        }
        return stats;
    }

    private void handleMissingBlocks(Map<BlockIDAndGSTuple, BlockInfoProjected> dbBlocks) throws ExecutionException, InterruptedException {
        long startTime = System.currentTimeMillis();
        ArrayList futures = new ArrayList(this.cloudBlocks.size());
        for (BlockInfoProjected blockInfoProjected : dbBlocks.values()) {
            futures.add(this.executorService.submit(new MissingBlockHandler(blockInfoProjected)));
        }
        for (Future future : futures) {
            future.get();
        }
        LOG.info((Object)("HopsFS-Cloud. " + dbBlocks.size() + " blocks successfully added to missing list. Time taken: " + (System.currentTimeMillis() - startTime) + " ms."));
    }

    private Map<BlockIDAndGSTuple, BlockInfoProjected> getExistingBlocksFromDB() throws IOException {
        long startTime = System.currentTimeMillis();
        Map blocksMap = (Map)new LightWeightRequestHandler(HDFSOperationType.GET_ALL_PROVIDED_BLOCKS){

            public Object performTask() throws IOException {
                BlockInfoDataAccess da = (BlockInfoDataAccess)HdfsStorageFactory.getDataAccess(BlockInfoDataAccess.class);
                return da.getAllProvidedBlocksIDs();
            }
        }.handle();
        LOG.info((Object)("HopsFS-Cloud: Reading all the blocks ( Size: " + blocksMap.size() + " ) from DB took " + (System.currentTimeMillis() - startTime) + " ms."));
        return blocksMap;
    }

    @VisibleForTesting
    public Map<BlockIDAndGSTuple, CloudBlock> getExistingBlocksFromCloud() throws IOException {
        long startTime = System.currentTimeMillis();
        List<String> dirs = this.cloudConnector.getAllHopsFSDirectories(this.buckets);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("HopsFS-Cloud. Total Prefixes : " + dirs));
        }
        HashMap<BlockIDAndGSTuple, CloudBlock> cloudBlocks = new HashMap<BlockIDAndGSTuple, CloudBlock>();
        ArrayList<Future<Map<BlockIDAndGSTuple, CloudBlock>>> futures = new ArrayList<Future<Map<BlockIDAndGSTuple, CloudBlock>>>();
        for (String string : dirs) {
            futures.add(this.executorService.submit(new ListCloudPrefixs(string)));
        }
        for (Future future : futures) {
            try {
                Map blocksSubSet = (Map)future.get();
                cloudBlocks.putAll(blocksSubSet);
            }
            catch (ExecutionException e) {
                LOG.error((Object)"Exception was thrown during listing cloud storage", (Throwable)e);
                Throwable throwable = e.getCause();
                if (throwable instanceof IOException) {
                    throw (IOException)throwable;
                }
                throw new IOException(e);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        LOG.info((Object)("HopsFS-Cloud: Reading all the blocks ( Size: " + cloudBlocks.size() + " ) from cloud took " + (System.currentTimeMillis() - startTime) + " ms."));
        return cloudBlocks;
    }

    private boolean recoverDeletedBlock(long blockID, long gs) throws IOException {
        Block blockTmp = new Block(blockID);
        blockTmp.setGenerationStampNoPersistance(gs);
        return this.cloudConnector.restoreDeletedBlock(this.buckets.get(0), CloudHelper.getBlockKey(this.prefixSize, blockTmp)) && this.cloudConnector.restoreDeletedBlock(this.buckets.get(0), CloudHelper.getMetaFileKey(this.prefixSize, blockTmp));
    }

    private void deleteCloudBlock(Block block) throws IOException {
        this.cloudConnector.deleteObject(this.buckets.get(0), CloudHelper.getBlockKey(this.prefixSize, block));
        this.cloudConnector.deleteObject(this.buckets.get(0), CloudHelper.getMetaFileKey(this.prefixSize, block));
    }

    public class RestoreStats {
        private long matching;
        private long recovered;
        private long missingFromCloud;
        private long excessCloudBlocks;

        public long getMatching() {
            return this.matching;
        }

        public void setMatching(long matching) {
            this.matching = matching;
        }

        public long getRecovered() {
            return this.recovered;
        }

        public void setRecovered(long recovered) {
            this.recovered = recovered;
        }

        public long getMissingFromCloud() {
            return this.missingFromCloud;
        }

        public void setMissingFromCloud(long missingFromCloud) {
            this.missingFromCloud = missingFromCloud;
        }

        public long getExcessCloudBlocks() {
            return this.excessCloudBlocks;
        }

        public void setExcessCloudBlocks(long excessCloudBlocks) {
            this.excessCloudBlocks = excessCloudBlocks;
        }

        public String toString() {
            return "Matching: " + this.matching + " Recoverd: " + this.recovered + " Missing from cloud:" + this.missingFromCloud + " Excess cloud blocks: " + this.excessCloudBlocks;
        }
    }

    private class MissingBlockHandler
    implements Callable {
        private final BlockInfoProjected block;

        MissingBlockHandler(BlockInfoProjected block) {
            this.block = block;
        }

        public Object call() throws Exception {
            new LightWeightRequestHandler(HDFSOperationType.ADD_UNDER_REPLICATED_BLOCK){

                public Object performTask() throws IOException {
                    UnderReplicatedBlockDataAccess da = (UnderReplicatedBlockDataAccess)HdfsStorageFactory.getDataAccess(UnderReplicatedBlockDataAccess.class);
                    UnderReplicatedBlock urb = new UnderReplicatedBlock(4, MissingBlockHandler.this.block.getBlockId(), MissingBlockHandler.this.block.getInodeId(), 1);
                    ArrayList<UnderReplicatedBlock> newBlks = new ArrayList<UnderReplicatedBlock>();
                    newBlks.add(urb);
                    da.prepare((Collection)Collections.EMPTY_LIST, newBlks, (Collection)Collections.EMPTY_LIST);
                    return null;
                }
            }.handle();
            return null;
        }
    }

    private class BlockDeleter
    implements Callable {
        private final Block block;

        BlockDeleter(Block block) {
            this.block = block;
        }

        public Object call() throws Exception {
            HopsFSRestore.this.deleteCloudBlock(this.block);
            return null;
        }
    }

    private class BlockRecoverer
    implements Callable<BlockIDAndGSTuple> {
        private final BlockIDAndGSTuple id;

        BlockRecoverer(BlockIDAndGSTuple id) {
            this.id = id;
        }

        @Override
        public BlockIDAndGSTuple call() throws Exception {
            if (HopsFSRestore.this.recoverDeletedBlock(this.id.getBlockID(), this.id.getGs())) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("HopsFS-Cloud. Block ID: " + this.id + " recovered from an old version"));
                }
                return this.id;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("HopsFS-Cloud. Block ID: " + this.id + " Failed to recover from an old version"));
            }
            return new BlockIDAndGSTuple(Block.NON_EXISTING_BLK_ID, Block.NON_EXISTING_BLK_GS);
        }
    }

    private class ListCloudPrefixs
    implements Callable<Map<BlockIDAndGSTuple, CloudBlock>> {
        private final String prefix;

        ListCloudPrefixs(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public Map<BlockIDAndGSTuple, CloudBlock> call() throws Exception {
            return HopsFSRestore.this.cloudConnector.getAll(this.prefix, HopsFSRestore.this.buckets);
        }
    }
}

