/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.sync;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsClientConfigKeys;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudReadUnderConstructionBlock {

  static final Log LOG = LogFactory.getLog(TestCloudReadUnderConstructionBlock.class);
  static String testBucketPrefix = "hops-test-TCRUCB";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudReadUnderConstructionBlock(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  MiniDFSCluster startCluster(int numDNs, long blkSize) throws IOException {

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(FSNamesystem.class).setLevel(Level.DEBUG);
    Logger.getLogger(DFSClient.class).setLevel(Level.DEBUG);

    Configuration conf = new Configuration();
    conf.setBoolean(DFSConfigKeys.DFS_CLOUD_STORE_SMALL_FILES_IN_DB_KEY, false);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_MAX_ATTEMPTS_KEY, /*default 15*/ 1);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_RETRY_MAX_ATTEMPTS_KEY, /*default 10*/ 1);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_SLEEPTIME_BASE_KEY, /*default 500*/ 500);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_SLEEPTIME_MAX_KEY, /*default 15000*/1000);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_CONNECTION_RETRIES_KEY, /*default 0*/ 0);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_CONNECTION_RETRIES_ON_SOCKET_TIMEOUTS_KEY,
            /*default 0*/0);
    conf.setInt(DFSConfigKeys.IPC_CLIENT_CONNECT_MAX_RETRIES_ON_SOCKET_TIMEOUTS_KEY, /*default
    45*/ 2);
    conf.setInt(DFSConfigKeys.IPC_CLIENT_CONNECT_MAX_RETRIES_KEY, /*default 10*/ 1);
    conf.set(HdfsClientConfigKeys.Retry.POLICY_SPEC_KEY, "1000,2");

    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 99);
    conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, numDNs);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, blkSize);

    conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
    //if a datanode fails then the unfinished block report entry will linger for some time
    //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
    //able to block report. Reducing the BR Max process time to quickly reclaim
    //unfinished block reports
    conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5 * 1000);
    conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).numDataNodes(numDNs)
            .storageTypes(CloudTestHelper.genStorageTypes(numDNs)).format(true).build();
    cluster.waitActive();
    return cluster;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  /*
   */
  public void TestCloudReadUnderConstructionBlock() throws IOException {
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 512 * 1024 * 1024;
      final int NUM_DN = 3;

      cluster = startCluster(NUM_DN, BLKSIZE);
      DistributedFileSystem dfs = cluster.getFileSystem();
      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      FSDataOutputStream out = dfs.create(new Path("/dir/file"), (short) 1);
      int chunkSize = 64 * 1024;
      byte[] somedata = new byte[chunkSize];
      out.write(somedata);
      out.hsync();
      int read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + chunkSize, read == chunkSize);
      out.write(somedata);
      out.hsync();
      read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + chunkSize * 2, read == chunkSize * 2);
      out.write(somedata);
      out.hsync();
      read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + chunkSize * 3, read == chunkSize * 3);

      out.write(somedata);
      read = readData(cluster, "/dir/file");
      assert read >= chunkSize * 3 && read <= chunkSize * 4; // new data may or may not be visible

      String poolId = cluster.getNamesystem().getBlockPoolId();
      int dnToKill = -1;
      for (int i = 0; i < NUM_DN; i++) {
        if (cluster.getDataNodes().get(i).getFSDataset().getReplicaString(poolId, 0).compareTo("null") != 0) {
          dnToKill = i;
          LOG.info("Killing  " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath());
          break;
        }
      }

      //abort the client
      dfs.getClient().getLeaseRenewer().interruptAndJoin();
      dfs.getClient().abort();
      LOG.info("HopsFS-Cloud. Aborted the client");

      cluster.stopDataNode(dnToKill);
      waitDNCount(cluster, NUM_DN - 1);

      assert cluster.getNamesystem().getLeaseManager().countLease() == 1;

      read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + chunkSize * 3, read == chunkSize * 3);

      //recover lease and return
      cluster.setLeasePeriod(3 * 1000, 10 * 1000);

      long startTime = System.currentTimeMillis();
      while (true) {
        if ((System.currentTimeMillis() - startTime) < 60 * 1000) {
          if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
            break;
          }
        }
        Thread.sleep(1000);
      }
      assert cluster.getNamesystem().getLeaseManager().countLease() == 0;


      Thread.sleep(5000);

      read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + chunkSize * 3, read == chunkSize * 3);

      CloudTestHelper.matchMetadata(cluster.getConfiguration(0));
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  int readData(MiniDFSCluster cluster, String file) throws IOException {
    byte[] somedata = new byte[1024];
    DistributedFileSystem newdfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
    FSDataInputStream in = newdfs.open(new Path(file));
    int totRead = 0;
    int read = 0;
    do {
      read = in.read(somedata);
      if (read != -1) {
        totRead += read;
      }
    } while (read != -1);

    return totRead;
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
