/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.sync;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;
import java.util.Random;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudHsyncMTimes4 extends HsyncMTimesHelper {

  static final Log LOG = LogFactory.getLog(TestCloudHsyncMTimes4.class);
  static String testBucketPrefix = "hops-test-TCHMT4";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudHsyncMTimes4(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60 * 15);

  @Rule
  public Timeout timeout = Timeout.seconds(60 * 15);

  @Test
  public void TestHsyncMTimesX() throws IOException {
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 128 * 1024 * 1024;
      final int NUM_DN = 3;
      cluster = startCluster(testname.getMethodName(), defaultCloudProvider, testBucketPrefix, NUM_DN,
        BLKSIZE);
      DistributedFileSystem dfs = cluster.getFileSystem();
      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int initSize = 1024 * 1024 + 1;
      Random rand = new Random();
      int bytesWritten = 0;
      FSDataOutputStream out = dfs.create(new Path("/dir/file"));
      HopsFilesTestHelper.writeData(out, 0, initSize);
      out.close();
      bytesWritten += initSize;

      for (int i = 0; i < 10; i++) {
        int size = rand.nextInt(512 * 1024);
        LOG.info(i + " Rand: " + size);
        out = dfs.append(new Path("/dir/file"));

        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        size = 1;
        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        size = rand.nextInt(512 * 1024);
        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        out.close();
        LOG.info(i + " File closed");
      }

      verifyFile(dfs, "/dir/file", bytesWritten);

      for (int i = 0; i < 10; i++) {
        int size = rand.nextInt(512 * 1024);
        LOG.info(i + " Rand: " + size);
        out = dfs.append(new Path("/dir/file"));

        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        size = 1;
        HopsFilesTestHelper.writeData(out, bytesWritten, size);
        out.hflush();
        bytesWritten += size;

        out.close();
        LOG.info(i + " File closed");
      }
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
