/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.sync;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsClientConfigKeys;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import java.io.IOException;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;

public class AppendAndHflushClientFailuresHelper {

  static final Log LOG = LogFactory.getLog(AppendAndHflushClientFailuresHelper.class);

  MiniDFSCluster startCluster(String testname,
                              CloudProvider defaultCloudProvider, String testBucketPrefix,
                              int numDNs,
                              long blkSize) throws IOException {

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(FSNamesystem.class).setLevel(Level.DEBUG);
    Logger.getLogger(DFSClient.class).setLevel(Level.DEBUG);

    Configuration conf = new Configuration();
    conf.setBoolean(DFSConfigKeys.DFS_CLOUD_STORE_SMALL_FILES_IN_DB_KEY, false);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_MAX_ATTEMPTS_KEY, /*default 15*/ 1);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_RETRY_MAX_ATTEMPTS_KEY, /*default 10*/ 1);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_SLEEPTIME_BASE_KEY, /*default 500*/ 500);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_SLEEPTIME_MAX_KEY, /*default 15000*/1000);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_CONNECTION_RETRIES_KEY, /*default 0*/ 0);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_FAILOVER_CONNECTION_RETRIES_ON_SOCKET_TIMEOUTS_KEY,
      /*default 0*/0);
    conf.setInt(DFSConfigKeys.IPC_CLIENT_CONNECT_MAX_RETRIES_ON_SOCKET_TIMEOUTS_KEY, /*default
    45*/ 2);
    conf.setInt(DFSConfigKeys.IPC_CLIENT_CONNECT_MAX_RETRIES_KEY, /*default 10*/ 1);
    conf.set(HdfsClientConfigKeys.Retry.POLICY_SPEC_KEY, "1000,2");

    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 99);
    conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, numDNs);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, blkSize);

    conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
    //if a datanode fails then the unfinished block report entry will linger for some time
    //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
    //able to block report. Reducing the BR Max process time to quickly reclaim
    //unfinished block reports
    conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5 * 1000);
    conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).numDataNodes(numDNs)
      .storageTypes(CloudTestHelper.genStorageTypes(numDNs)).format(true).build();
    cluster.waitActive();
    return cluster;
  }

  public void testAppendAndHflushClientFailures(String testname,
                                                CloudProvider defaultCloudProvider, String testBucketPrefix,
                                                int blkSize, int initSize,
                                                int appendSize) throws IOException {
    MiniDFSCluster cluster = null;
    try {
      final int NUM_DN = 3;

      cluster = startCluster(testname, defaultCloudProvider, testBucketPrefix, NUM_DN, blkSize);
      DistributedFileSystem dfs = cluster.getFileSystem();
      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");
      cluster.setLeasePeriod(10 * 1000, 30 * 1000);

      writeFile(dfs, "/dir/file1", initSize);  // write to cloud
      int dataWritten = initSize;
      CloudTestHelper.matchMetadata(cluster.getConfiguration(0));

      LOG.info("Opening the file for append");
      for (int i = 0; i < 5; i++) {
        LOG.info("Iteration " + i);

        DistributedFileSystem newdfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
        FSDataOutputStream out = newdfs.append(new Path("/dir/file1"));
        HopsFilesTestHelper.writeData(out, dataWritten, appendSize);
        out.hsync();
        dataWritten += appendSize;

        HopsFilesTestHelper.writeData(out, dataWritten, appendSize);
        out.hflush();
        dataWritten += appendSize;

        // file lenght may or may not have been updated when hflush/hsync is called second time
        long length = dfs.getFileStatus(new Path("/dir/file1")).getLen();
        assert length >= (dataWritten - appendSize) && length <= dataWritten;

        // -----------kill the client -----------
        newdfs.getClient().getLeaseRenewer().interruptAndJoin();
        newdfs.getClient().abort();
        LOG.info("HopsFS-Cloud. Aborted the client");

        // ----------- wait for lease recovery -----------
        long startTime = System.currentTimeMillis();
        while (true) {
          if ((System.currentTimeMillis() - startTime) < 120 * 1000) {
            if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
              break;
            }
          }
          LOG.info("waiting for lease recovery");
          Thread.sleep(1000);
        }

        verifyFile(cluster.getNewFileSystemInstance(0), "/dir/file1", dataWritten);

        ProvidedBlocksChecker pbc =
          cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();
        long count = pbc.getProvidedBlockReportsCount();
        pbc.scheduleBlockReportNow();
        long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, count + 1);
        Thread.sleep(5000);

        verifyFile(cluster.getNewFileSystemInstance(0), "/dir/file1", dataWritten);
        CloudTestHelper.matchMetadata(cluster.getConfiguration(0));
      }


    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }
}
