/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.storages;

import com.amazonaws.services.s3.AmazonS3Client;
import com.azure.storage.blob.BlobServiceClient;
import com.lc.repackaged.com.google.cloud.storage.Storage;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.BlockMovedToColdStorageException;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.UUID;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudAllCloudOperations {

  static final Log LOG = LogFactory.getLog(TestCloudAllCloudOperations.class);

  static String testBucketPrefix = "hops-test-TCACO";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudAllCloudOperations(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Test
  public void TestCloudAllCloudOperations() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 1 * 1024 * 1024;
      final int FILESIZE = 1 * BLKSIZE;

      final String FILE_NAME1 = "/dir/TEST-FLIE1";
      final int NUM_DN = 3;
      String bucket = "";

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      conf.setBoolean(DFSConfigKeys.DFS_DN_CLOUD_BYPASS_CACHE_KEY, true);
      conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
      conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
      conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, NUM_DN);
      bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
              DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);

      CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      writeFile(dfs, FILE_NAME1, FILESIZE);
      moveBlocksToColdStorage(conf, cloud, bucket);

      BlockInfoContiguous blk =
              (BlockInfoContiguous) CloudTestHelper.findAllBlocks().values().iterator().next();
      String blk0ObjectKey = CloudHelper.getBlockKey(prefixSize, blk);
      String blk0MetaKey = CloudHelper.getMetaFileKey(prefixSize, blk);

      // this is expected to fail
      try {
        verifyFile(dfs, FILE_NAME1, FILESIZE);
        if(defaultCloudProvider != CloudProvider.GCS) {
          fail("reading the file should have failed");
        }
      } catch (IOException e) {
        if (!(e instanceof BlockMovedToColdStorageException &&
                defaultCloudProvider != CloudProvider.GCS)) {
          LOG.info(e, e);
          fail();
        }
      }

      // these operations are not expected to fail
      try {

        assert cloud.isVersioningSupported(bucket);

        cloud.getUserMetaData(bucket, blk0ObjectKey);
        cloud.objectExists(bucket, blk0ObjectKey);
        cloud.objectExists(bucket, blk0ObjectKey);
        assert cloud.getAll(CloudHelper.ROOT_PREFIX,
                CloudHelper.getBucketsFromConf(conf)).size() == CloudTestHelper.findAllBlocks().size();

        // delete a block in
        cloud.deleteObject(bucket, blk0ObjectKey);
        cloud.deleteObject(bucket, blk0MetaKey);


        cloud.restoreDeletedBlock(bucket, blk0ObjectKey);
        cloud.restoreDeletedBlock(bucket, blk0MetaKey);


        cloud.deleteObject(bucket, blk0ObjectKey);
        cloud.deleteObject(bucket, blk0MetaKey);
        cloud.deleteAllVersions(bucket, blk0ObjectKey);
        cloud.deleteAllVersions(bucket, blk0MetaKey);
      } catch (Exception e) {
        LOG.error(e, e);
        fail();
      }

    } catch (Exception e) {
      LOG.error(e, e);
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  private void moveBlocksToColdStorage(Configuration conf, CloudPersistenceProvider cloud,
                                       String bucket) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
      DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);

    Map<BlockIDAndGSTuple, BlockInfoContiguous> blocks = CloudTestHelper.findAllBlocks();
    for (BlockInfoContiguous blk : blocks.values()) {
      File tmpBlkFile = new File("/tmp/" + UUID.randomUUID());
      File tmpMetaFile = new File("/tmp/" + UUID.randomUUID());

      String blkKey = CloudHelper.getBlockKey(prefixSize, blk);
      String metaKey = CloudHelper.getMetaFileKey(prefixSize, blk);

      cloud.downloadObject(bucket, blkKey, tmpBlkFile);
      cloud.downloadObject(bucket, metaKey, tmpMetaFile);

      Map<String, String> blkMetadata = cloud.getUserMetaData(bucket, blkKey);
      Map<String, String> metaMetadata = cloud.getUserMetaData(bucket, metaKey);

      cloud.deleteObject(bucket, blkKey);
      cloud.deleteObject(bucket, metaKey);

      if (defaultCloudProvider == CloudProvider.AWS) {
        CloudTestHelper.moveToGlacier((AmazonS3Client) cloud.getCloudClient(), bucket,
          blkKey, tmpBlkFile, blkMetadata);
        CloudTestHelper.moveToGlacier((AmazonS3Client) cloud.getCloudClient(), bucket,
          metaKey, tmpMetaFile, metaMetadata);
      } else if (defaultCloudProvider == CloudProvider.AZURE) {
        CloudTestHelper.moveToArchive((BlobServiceClient) cloud.getCloudClient(), bucket,
          blkKey, tmpBlkFile, blkMetadata);
        CloudTestHelper.moveToArchive((BlobServiceClient) cloud.getCloudClient(), bucket,
          metaKey, tmpMetaFile, metaMetadata);
      } else if (defaultCloudProvider == CloudProvider.GCS) {
        CloudTestHelper.moveToColdLine((Storage) cloud.getCloudClient(), bucket,
          blkKey, tmpBlkFile, blkMetadata);
        CloudTestHelper.moveToColdLine((Storage) cloud.getCloudClient(), bucket,
          metaKey, tmpMetaFile, metaMetadata);
      } else {
        fail("Cloud provider not supported");
      }

      tmpBlkFile.delete();
      tmpMetaFile.delete();
    }
  }
  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
