/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.snapshots;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.server.blockmanagement.HopsFSRestore;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicLong;

import static org.apache.hadoop.hdfs.server.namenode.cloud.snapshots.TestCloudLargeNoOFBlock.createDummyBlocks;
import static org.apache.hadoop.hdfs.server.namenode.cloud.snapshots.TestCloudLargeNoOFBlock.setupDB;
import static org.junit.Assert.assertEquals;

@RunWith(Parameterized.class)
public class TestCloudPrefixes {

  static final Log LOG = LogFactory.getLog(TestCloudPrefixes.class);
  static String testBucketPrefix = "hops-test-TCP";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudPrefixes(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void TestCloudPrefixes() throws IOException, InterruptedException, ExecutionException {

    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.INFO);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    setupDB(conf);

    CloudPersistenceProvider cloudPersistenceProvider = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudPersistenceProvider.format(CloudHelper.getBucketsFromConf(conf));

    final int maxBlocks = 3000;
    final int prefixSize = 30;
    createDummyBlocks(maxBlocks, prefixSize, new AtomicLong(-1), cloudPersistenceProvider, conf,
            true, false);

    List<String> dirs = cloudPersistenceProvider.getAllHopsFSDirectories(CloudHelper.getBucketsFromConf(conf));
    LOG.info("Dirs are: " + Arrays.toString(dirs.toArray()));
    assertEquals(maxBlocks / prefixSize, dirs.size());

    HopsFSRestore hopsFSRestore = new HopsFSRestore(conf);
    assertEquals(maxBlocks, hopsFSRestore.getExistingBlocksFromCloud().size());
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
