/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.snapshots;

import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.HdfsVariables;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.CloudBucketDataAccess;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.HopsFSRestore;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudLargeNoOFBlock {

  static final Log LOG = LogFactory.getLog(TestCloudLargeNoOFBlock.class);
  static String testBucketPrefix = "hops-test-TCLNOB";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudLargeNoOFBlock(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void TestCloudLargeNoOFBlock() throws IOException, InterruptedException, ExecutionException {

    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.INFO);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    // ID: 0 --------------------------------- maxBlocks--------------------additionBlocks
    // DB: <------ DB HAS------------------------------>
    // S3: <--Deleted --><- Perm Del-><- S3 Has--------><--------------More blocks-------->

    //These are the blocks that both s3 and db has
    final long maxBlocks = 1000;

    // out of 'maxBlocks' these are the blocks that have been
    // deleted. Using versioning these blocks can be recovered
    final long deletedBlocs = maxBlocks / 2;


    // these are blocks that have been permanently deleted
    // that is, all versions of this block have been deleted
    long permanentlyDeletedBlocks = 100;
    if (defaultCloudProvider.equals(CloudProvider.AZURE)) {
      // when soft deletes are enabled then there is no way to
      // permanently delete objects before the retention perios is over
      permanentlyDeletedBlocks = 0;
    }

    // These are additional blocks that are in S3 but not in DB
    final long additionalBlocks = maxBlocks;
    final int prefixSize = 30;
    final long expectedMatchingBlks =
            (maxBlocks) - (deletedBlocs + permanentlyDeletedBlocks);
    final long expectedMissingFromCloud = permanentlyDeletedBlocks;
    final long expectedRecoveredBlocks = deletedBlocs;
    final long expectedExcessBlocks = additionalBlocks;


    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    conf.setInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, prefixSize);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
    conf.setInt(DFSConfigKeys.DFS_CLOUD_BACKUP_RESTORE_THREADS_KEY,
            DFSConfigKeys.DFS_CLOUD_BACKUP_RESTORE_THREADS_DEFAUlT);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    setupDB(conf);
    NameNode.format(conf);

    CloudPersistenceProvider cloudPersistenceProvider = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudPersistenceProvider.format(CloudHelper.getBucketsFromConf(conf));

    //These blocks are there in both S3 and DB
    LOG.info("< --- Creating " + maxBlocks + " --- >");
    createDummyBlocks(maxBlocks, prefixSize, new AtomicLong(-1), cloudPersistenceProvider, conf,
            false/*skipDB*/, false/*skipCloud*/);

    // delete some of the blocks form S3 so that they are restored later
    LOG.info("< --- Deleting " + deletedBlocs + " --- >");
    deleteS3Blocks(deletedBlocs, new AtomicLong(-1), cloudPersistenceProvider, conf, prefixSize,
            false/*delete versions*/);

    // permanently delet some othe blocks
    LOG.info("< --- Permanently Deleting " + permanentlyDeletedBlocks + " --- >");
    deleteS3Blocks(permanentlyDeletedBlocks, new AtomicLong(deletedBlocs), cloudPersistenceProvider, conf,
            prefixSize,
            true/*delete versions*/);

    // create excess blocks
    LOG.info("< --- Creatting Excess Blocks " + additionalBlocks + " --- >");
    HdfsVariables.incrementBlockIdCounter(maxBlocks);
    createDummyBlocks(additionalBlocks, prefixSize, new AtomicLong(maxBlocks), cloudPersistenceProvider,
            conf, true/*skipDB*/, false/*skipCloud*/);

    HopsFSRestore hopsFSRestore = new HopsFSRestore(conf);
    HopsFSRestore.RestoreStats stats = hopsFSRestore.compareAndFixState();
    LOG.info("Restore stats: "+stats);

    assertEquals(expectedMatchingBlks, stats.getMatching());
    assertEquals(expectedRecoveredBlocks, stats.getRecovered());
    assertEquals(expectedMissingFromCloud, stats.getMissingFromCloud());
    assertEquals(expectedExcessBlocks, stats.getExcessCloudBlocks());
    assertEquals(expectedMissingFromCloud, CloudTestHelper.findAllUnderReplicatedBlocks().size());
  }

  static void setupDB(Configuration conf) throws IOException {
    HdfsStorageFactory.setConfiguration(conf);
    HdfsStorageFactory.formatHdfsStorage();
    final LightWeightRequestHandler addBucket = new LightWeightRequestHandler(
            HDFSOperationType.TEST) {
      @Override
      public Object performTask() throws IOException {
        HdfsStorageFactory.getConnector().writeLock();
        CloudBucketDataAccess da = (CloudBucketDataAccess) HdfsStorageFactory
                .getDataAccess(CloudBucketDataAccess.class);
        da.addBucket(CloudHelper.getBucketsFromConf(conf).get(0));
        return null;
      }
    };
    addBucket.handle();
  }

  public void deleteS3Blocks(final long maxBlocksToDelete,
                             final AtomicLong initialBlockID,
                             final CloudPersistenceProvider cloudPersistenceProvider,
                             final Configuration conf, int prefixSize,
                             final boolean deleteVersions) throws InterruptedException {

    Thread[] threads = new Thread[10];
    AtomicInteger blocksDeleted = new AtomicInteger(0);
    for (int i = 0; i < threads.length; i++) {
      threads[i] = new Thread(new Runnable() {

        @Override
        public void run() {
          try {
            while (true) {
              long blocksDeletedSoFar = blocksDeleted.incrementAndGet();
              if (blocksDeletedSoFar <= maxBlocksToDelete) {
                long blockID = initialBlockID.incrementAndGet();
                Block b = new Block();
                b.setBlockIdNoPersistance(blockID);
                b.setGenerationStampNoPersistance(1);

                cloudPersistenceProvider.deleteObject(
                        CloudHelper.getBucketsFromConf(conf).get(0),
                        CloudHelper.getBlockKey(prefixSize, b));
                cloudPersistenceProvider.deleteObject(
                        CloudHelper.getBucketsFromConf(conf).get(0),
                        CloudHelper.getMetaFileKey(prefixSize, b));

                if (deleteVersions) {
                  cloudPersistenceProvider.deleteAllVersions(
                          CloudHelper.getBucketsFromConf(conf).get(0),
                          CloudHelper.getBlockKey(prefixSize, b));
                  cloudPersistenceProvider.deleteAllVersions(
                          CloudHelper.getBucketsFromConf(conf).get(0),
                          CloudHelper.getMetaFileKey(prefixSize, b));
                }

                if ((blocksDeletedSoFar % 500) == 0) {
                  LOG.info("Deleted " + blocksDeletedSoFar + " blocks so far");

                }

              } else {
                blocksDeleted.decrementAndGet();
                break;
              }
            }
          } catch (IOException e) {
            e.printStackTrace();
            fail();
          }
        }
      });
      threads[i].start();
    }
    for (int i = 0; i < threads.length; i++) {
      threads[i].join();
    }
    LOG.info(blocksDeleted + " Blocks Deleted ");

  }

  static void createDummyBlocks(final long maxBlocksToCreate, final int prefixSize,
                                 final AtomicLong initialBlockID,
                                 final CloudPersistenceProvider cloudPersistenceProvider,
                                 final Configuration conf,
                                 final boolean skipDB,
                                 final boolean skipCloud
  ) throws IOException,
          InterruptedException {
    //creat test file
    File emptyFile = new File("/tmp/" + UUID.randomUUID().toString());
    emptyFile.createNewFile();
    HashMap<String, String> metadata = new HashMap<>();

    AtomicInteger blocksCreated = new AtomicInteger(0);
    Thread[] threads = new Thread[10];
    for (int i = 0; i < threads.length; i++) {
      threads[i] = new Thread(new Runnable() {
        @Override
        public void run() {
          try {
            while (true) {
              int blocksCreatedSoFar = blocksCreated.incrementAndGet();
              if (blocksCreatedSoFar <= maxBlocksToCreate) {
                long blockID = initialBlockID.incrementAndGet();

                if (!skipCloud) {
                  Block dummyBlock = new Block();
                  dummyBlock.setGenerationStampNoPersistance(1);
                  dummyBlock.setBlockIdNoPersistance(blockID);
                  cloudPersistenceProvider.uploadObject(
                          CloudHelper.getBucketsFromConf(conf).get(0),
                          CloudHelper.getMetaFileKey(prefixSize, dummyBlock)
                          , emptyFile,
                          metadata);
                  cloudPersistenceProvider.uploadObject(
                          CloudHelper.getBucketsFromConf(conf).get(0),
                          CloudHelper.getBlockKey(prefixSize, dummyBlock)
                          , emptyFile,
                          metadata);

                }

                if (!skipDB) {
                  final LightWeightRequestHandler addblock = new LightWeightRequestHandler(
                          HDFSOperationType.TEST) {
                    @Override
                    public Object performTask() throws IOException {
                      HdfsStorageFactory.getConnector().writeLock();
                      BlockInfoDataAccess da = (BlockInfoDataAccess) HdfsStorageFactory
                              .getDataAccess(BlockInfoDataAccess.class);
                      BlockInfoContiguous bic = new BlockInfoContiguous();
                      bic.setBlockIdNoPersistance(blockID);
                      bic.setGenerationStampNoPersistance(1);
                      bic.setCloudBucketNoPersistance(CloudHelper.getBucketsFromConf(conf).get(0));
                      List<BlockInfoContiguous> newblks = new ArrayList<BlockInfoContiguous>();
                      newblks.add(bic);
                      da.prepare(Collections.EMPTY_LIST, newblks, Collections.EMPTY_LIST);
                      return null;
                    }
                  };
                  addblock.handle();

                }

                if ((blocksCreatedSoFar % 500) == 0) {
                  LOG.info("Created " + blocksCreatedSoFar + " blocks so far");

                }

              } else {
                blocksCreated.decrementAndGet();
                break;
              }
            }
          } catch (IOException e) {
            e.printStackTrace();
            fail(e.getMessage());
          }
        }
      });
      threads[i].start();
    }

    for (int i = 0; i < threads.length; i++) {
      threads[i].join();
    }
    LOG.info(blocksCreated + " Blocks Created ");
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
