/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.snapshots;

import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.BlockMissingException;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.DatanodeID;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeManager;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksCheckerFaultInjector;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.ProvidedReplicaBeingWritten;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInfo;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures;
import org.apache.hadoop.test.PathUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;

import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import java.io.File;

import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.Callable;

import static org.junit.Assert.*;
import static org.mockito.Matchers.any;

@RunWith(Parameterized.class)
public class TestCloudHSYNCBlkMetaFileUploadOrder {

  static final Log LOG = LogFactory.getLog(TestCloudHSYNCBlkMetaFileUploadOrder.class);
  static String testBucketPrefix = "hops-test-TCHSYC";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudHSYNCBlkMetaFileUploadOrder(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60 * 15);

  @Rule
  public Timeout timeout = Timeout.seconds(60 * 15);

  final int buffer_length = 128 * 1024;
  final int BLKSIZE = 8 * 1024 * 1024;
  int prefixSize = DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT;
  boolean spySleepLoc1 = false;
  boolean spySleepLoc2 = false;
  int spySleepDuration = 10000;

  private void setupConf(Configuration conf) throws IOException {

    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_MAX_BLOCK_ACQUIRE_FAILURES_KEY, 0);


    // dead datanodes
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 2000);
    conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 2L);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_REPLICATION_PENDING_TIMEOUT_SEC_KEY, 2);
    //conf.setInt(DFSConfigKeys.DFS_CLIENT_SOCKET_TIMEOUT_KEY, 10000);

    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    createEmptyExcludeFile(conf);
  }


  /*
   * In old code
   * ===========
   * On hsync/flush when we uploaded a block, we first deleted the old version of the block,
   * and then uploaded the new version of the block. If another client tried to read the block
   * in parallel then it was possible that it would not find the block as the other client had
   * deleted all the versions of the block, and it was trying to upload the new version.
   * In this test first we test that indeed this is the case.
   *
   * The fix is simple, we first upload the new version of the block and then delete the old
   * versions.
   */

  @Test
  public void TestUploadOrderReadFromNonCacheDN1() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    CloudPersistenceProvider cloud = null;
    final int NUM_DN = 10;

    // set the configuration
    Configuration conf = new Configuration();
    setupConf(conf);
    prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);

    MiniDFSCluster cluster = null;
    ExecutorService executor = Executors.newFixedThreadPool(1);
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      // both block and meta files are missing
      spySleepLoc1 = false;
      spySleepLoc2 = false;
      testUploadOrderReadFromNonCacheDNOldCode(conf, cluster, dfs, true, false, NUM_DN);

      // meta file is missing
      spySleepLoc1 = false;
      spySleepLoc2 = false;
      testUploadOrderReadFromNonCacheDNOldCode(conf, cluster, dfs, false, true, NUM_DN);


      spySleepLoc1 = false;
      spySleepLoc2 = false;
      testUploadOrderReadFromNonCacheDNNewCode(conf, cluster, dfs, true, NUM_DN);
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }

  /*
   * In old code
   * ===========
   * Similar to previous test. Testing old code with block reports.
   * On hsync/flush when we uploaded a block, we first deleted the old version of the block,
   * and then uploaded the new version of the block. If a block report is triggers
   * after the block is deleted then it will see that block is missing from the bucket and
   * mark it as corrupt.
   *
   * The fix is simple, we first upload the new version of the block and then delete the old
   * versions.
   */

  @Test
  public void TestHsyncWithBlockReports() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    CloudPersistenceProvider cloud = null;
    final int NUM_DN = 1;

    // set the configuration
    Configuration conf = new Configuration();
    setupConf(conf);
    prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);

    MiniDFSCluster cluster = null;
    ExecutorService executor = Executors.newFixedThreadPool(1);
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      // both block and meta files are missing
      testUploadOrderWithBlockReport(conf, cluster, dfs);

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }


  public void testUploadOrderReadFromNonCacheDNOldCode(Configuration conf,
                                                       MiniDFSCluster cluster,
                                                       DistributedFileSystem dfs,
                                                       boolean spySleepLoc1,
                                                       boolean spySleepLoc2,
                                                       int NUM_DN) throws Exception {
    ExecutorService executor = Executors.newFixedThreadPool(1);
    String fileName = "/file";
    AsyncWriter asynWriter = new AsyncWriter(dfs, fileName);
    Future asynWriterFuture = executor.submit(asynWriter);

    while (!asynWriter.firstSyncDone()) {
      Thread.sleep(50);
    }

    // Error injection. Install spy to slowdown upload
    LOG.info("Installing spy");
    installSpyOldCode(cluster, conf);

    long blockID = (dfs.getClient().getLocatedBlocks(fileName, 0).getLastLocatedBlock().getBlock().getBlockId());

    //Enable error injection. Force block manager to return a DN that does not have the cached block
    String DNUUID = findDNUUIDForBlock(cluster, conf, blockID, NUM_DN);
    cluster.getNamesystem().getBlockManager().errInjIgnoreCloudCache(true, DNUUID);

    FSDataInputStream in = dfs.open(new Path(fileName));
    LOG.info("File opened for reading");

    this.spySleepLoc1 = spySleepLoc1; // sleep after deleting the block and meta block's all versions
    this.spySleepLoc2 = spySleepLoc2; // sleep after uploading the block file. Meta file is uploaded after the pause

    // resume
    asynWriter.resume();
    Thread.sleep(2000);

    try {
      byte[] buffer = new byte[1024];
      int read = 0;
      read = in.read(0, buffer, 0, 1024);
      in.close();
      LOG.info(read + " bytes read");
      fail("Expecting BlockMissingException");
    } catch (BlockMissingException e) {
      //expected
    }

    asynWriterFuture.get();

    //uninstall spy
    LOG.info("Uninstalling spy");
    unInstallSpy(cluster, conf);

    //Disable error injection.
    cluster.getNamesystem().getBlockManager().errInjIgnoreCloudCache(false, "");

    int read = readData(cluster, fileName);
    assert read == 3 * buffer_length;
  }


  public void testUploadOrderWithBlockReport(Configuration conf,
                                             MiniDFSCluster cluster,
                                             DistributedFileSystem dfs) throws Exception {
    String fileName = "/file";
    ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

    LOG.info("Installing error injector");
    installPBCFaultInjector();

    byte buffer[] = new byte[buffer_length];
    FSDataOutputStream out = dfs.create(new Path(fileName));
    for (int i = 0; i < 1; i++) {
      out.write(buffer);
      out.hsync();
    }

    // The block report first reads the blocks from the cloud and then from the
    // DB. Testing the scenario where the PBC reads the blocks list from cloud.
    // The client then writes more data to the block and then closes the block.
    // The cloud view will not match with the database view.

    // Trigger block report
    long brCount = pbc.getProvidedBlockReportsCount();
    pbc.scheduleBlockReportNow();
    waitForBRToStartErrorInj();

    for (int i = 0; i < 1; i++) {
      out.write(buffer);
      out.hsync();
    }

    out.close();

    long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, brCount + 1);
    assertTrue("Exptected " + brCount + 1 + " Got: " + ret, (brCount + 1) == ret);

    // expecting a wrongly identified corrupt block
    assert CloudTestHelper.findAllUnderReplicatedBlocks().size() == 0;


    // one more time. in the previous block report the block was in
    // underconstruction state. Retry again with block closed.
    brCount = pbc.getProvidedBlockReportsCount();
    pbc.scheduleBlockReportNow();
    waitForBRToStartErrorInj();
    ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, brCount + 1);
    assertTrue("Exptected " + brCount + 1 + " Got: " + ret, (brCount + 1) == ret);


    LOG.info("Uninstalling error injector");
    unInstallPBCFaultInjector();

    // read is expected to work as the block is not actually corrupt
    int read = readData(cluster, fileName);
    assert read == 2 * buffer_length;
  }

  public void waitForBRToStartErrorInj() throws InterruptedException {
    for (int i = 0; i < 100; i ++){
      if (failureInjectorDelayInprogress) {
        return;
      } else {
        Thread.sleep(100);
      }
    }
    fail("Failed. BR has not started");
  }


  public void testUploadOrderReadFromNonCacheDNNewCode(Configuration conf,
                                                       MiniDFSCluster cluster,
                                                       DistributedFileSystem dfs,
                                                       boolean spySleepLoc1,
                                                       final int NUM_DN) throws Exception {
    ExecutorService executor = Executors.newFixedThreadPool(1);
    String fileName = "/file";
    AsyncWriter asynWriter = new AsyncWriter(dfs, fileName);
    Future asynWriterFuture = executor.submit(asynWriter);

    while (!asynWriter.firstSyncDone()) {
      Thread.sleep(50);
    }

    // Error injection. Install spy to slowdown upload
    LOG.info("Installing spy");
    installSpyNewCode(cluster, conf);

    long blockID = (dfs.getClient().getLocatedBlocks(fileName, 0).getLastLocatedBlock().getBlock().getBlockId());

    //Enable error injection. Force block manager to return a DN that does not have the cached block
    String DNUUID = findDNUUIDForBlock(cluster, conf, blockID, NUM_DN);
    cluster.getNamesystem().getBlockManager().errInjIgnoreCloudCache(true, DNUUID);

    FSDataInputStream in = dfs.open(new Path(fileName));
    LOG.info("File opened for reading");

    this.spySleepLoc1 = spySleepLoc1; // sleep after uploading the block file. Meta file is uploaded after the pause

    // resume
    asynWriter.resume();
    Thread.sleep(2000);

    try {
      byte[] buffer = new byte[1024];
      int read = 0;
      read = in.read(0, buffer, 0, 1024);
      in.close();
      LOG.info(read + " bytes read");
    } catch (Exception e) {
      e.printStackTrace();
      fail("Did not expect any exception. Exception Got: " + e.getMessage());
    }

    asynWriterFuture.get();

    //uninstall spy
    LOG.info("Uninstalling spy");
    unInstallSpy(cluster, conf);

    //Disable error injection.
    cluster.getNamesystem().getBlockManager().errInjIgnoreCloudCache(false, "");

    int read = readData(cluster, fileName);
    assert read == 3 * buffer_length;
  }

  void markDNDead(MiniDFSCluster cluster, Configuration conf, int NUM_DN) throws IOException,
          InterruptedException {

    //find datanode to decommission
    String poolId = cluster.getNamesystem().getBlockPoolId();
    int dnToKill = -1;
    for (int i = 0; i < NUM_DN; i++) {
      if (cluster.getDataNodes().get(i).getFSDataset().getReplicaString(poolId, 0/*blockid*/).compareTo("null") != 0) {
        dnToKill = i;
        LOG.info("Choosing DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " to be marked daead");
        break;
      }
    }

    if (dnToKill != -1) {
      DataNode dn = cluster.getDataNodes().get(dnToKill);
      DatanodeID dnId = dn.getDatanodeId();
      final FSNamesystem ns = cluster.getNameNode().getNamesystem();
      final DatanodeManager dm = ns.getBlockManager().getDatanodeManager();
      dm.removeDatanode(dnId, false);
      LOG.info("Removed storages: " + Arrays.toString(dm.getSidsOnDatanode(dn.getDatanodeUuid()).toArray()));
    }
  }


  String findDNUUIDForBlock(MiniDFSCluster cluster, Configuration conf, long blockID,
                            int NUM_DN) throws IOException, InterruptedException {
    String poolId = cluster.getNamesystem().getBlockPoolId();
    int dnToKill = -1;
    for (int i = 0; i < NUM_DN; i++) {
      if (cluster.getDataNodes().get(i).getFSDataset().getReplicaString(poolId, blockID/*blockid*/).compareTo("null") != 0) {
        LOG.info("DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " stores block: " + blockID);
        return cluster.getDataNodes().get(i).getDatanodeUuid();
      }
    }
    fail("Failed to find DN that stores block: " + blockID);
    return "";
  }

  void killDN(MiniDFSCluster cluster, Configuration conf, int NUM_DN) throws IOException,
          InterruptedException {

    //find datanode to decommission
    String poolId = cluster.getNamesystem().getBlockPoolId();
    int dnToKill = -1;
    for (int i = 0; i < NUM_DN; i++) {
      if (cluster.getDataNodes().get(i).getFSDataset().getReplicaString(poolId, 0/*blockid*/).compareTo("null") != 0) {
        dnToKill = i;
        LOG.info("Choosing DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " to kill");
        break;
      }
    }

    if (dnToKill != -1) {
      DataNode dn = cluster.getDataNodes().get(dnToKill);
      DatanodeID dnId = dn.getDatanodeId();
      final FSNamesystem ns = cluster.getNameNode().getNamesystem();
      final DatanodeManager dm = ns.getBlockManager().getDatanodeManager();
      LOG.info("Removing storages: " + Arrays.toString(dm.getSidsOnDatanode(dn.getDatanodeUuid()).toArray()));
      cluster.stopDataNode(dnToKill);
      TestCloudDNFailures.waitDNCount(cluster, NUM_DN - 1);
    }
  }

  void decommissionDN(MiniDFSCluster cluster, Configuration conf, int NUM_DN) throws IOException,
          InterruptedException {

    //find datanode to decommission
    String poolId = cluster.getNamesystem().getBlockPoolId();
    int dnToDecommission = -1;
    for (int i = 0; i < NUM_DN; i++) {
      if (cluster.getDataNodes().get(i).getFSDataset().getReplicaString(poolId, 0/*blockid*/).compareTo("null") != 0) {
        dnToDecommission = i;
        LOG.info("Choosing DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " to decommission");
        break;
      }
    }

    String decommissionedDNName = "";
    if (dnToDecommission != -1) {
      decommissionedDNName =
              cluster.getDataNodes().get(dnToDecommission).getXferAddress().getHostString() + ":" +
                      cluster.getDataNodes().get(dnToDecommission).getXferAddress().getPort();
      LOG.info("Datanode to decommission " + decommissionedDNName);
      cluster.getNamesystem().updateExcludeList(decommissionedDNName, conf);
      cluster.getNamesystem().getBlockManager().getDatanodeManager().refreshNodes(conf);
    } else {
      fail("Unable to find DN to decommission");
    }

    int retries = 100;
    for (int i = 0; i < retries; i++) {
      if (cluster.getFileSystem().dfs.datanodeReport(HdfsConstants.DatanodeReportType.DECOMMISSIONING).length != 1) {
        Thread.sleep(1000);
        LOG.info("Waiting for datanode " + decommissionedDNName + " to decommission");
      } else {
        break;
      }
    }

    if (cluster.getFileSystem().dfs.datanodeReport(HdfsConstants.DatanodeReportType.DECOMMISSIONING).length != 1) {
      fail("Node did not decommission");
    }

    LOG.info("Decommissioning completed ");
  }

  void createEmptyExcludeFile(Configuration conf) throws IOException {
    FileSystem localFileSys = FileSystem.getLocal(conf);
    File workingDir = new File(localFileSys.getWorkingDirectory().toUri());
    File excludeFile = new File(workingDir.getAbsoluteFile() + PathUtils.getTestDirName(getClass()) + "/work-dir/decommission");
    if (excludeFile.exists()) {
      excludeFile.delete();
    }
    excludeFile.getParentFile().mkdirs();
    excludeFile.createNewFile();
    conf.set(DFSConfigKeys.DFS_HOSTS_EXCLUDE, excludeFile.getAbsolutePath());
  }

  CloudFsDatasetImpl[] cloudFsDatasetImpls;

  void installSpyOldCode(MiniDFSCluster cluster, Configuration conf) throws IOException {
    cloudFsDatasetImpls = new CloudFsDatasetImpl[cluster.getDataNodes().size()];
    for (int i = 0; i < cluster.getDataNodes().size(); i++) {
      cloudFsDatasetImpls[i] =
              ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset());
      CloudFsDatasetImpl mockedCloudFsDatasetImpl = Mockito.spy(cloudFsDatasetImpls[i]);
      final CloudFsDatasetImpl cloudImpl = cloudFsDatasetImpls[i];
      Answer newSyncFn = new Answer() {
        @Override
        public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
          CloudPersistenceProvider cloud = cloudImpl.getCloudConnector();
          boolean isVersioningSupported =
                  cloud.isVersioningSupported(CloudHelper.getBucketsFromConf(conf).get(0));
          ExtendedBlock b = (ExtendedBlock) invocationOnMock.getArguments()[0];
          ReplicaInfo replicaInfo = mockedCloudFsDatasetImpl.getReplicaInfo(b);

          File blockFile = replicaInfo.getBlockFile();
          File metaFile = replicaInfo.getMetaFile();

          String blockFileKey = CloudHelper.getBlockKey(prefixSize, b.getLocalBlock());
          String metaFileKey = CloudHelper.getMetaFileKey(prefixSize, b.getLocalBlock());

          if (replicaInfo instanceof ProvidedReplicaBeingWritten) {
            ((ProvidedReplicaBeingWritten) replicaInfo).setSynced(true);
            ((ProvidedReplicaBeingWritten) replicaInfo).setCancellMultipart(true);

            boolean isMultiPart = ((ProvidedReplicaBeingWritten) replicaInfo).isMultipart();
            if (isMultiPart) {
              ((ProvidedReplicaBeingWritten) replicaInfo).setMultipart(false);
              //abort the multipart for this block
              cloud.abortMultipartUpload(b.getCloudBucket(), blockFileKey,
                      ((ProvidedReplicaBeingWritten) replicaInfo).getUploadID());
            }
          }

          if (isVersioningSupported) {
            LOG.info("Spy versioning is supported");
            if (cloud.objectExists(b.getCloudBucket(), blockFileKey)) {
              cloud.deleteAllVersions(b.getCloudBucket(), blockFileKey);
            }

            if (cloud.objectExists(b.getCloudBucket(), metaFileKey)) {
              cloud.deleteAllVersions(b.getCloudBucket(), metaFileKey);
            }
          }

          if (spySleepLoc1) {
            LOG.info("Spy Sleeping. Loc 1");
            Thread.sleep(spySleepDuration);
            LOG.info("Spy woke up");
          }

          cloud.uploadObject(b.getCloudBucket(), metaFileKey, metaFile,
                  cloudImpl.getMetaMetadata(b.getLocalBlock(), metaFile, blockFile));

          if (spySleepLoc2) {
            LOG.info("Spy Sleeping. Loc 2");
            Thread.sleep(spySleepDuration);
            LOG.info("Spy woke up");
          }

          cloud.uploadObject(b.getCloudBucket(), blockFileKey, blockFile,
                  cloudImpl.getBlockFileMetadata(b.getLocalBlock()));
          return null;
        }
      };
      Mockito.doAnswer(newSyncFn).when(mockedCloudFsDatasetImpl).syncToCloud(any());
      cluster.getDataNodes().get(i).setFSDataset(mockedCloudFsDatasetImpl);
    }
  }

  ProvidedBlocksCheckerFaultInjector pbcFailureInjector;
  boolean failureInjectorDelayInprogress = false;
  void installPBCFaultInjector() throws IOException {
    ProvidedBlocksCheckerFaultInjector faultInjector
            = Mockito.mock(ProvidedBlocksCheckerFaultInjector.class);
    pbcFailureInjector = ProvidedBlocksCheckerFaultInjector.instance;
    ProvidedBlocksCheckerFaultInjector.instance = faultInjector;

    Answer sleep = new Answer() {
      @Override
      public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
        Map<BlockIDAndGSTuple, BlockInfoContiguous> dbBlocksMap = (Map<BlockIDAndGSTuple,
                BlockInfoContiguous>) invocationOnMock.getArguments()[0];
        if (dbBlocksMap.size() != 0) {
          failureInjectorDelayInprogress = true;
          Thread.sleep(10000);
          failureInjectorDelayInprogress = false;
        }
        return null;
      }
    };
    Mockito.doAnswer(sleep).when(faultInjector).errorAfterReadingBlocks(any());
  }

  void unInstallPBCFaultInjector() throws IOException {
    ProvidedBlocksCheckerFaultInjector.instance = pbcFailureInjector;
  }

  void installSpyNewCode(MiniDFSCluster cluster, Configuration conf) throws IOException {
    cloudFsDatasetImpls = new CloudFsDatasetImpl[cluster.getDataNodes().size()];
    for (int i = 0; i < cluster.getDataNodes().size(); i++) {
      cloudFsDatasetImpls[i] =
              ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset());
      CloudFsDatasetImpl mockedCloudFsDatasetImpl = Mockito.spy(cloudFsDatasetImpls[i]);
      final CloudFsDatasetImpl cloudImpl = cloudFsDatasetImpls[i];
      Answer newSyncFn = new Answer() {
        @Override
        public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
          CloudPersistenceProvider cloud = cloudImpl.getCloudConnector();
          boolean isVersioningSupported =
                  cloud.isVersioningSupported(CloudHelper.getBucketsFromConf(conf).get(0));
          ExtendedBlock b = (ExtendedBlock) invocationOnMock.getArguments()[0];
          ReplicaInfo replicaInfo = mockedCloudFsDatasetImpl.getReplicaInfo(b);

          File blockFile = replicaInfo.getBlockFile();
          File metaFile = replicaInfo.getMetaFile();

          String blockFileKey = CloudHelper.getBlockKey(prefixSize, b.getLocalBlock());
          String metaFileKey = CloudHelper.getMetaFileKey(prefixSize, b.getLocalBlock());

          if (replicaInfo instanceof ProvidedReplicaBeingWritten) {
            ((ProvidedReplicaBeingWritten) replicaInfo).setSynced(true);
            ((ProvidedReplicaBeingWritten) replicaInfo).setCancellMultipart(true);

            boolean isMultiPart = ((ProvidedReplicaBeingWritten) replicaInfo).isMultipart();
            if (isMultiPart) {
              ((ProvidedReplicaBeingWritten) replicaInfo).setMultipart(false);
              //abort the multipart for this block
              cloud.abortMultipartUpload(b.getCloudBucket(), blockFileKey,
                      ((ProvidedReplicaBeingWritten) replicaInfo).getUploadID());
            }
          }

          cloud.uploadObject(b.getCloudBucket(), metaFileKey, metaFile,
                  cloudImpl.getMetaMetadata(b.getLocalBlock(), metaFile, blockFile));

          if (spySleepLoc1) {
            LOG.info("New Code Spy Sleeping. Loc 1");
            Thread.sleep(spySleepDuration);
            LOG.info("Spy woke up");
          }

          cloud.uploadObject(b.getCloudBucket(), blockFileKey, blockFile,
                  cloudImpl.getBlockFileMetadata(b.getLocalBlock()));

          if (isVersioningSupported) {
            LOG.info("Spy versioning is supported");
            if (cloud.objectExists(b.getCloudBucket(), blockFileKey)) {
              cloud.deleteOldVersions(b.getCloudBucket(), blockFileKey);
            }

            if (cloud.objectExists(b.getCloudBucket(), metaFileKey)) {
              cloud.deleteOldVersions(b.getCloudBucket(), metaFileKey);
            }
          }

          return null;
        }
      };
      Mockito.doAnswer(newSyncFn).when(mockedCloudFsDatasetImpl).syncToCloud(any());
      cluster.getDataNodes().get(i).setFSDataset(mockedCloudFsDatasetImpl);
    }
  }


  void unInstallSpy(MiniDFSCluster cluster, Configuration conf) throws IOException {
    for (int i = 0; i < cluster.getDataNodes().size(); i++) {
      cluster.getDataNodes().get(i).setFSDataset(cloudFsDatasetImpls[i]);
    }

  }

  class AsyncWriter implements Callable {
    DistributedFileSystem dfs;
    String filePath;
    boolean resume = false;
    boolean firstSyncDone = false;

    AsyncWriter(DistributedFileSystem dfs, String filePath) {
      this.dfs = dfs;
      this.filePath = filePath;
    }

    void resume() {
      this.resume = true;
    }

    boolean firstSyncDone() {
      return firstSyncDone;
    }

    @Override
    public Object call() throws Exception {
      boolean flush = false;
      byte buffer[] = new byte[buffer_length];
      FSDataOutputStream out = dfs.create(new Path(filePath));

      out.write(buffer);
      if (flush) {
        out.hflush();
      } else {
        out.hsync();
      }
      firstSyncDone = true;

      LOG.info("Writer paused");
      while (!resume) {
        Thread.sleep(100);
      }
      LOG.info("Writer resumed");

      out.write(buffer);
      LOG.info("Writer hsync called");
      if (flush) {
        out.hflush();
      } else {
        out.hsync();
      }
      LOG.info("Writer hsync completed");
      out.write(buffer);
      out.close();
      return null;
    }
  }

  int readData(MiniDFSCluster cluster, String file) throws IOException {
    byte[] somedata = new byte[1024];
    DistributedFileSystem newdfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
    FSDataInputStream in = newdfs.open(new Path(file));
    int totRead = 0;
    int read = 0;
    do {
      read = in.read(somedata);
      if (read != -1) {
        totRead += read;
      }
    } while (read != -1);

    return totRead;
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
