/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.snapshots;

import com.amazonaws.services.s3.model.S3VersionSummary;
import com.lc.repackaged.com.google.cloud.storage.Blob;
import io.hops.exception.StorageException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.dal.LeasePathDataAccess;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.hdfs.*;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.HopsFSRestore;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.*;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudHSYNC {

  static final Log LOG = LogFactory.getLog(TestCloudHSYNC.class);
  static String testBucketPrefix = "hops-test-TCHSYC";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudHSYNC(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  //write a block and call hsync multiple times
  //make sure that the block has only one version
  //kill client and take a snapshot
  //continue. the file will be recovered by the lease manager
  //revert to old snapshot and the file will be recovered
  //again.
  @Test
  public void TestCloudHSYNC() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.TRACE);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int BLKSIZE = 8 * 1024 * 1024;
    final int NUM_DN = 3;
    final String BACKUP1 = "backup1";
    final String BACKUP2 = "backup2";
    CloudPersistenceProvider cloud = null;

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    int NUM_FILES = 10;
    byte buffer[] = new byte[128*1024];
    LOG.info("________________ Run 1 ________________");
    MiniDFSCluster cluster = null;
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      HdfsStorageFactory.clearBackupsTestingOnly();
      DistributedFileSystem dfs = cluster.getFileSystem();
      cloud =  CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      //write a file to the cloud

      FSDataOutputStream outs[] = new FSDataOutputStream[NUM_FILES];
      for (int i = 0; i < NUM_FILES; i++) {
        String fileName = "/files-run1/file" + i;
        outs[i] = dfs.create(new Path(fileName));
        outs[i].write(buffer);
        outs[i].hsync();
        outs[i].write(buffer);
        outs[i].hsync();
        outs[i].write(buffer);

        Thread.sleep(2000);
        int read = readData(cluster, fileName);
        assert read >= 2*buffer.length && read < 3*buffer.length;

        RemoteIterator<LocatedFileStatus> status  = dfs.listLocatedStatus(new Path(fileName));
        assert status.hasNext(); // there has to be a block
        while (status.hasNext()) {
          LocatedFileStatus locFileStatus = status.next();
          assert locFileStatus.getBlockLocations()[0].getLength() == buffer.length;
        }
      }

      // check for versions
      checkBlockVersions(cloud, conf, 1);

      dfs.getClient().getLeaseRenewer().interruptAndJoin();
      dfs.getClient().abort();
      LOG.info("HopsFS-Cloud. Aborted the client");

      assert countActiveLeases() == NUM_FILES;

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
        HdfsStorageFactory.backupTestingOnly(BACKUP1);
      }
    }


    LOG.info("________________ Run 2 ________________");
    // restart
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem();

      assertTrue("Expecting 5 block but found " + cluster.getNamesystem().getTotalBlocks(),
              cluster.getNamesystem().getTotalBlocks() == NUM_FILES);
      assert countActiveLeases() == NUM_FILES;

      // lease  recovery
      cluster.setLeasePeriod(3 * 1000, 10 * 1000);
      long startTime = System.currentTimeMillis();
      while (true) {
        if ((System.currentTimeMillis() - startTime) < 60 * 1000) {
          if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
            break;
          }
        }
        Thread.sleep(1000);
      }

      //make sure that block cache invalidation commands have been sent
      Thread.sleep(5000);

      for (int i = 0; i < NUM_FILES; i++) {
        String fileName = "/files-run1/file" + i;

        int read = readData(cluster, fileName);
        assert read == 2*buffer.length;

        RemoteIterator<LocatedFileStatus> status  = dfs.listLocatedStatus(new Path(fileName));
        assert status.hasNext(); // there has to be a block
        while (status.hasNext()) {
          LocatedFileStatus locFileStatus = status.next();
          assert locFileStatus.getBlockLocations()[0].getLength() == 2*buffer.length;
        }
      }

      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == NUM_FILES;
      assertTrue("Expecting "+NUM_FILES+" blocks but got " + cluster.getNamesystem().getTotalBlocks(),
              cluster.getNamesystem().getTotalBlocks() == NUM_FILES);

      // check versions
      checkBlockVersions(cloud, conf, 1);

      //overwrite
      for (int i = 0; i < NUM_FILES; i++) {
        String fileName = "/files-run1/file" + i;
        FSDataOutputStream out = dfs.create(new Path(fileName));
        out.writeBytes("somedata");
        out.close();
      }

      //wait for block deletion
      Thread.sleep(5000);

      checkBlockVersions(cloud, conf, 2);

      // all files are now in DB
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == 0;
      assertTrue("Expecting 5 in-memory file. Got: " + HopsFilesTestHelper.countInMemoryDBFiles()
              , HopsFilesTestHelper.countInMemoryDBFiles() == NUM_FILES);
      assertTrue("Expecting 0 block but found " + cluster.getNamesystem().getTotalBlocks(),
              cluster.getNamesystem().getTotalBlocks() == 0);


      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
        HdfsStorageFactory.backupTestingOnly(BACKUP2);
      }
    }

    LOG.info("________________ Run 3 ________________");
    // Restore
    HdfsStorageFactory.restoreTestingOnly(BACKUP1);
    NameNode.rollbackDeletedCloudBlocks(conf);

    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem();

      checkBlockVersions(cloud, conf, 1);

      //leases are active again
      assertTrue("Expecting 5 block but found " + cluster.getNamesystem().getTotalBlocks(),
              cluster.getNamesystem().getTotalBlocks() == NUM_FILES);
      assert countActiveLeases() == NUM_FILES;

      for (int i = 0; i < NUM_FILES; i++) {
        String fileName = "/files-run1/file" + i;
        RemoteIterator<LocatedFileStatus> status  = dfs.listLocatedStatus(new Path(fileName));
        assert status.hasNext(); // there has to be a block
        while (status.hasNext()) {
          LocatedFileStatus locFileStatus = status.next();
          assert locFileStatus.getBlockLocations()[0].getLength() == buffer.length;
        }

        int read = readData(cluster, fileName);
        assertTrue("Expected: "+2*buffer.length+" Got: "+read,  read == 2*buffer.length);

        //check versions
      }

      // lease  recovery
      cluster.setLeasePeriod(3 * 1000, 10 * 1000);
      long startTime = System.currentTimeMillis();
      while (true) {
        if ((System.currentTimeMillis() - startTime) < 60 * 1000) {
          if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
            break;
          }
        }
        Thread.sleep(1000);
      }

      //make sure that block cache invalidation commands have been sent
      Thread.sleep(5000);

      for (int i = 0; i < NUM_FILES; i++) {
        String fileName = "/files-run1/file" + i;
        RemoteIterator<LocatedFileStatus> status  = dfs.listLocatedStatus(new Path(fileName));
        assert status.hasNext(); // there has to be a block
        while (status.hasNext()) {
          LocatedFileStatus locFileStatus = status.next();
          assert locFileStatus.getBlockLocations()[0].getLength() == 2 * buffer.length;
        }

        int read = readData(cluster, fileName);
        assert read == 2*buffer.length;

        //check versions
      }
      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }

  //make sure that all the blocks in the cloud have only one version
  void checkBlockVersions(CloudPersistenceProvider cloud, Configuration conf, int count) throws IOException {
    if (defaultCloudProvider == CloudProvider.AWS) {
      CloudPersistenceProviderS3Impl s3 = (CloudPersistenceProviderS3Impl) cloud;
      Map<BlockIDAndGSTuple, CloudBlock> cblks = s3.getAll(CloudHelper.ROOT_PREFIX, CloudHelper.getBucketsFromConf(conf));

      int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
              DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
      for(CloudBlock cb : cblks.values()){
        String objetKey = CloudHelper.getBlockKey(prefixSize, cb.getBlock());
        List<S3VersionSummary> versions =
                s3.listAllVersions(CloudHelper.getBucketsFromConf(conf).get(0), objetKey);
        assertTrue("Expected: "+count+" Got: "+ versions.size(), versions.size() == count);
      }

    } else if (defaultCloudProvider == CloudProvider.AZURE) {
      //We do not use versioning in AZURE
      return;
    } else if(defaultCloudProvider == CloudProvider.GCS) {
      CloudPersistenceProviderGCSImpl gcs = (CloudPersistenceProviderGCSImpl) cloud;
      Map<BlockIDAndGSTuple, CloudBlock> cblks = gcs.getAll(CloudHelper.ROOT_PREFIX,
              CloudHelper.getBucketsFromConf(conf));

      int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
              DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
      for(CloudBlock cb : cblks.values()){
        String objetKey = CloudHelper.getBlockKey(prefixSize, cb.getBlock());
        List<Blob> versions =
                gcs.listAllVersions(CloudHelper.getBucketsFromConf(conf).get(0), objetKey);
        if (versions.size() != count ){
          LOG.info("Blk Key: "+objetKey+" Versions "+Arrays.toString(versions.toArray()));
          fail("Expected: "+count+" Got: "+ versions.size());
        }
      }
    } else {
      fail("Cloud provider not yet supported");
    }
  }

  int countActiveLeases() throws StorageException {
    LeasePathDataAccess lpda = (LeasePathDataAccess) HdfsStorageFactory
            .getDataAccess(LeasePathDataAccess.class);
    return lpda.findAll().size();
  }

  int readData(MiniDFSCluster cluster, String file) throws IOException {
    byte[] somedata = new byte[1024];
    DistributedFileSystem newdfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
    FSDataInputStream in = newdfs.open(new Path(file));
    int totRead = 0;
    int read = 0;
    do {
      read = in.read(somedata);
      if (read != -1) {
        totRead += read;
      }
    } while (read != -1);

    return totRead;
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
