/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.snapshots;

import io.hops.metadata.HdfsStorageFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.HopsFSRestore;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudFSOperations {

  static final Log LOG = LogFactory.getLog(TestCloudFSOperations.class);
  static String testBucketPrefix = "hops-test-TCFSO";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudFSOperations(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void TestCloudFSOperations() throws IOException {
    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.TRACE);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int BLKSIZE = 8 * 1024 * 1024;
    final int FILESIZE = 1 * BLKSIZE;
    final int NUM_DN = 2;
    final String BACKUP1 = "backup1";
    final String BACKUP2 = "backup2";

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    LOG.info("________________ Run 1 ________________");
    MiniDFSCluster cluster = null;
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      HdfsStorageFactory.clearBackupsTestingOnly();
      DistributedFileSystem dfs = cluster.getFileSystem();

      //write a file to the cloud
      String firstLine = "run1-small-files\n";
      for (int i = 0; i < 5; i++) {
        FSDataOutputStream out = dfs.create(new Path("/small-files-run1/file" + i));
        out.writeBytes(firstLine);
        out.close();
      }

      firstLine = "run1-large-files\n";
      byte[] data = new byte[FILESIZE - firstLine.getBytes().length];
      for (int i = 0; i < 5; i++) {
        FSDataOutputStream out = dfs.create(new Path("/large-files-run1/file" + i));
        out.write(firstLine.getBytes());
        out.write(data);
        out.close();
      }

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
        HdfsStorageFactory.backupTestingOnly(BACKUP1);
      }
    }

    LOG.info("________________ Run 2 ________________");
    // restart
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem();

      assertTrue("Expecting 5 in-memory file. Got: " + HopsFilesTestHelper.countInMemoryDBFiles(), HopsFilesTestHelper.countInMemoryDBFiles() == 5);
      assertTrue("Expecting 5 block but found " + cluster.getNamesystem().getTotalBlocks(), cluster.getNamesystem().getTotalBlocks() == 5);

      //overwrite a file to the cloud
      String firstLine = "run2-small-files\n";
      for (int i = 0; i < 5; i++) {
        FSDataOutputStream out = dfs.create(new Path("/small-files-run1/file" + i));
        out.writeBytes(firstLine);
        out.close();
      }

      firstLine = "run2-large-files\n";
      byte[] data = new byte[FILESIZE - firstLine.getBytes().length];
      for (int i = 0; i < 5; i++) {
        FSDataOutputStream out = dfs.create(new Path("/large-files-run1/file" + i));
        out.write(firstLine.getBytes());
        out.write(data);
        out.close();
      }

      for (int i = 0; i < 5; i++) {
        writeFile(dfs, "/large-files-run2/file" + i, FILESIZE);
      }

      assertTrue("Expecting 5 in-memory file. Got: " + HopsFilesTestHelper.countInMemoryDBFiles(), HopsFilesTestHelper.countInMemoryDBFiles() == 5);
      assertTrue("Expecting 10 block but found " + cluster.getNamesystem().getTotalBlocks(),
              cluster.getNamesystem().getTotalBlocks() == 10);

      Thread.sleep(5000);

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
        HdfsStorageFactory.backupTestingOnly(BACKUP2);
      }
    }

    LOG.info("________________ Run 3 ________________");
    // Restore
    HdfsStorageFactory.restoreTestingOnly(BACKUP1);
    NameNode.rollbackDeletedCloudBlocks(conf);

    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem();

      //this is the expected state
      assertTrue("Expecting 5 in-memory file. Got: " + HopsFilesTestHelper.countInMemoryDBFiles(), HopsFilesTestHelper.countInMemoryDBFiles() == 5);
      assertTrue("Expecting 5 block but found " + cluster.getNamesystem().getTotalBlocks(), cluster.getNamesystem().getTotalBlocks() == 5);


      //verify that we are reading the old data
      for (int i = 0; i < 5; i++) {
        FSDataInputStream in = dfs.open(new Path("/small-files-run1/file" + i));
        String line = in.readLine();
        assertTrue(line.startsWith("run1-small-files"));
        in.close();
      }

      for (int i = 0; i < 5; i++) {
        FSDataInputStream in = dfs.open(new Path("/large-files-run1/file" + i));
        String line = in.readLine();
        assertTrue(line.startsWith("run1-large-files"));
        in.close();
      }

      for (int i = 0; i < 5; i++) {
        try {
          verifyFile(dfs, "/large-files-run2/file" + i, FILESIZE);
          fail("This file should not be there");
        } catch (FileNotFoundException e) {
        }
      }

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
