/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSOutputStream;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.namenode.FSDirectory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

@RunWith(Parameterized.class)
public class TestCloudUpdateQuotaForFSync {
  
  private static final int BLOCKSIZE = 128 * 1024;
  private static final short REPLICATION = 4;
  static final long seed = 0L;
  private static final Path dir = new Path("/TestQuotaUpdate");
  private Configuration conf;
  private MiniDFSCluster cluster;
  private FSDirectory fsdir;
  private DistributedFileSystem dfs;
  private int leaseCreationLockRows;
  private Path rootDir = new Path("/");
  
  static String testBucketPrefix = "hops-test-TCUQFFS";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudUpdateQuotaForFSync(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Before
  public void setUp() throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
  
    conf = new Configuration();
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCKSIZE);
    leaseCreationLockRows = conf.getInt(DFSConfigKeys.DFS_LEASE_CREATION_LOCKS_COUNT_KEY,
            DFSConfigKeys.DFS_LEASE_CREATION_LOCKS_COUNT_DEFAULT);
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, 0);
  
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    cluster = new MiniDFSCluster.Builder(conf).numDataNodes(REPLICATION)
        .storageTypes(CloudTestHelper.genStorageTypes(REPLICATION)).format(true).build();
    cluster.waitActive();
    
    fsdir = cluster.getNamesystem().getFSDirectory();
    dfs = cluster.getFileSystem();
  
    dfs.setStoragePolicy(rootDir, HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
  }

  @After
  public void tearDown() throws Exception {
    if (cluster != null) {
      cluster.shutdown();
    }
  }

  /**
   * Test if the quota can be correctly updated when file length is updated
   * through fsync
   */
  @Test
  public void testUpdateQuotaForFSync() throws Exception {
    final Path foo = new Path("/foo");
    final Path bar = new Path(foo, "bar");
    DFSTestUtil.createFile(dfs, bar, BLOCKSIZE, REPLICATION, 0L);
    dfs.setQuota(foo, Long.MAX_VALUE - 1, Long.MAX_VALUE - 1);

    FSDataOutputStream out = dfs.append(bar);
    out.write(new byte[BLOCKSIZE / 4]);
    ((DFSOutputStream) out.getWrappedStream()).hsync(EnumSet.of(HdfsDataOutputStream.SyncFlag.UPDATE_LENGTH));

    //let time to the quota manager to update the quota
    DFSTestUtil.waitForQuotaUpdatesToBeApplied();
    
    //ContentSummary will not be the same as LastUpdatedContentSummary, so
    //skip checking for LastUpdatedContentSummary
    ContentSummary csummary = DFSTestUtil.getContentSummary(dfs, foo, false);
    assertEquals(2, csummary.getFileCount() + csummary.getDirectoryCount());
    
    //For cloud files, we update the last block length in the block reporting
    //(BLOCK_RECEIVED_AND_DELETED), thus creating a race condition on updated
    //length
    // that is the length to be BLOCKSIZE or BLOCKSIZE + BLOCKSIZE/4
    boolean lengthUpdated = false;
    for(long ds : Arrays.asList(BLOCKSIZE, (BLOCKSIZE + BLOCKSIZE/4))){
      lengthUpdated =
          ds == csummary.getSpaceConsumed() && ds == csummary.getTypeConsumed(StorageType.CLOUD);
      if(lengthUpdated)
        break;
    }
    assertTrue(lengthUpdated);
    
    // last update block should account for two complete blocks
    assertEquals(BLOCKSIZE * 2,
        dfs.getLastUpdatedContentSummary(foo).getSpaceConsumed());
    
    out.write(new byte[BLOCKSIZE / 4]);
    out.close();

    //let time to the quota manager to update the quota
    DFSTestUtil.waitForQuotaUpdatesToBeApplied();
    
    checkQuotas(foo, 2, (BLOCKSIZE + BLOCKSIZE / 2));
    checkQuotas(rootDir, 3, (BLOCKSIZE + BLOCKSIZE / 2));
  
    // append another block
    DFSTestUtil.appendFile(dfs, bar, BLOCKSIZE);
    //let time to the quota manager to update the quota
    DFSTestUtil.waitForQuotaUpdatesToBeApplied();
    
    checkQuotas(foo, 2, (BLOCKSIZE * 2 + BLOCKSIZE / 2));
    checkQuotas(rootDir, 3, (BLOCKSIZE * 2 + BLOCKSIZE / 2));
  
  }
  
  private void checkQuotas(Path dir, long ns, long ds)
      throws IOException, InterruptedException {
    ContentSummary csummary = DFSTestUtil.getContentSummary(dfs, dir);
    assertEquals(ns, csummary.getFileCount() + csummary.getDirectoryCount());
    assertEquals(ds, csummary.getSpaceConsumed());
    assertEquals(ds, csummary.getTypeConsumed(StorageType.CLOUD));
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
