/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.DSQuotaExceededException;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.io.IOUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * A class for testing quota-related commands
 */
@RunWith(Parameterized.class)
public class TestCloudSpaceCommands {

  public static final Log LOG = LogFactory.getLog(TestCloudSpaceCommands.class);
  
  static String testBucketPrefix = "hops-test-TCSC";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudSpaceCommands(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /**
   * Test HDFS operations that change disk space consumed by a directory tree.
   * namely create, rename, delete, append, and setReplication.
   * <p/>
   * This is based on testNamespaceCommands() above.
   */
  @Test
  public void testSpaceCommands() throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
  
    final Configuration conf = new HdfsConfiguration();
    // set a smaller block size so that we can test with smaller
    // diskspace quotas
    final int BLOCK_SIZE = 512;
    conf.setInt(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCK_SIZE);
    // Make it relinquish locks. When run serially, the result should
    // be identical.
    conf.setInt(DFSConfigKeys.DFS_CONTENT_SUMMARY_LIMIT_KEY, 2);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_QUOTA_UPDATE_INTERVAL_KEY, 1000);
    //Cloud setup
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, 0);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    final MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).format(true).numDataNodes(5)
            .storageTypes(CloudTestHelper.genStorageTypes(5)).build();
    
    final FileSystem fs = cluster.getFileSystem();
    assertTrue("Not a HDFS: " + fs.getUri(),
        fs instanceof DistributedFileSystem);
    final DistributedFileSystem dfs = (DistributedFileSystem) fs;
  
    // set cloud storage policy on root dir
    dfs.setStoragePolicy(new Path("/"),
        HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
    
    try {
      int fileLen = 1024;
      //Replication for cloud files is always 1
      final short replication = 1;
      int fileSpace = fileLen * replication;
      
      // create directory /nqdir0/qdir1/qdir20/nqdir30
      assertTrue(dfs.mkdirs(new Path("/nqdir0/qdir1/qdir20/nqdir30")));

      // HOP - Wait for asynchronous quota updates to be applied
      // set the quota of /nqdir0/qdir1 to 4 * fileSpace
      final Path quotaDir1 = new Path("/nqdir0/qdir1");
      dfs.setQuota(quotaDir1, HdfsConstants.QUOTA_DONT_SET, 4 * fileSpace);
      ContentSummary c = DFSTestUtil.getContentSummary(dfs,quotaDir1);
      assertEquals(c.getSpaceQuota(), 4 * fileSpace);

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // set the quota of /nqdir0/qdir1/qdir20 to 6 * fileSpace
      final Path quotaDir20 = new Path("/nqdir0/qdir1/qdir20");
      dfs.setQuota(quotaDir20, HdfsConstants.QUOTA_DONT_SET, 6 * fileSpace);
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      c = DFSTestUtil.getContentSummary(dfs,quotaDir20);
      assertEquals(c.getSpaceQuota(), 6 * fileSpace);

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // Create /nqdir0/qdir1/qdir21 and set its space quota to 2 * fileSpace
      final Path quotaDir21 = new Path("/nqdir0/qdir1/qdir21");
      assertTrue(dfs.mkdirs(quotaDir21));
      dfs.setQuota(quotaDir21, HdfsConstants.QUOTA_DONT_SET, 2 * fileSpace);
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceQuota(), 2 * fileSpace);

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // 5: Create directory /nqdir0/qdir1/qdir21/nqdir32
      Path tempPath = new Path(quotaDir21, "nqdir32");
      assertTrue(dfs.mkdirs(tempPath));

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // create a file under nqdir32/fileDir
      DFSTestUtil.createFile(dfs, new Path(tempPath, "fileDir/file1"), fileLen,
          replication, 0);
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceConsumed(), fileSpace);

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      FSDataOutputStream fout =
          dfs.create(new Path(quotaDir21, "nqdir33/file2"), replication);
      boolean hasException = false;
      try {
        // HOP - Write in single blocks and wait to trigger exception
        for (int i = 0; i < 2 * fileLen; i += BLOCK_SIZE) {
          fout.write(new byte[BLOCK_SIZE]);
          // ensure that the first block is written out (see FSOutputSummer#flush)
          fout.flush();
          DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        }
        fout.close();
      } catch (QuotaExceededException e) {
        hasException = true;
        IOUtils.closeStream(fout);
      }
      assertTrue(hasException);

      // delete nqdir33
      assertTrue(dfs.delete(new Path(quotaDir21, "nqdir33"), true));
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceConsumed(), fileSpace);
      assertEquals(c.getSpaceQuota(), 2 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
  
      // Verify space before the move:
      c = DFSTestUtil.getContentSummary(dfs,quotaDir20);
      assertEquals(c.getSpaceConsumed(), 0);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      
      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // Move /nqdir0/qdir1/qdir21/nqdir32 /nqdir0/qdir1/qdir20/nqdir30
      Path dstPath = new Path(quotaDir20, "nqdir30");
      Path srcPath = new Path(quotaDir21, "nqdir32");
      assertTrue(dfs.rename(srcPath, dstPath));
      
      // verify space after the move
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      c = DFSTestUtil.getContentSummary(dfs,quotaDir20);
      assertEquals(c.getSpaceConsumed(), fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      // verify space for its parent
      c = DFSTestUtil.getContentSummary(dfs,quotaDir1);
      assertEquals(c.getSpaceConsumed(), fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      // verify space for source for the move
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceConsumed(), 0);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      final Path file2 = new Path(dstPath, "fileDir/file2");
      int file2Len = 2 * fileLen;
      // create a larger file under /nqdir0/qdir1/qdir20/nqdir30
      DFSTestUtil.createFile(dfs, file2, file2Len, replication, 0);

      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      c = DFSTestUtil.getContentSummary(dfs,quotaDir20);
      assertEquals(c.getSpaceConsumed(), 3 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceConsumed(), 0);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));

      // HOP - Wait for asynchronous quota updates to be applied
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      // Reverse: Move /nqdir0/qdir1/qdir20/nqdir30 to /nqdir0/qdir1/qdir21/
      hasException = false;
      try {
        assertFalse(dfs.rename(dstPath, srcPath));
      } catch (DSQuotaExceededException e) {
        hasException = true;
      }
      assertTrue(hasException);
      // make sure no intermediate directories left by failed rename

      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      assertFalse(dfs.exists(srcPath));
      // directory should exist
      assertTrue(dfs.exists(dstPath));
      // verify space after the failed move
      c = DFSTestUtil.getContentSummary(dfs,quotaDir20);
      assertEquals(c.getSpaceConsumed(), 3 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      c = DFSTestUtil.getContentSummary(dfs,quotaDir21);
      assertEquals(c.getSpaceConsumed(), 0);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));

      // Test Append :
      Thread.sleep(10000);
      // verify space quota
      c = DFSTestUtil.getContentSummary(dfs,quotaDir1);
      assertEquals(c.getSpaceQuota(), 4 * fileSpace);
      // verify space before append;
      c = DFSTestUtil.getContentSummary(dfs,dstPath);
      assertEquals(c.getSpaceConsumed(), 3 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      Thread.sleep(10000);
      OutputStream out = dfs.append(file2);
      // appending 1 fileLen should succeed
      out.write(new byte[fileLen]);
      out.close();
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      file2Len += fileLen; // after append

      // verify space after append;
      c = DFSTestUtil.getContentSummary(dfs,dstPath);
      assertEquals(c.getSpaceConsumed(), 4 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      // now increase the quota for quotaDir1
      dfs.setQuota(quotaDir1, HdfsConstants.QUOTA_DONT_SET, 5 * fileSpace);
      // Now, appending more than 1 fileLen should result in an error
      out = dfs.append(file2);
      hasException = false;

      /*
       * Problem with partial append
       * How partial append test works
       * there is room for only two more blocks but the test tries to append three
       * blocks
       * Now the file size is maintained in the inode table. When ever there is 
       * a request for new block (if the quota is not exceeded) then the last 
       * block is committed and the file size is updated in the indoe table
       * 
       * here in the case the case of the third block the last block is committed
       * the size of the file is updated. However when the quota check fails and the
       * entire transaction is rolledback along with the changes of the file size
       * 
       * This file size is updted only if the file is successfully closed or a 
       * new block is added. 
       */
      try {
        // HOP - Write in single blocks and wait to trigger exception
        for (int i = 0; i < 2 ; i ++) {
          out.write(new byte[BLOCK_SIZE]);
          DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        }
        out.close();
        
        out = dfs.append(file2);
        out.write(new byte[BLOCK_SIZE]);
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        out.close();
      } catch (QuotaExceededException e) {
        hasException = true;
        IOUtils.closeStream(out);
      }
      assertTrue(hasException);
      file2Len += fileLen; // after partial append

      // verify space after partial append
      c = DFSTestUtil.getContentSummary(dfs,dstPath);
      assertEquals(c.getSpaceConsumed(), 5 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      
      // Test set replication :
      // Cloud files have replication of 1
      assertEquals(1, replication);
      assertEquals(replication, dfs.getFileStatus(file2).getReplication());
      dfs.setReplication(file2, (short) (replication + 1));
      assertEquals(replication, dfs.getFileStatus(file2).getReplication());
      
      
      // verify that space hasn't changed
      c = DFSTestUtil.getContentSummary(dfs,dstPath);
      assertEquals(c.getSpaceConsumed(), 5 * fileSpace);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      
      // Test HDFS-2053 :
      // Create directory /hdfs-2053
      final Path quotaDir2053 = new Path("/hdfs-2053");
      assertTrue(dfs.mkdirs(quotaDir2053));

      // Create subdirectories /hdfs-2053/{A,B,C}
      final Path quotaDir2053_A = new Path(quotaDir2053, "A");
      assertTrue(dfs.mkdirs(quotaDir2053_A));
      final Path quotaDir2053_B = new Path(quotaDir2053, "B");
      assertTrue(dfs.mkdirs(quotaDir2053_B));
      final Path quotaDir2053_C = new Path(quotaDir2053, "C");
      assertTrue(dfs.mkdirs(quotaDir2053_C));
      // Factors to vary the sizes of test files created in each subdir.
      // The actual factors are not really important but they allow us to create
      // identifiable file sizes per subdir, which helps during debugging.
      int sizeFactorA = 1;
      int sizeFactorB = 2;
      int sizeFactorC = 4;

      // Set space quota for subdirectory C
      dfs.setQuota(quotaDir2053_C, HdfsConstants.QUOTA_DONT_SET,
          (sizeFactorC + 1) * fileSpace);
      c = DFSTestUtil.getContentSummary(dfs,quotaDir2053_C);
      assertEquals(c.getSpaceQuota(), (sizeFactorC + 1) * fileSpace);

      // Create a file under subdirectory A
      DFSTestUtil.createFile(dfs, new Path(quotaDir2053_A, "fileA"),
          sizeFactorA * fileLen, replication, 0);
      c = DFSTestUtil.getContentSummary(dfs,quotaDir2053_A);
      assertEquals(c.getSpaceConsumed(), sizeFactorA * fileSpace);

      // Create a file under subdirectory B
      DFSTestUtil.createFile(dfs, new Path(quotaDir2053_B, "fileB"),
          sizeFactorB * fileLen, replication, 0);
      c = DFSTestUtil.getContentSummary(dfs,quotaDir2053_B);
      assertEquals(c.getSpaceConsumed(), sizeFactorB * fileSpace);
      // Create a file under subdirectory C (which has a space quota)
      DFSTestUtil.createFile(dfs, new Path(quotaDir2053_C, "fileC"),
          sizeFactorC * fileLen, replication, 0);
      c = DFSTestUtil.getContentSummary(dfs,quotaDir2053_C);
      assertEquals(c.getSpaceConsumed(), sizeFactorC * fileSpace);

      // Check space consumed for /hdfs-2053
      c = DFSTestUtil.getContentSummary(dfs,quotaDir2053);
      assertEquals(c.getSpaceConsumed(),
          (sizeFactorA + sizeFactorB + sizeFactorC) * fileSpace);

    } finally {
      cluster.shutdown();
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
