/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

/**
 * A class for testing quota-related commands
 */
@RunWith(Parameterized.class)
public class TestCloudSetQuotaLate {

  public static final Log LOG = LogFactory.getLog(TestCloudSetQuotaLate.class);
  
  static String testBucketPrefix = "hops-test-TCSQL";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudSetQuotaLate(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void testSetQuotaLate() throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
  
    final Configuration conf = new HdfsConfiguration();
    final int BLOCK_SIZE = 512;
    conf.setInt(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCK_SIZE);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_QUOTA_UPDATE_INTERVAL_KEY, 5000);
    //Cloud setup
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, 0);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    final MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(1)
            .storageTypes(CloudTestHelper.genStorageTypes(1)).build();
    try {
      final FileSystem fs = cluster.getFileSystem();
      final DistributedFileSystem dfs = (DistributedFileSystem) fs;
  
      // set cloud storage policy on root dir
      dfs.setStoragePolicy(new Path("/"),
          HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
      
      Path testFolder = new Path("/test");
      dfs.mkdirs(testFolder);

      Path testFile1 = new Path(testFolder, "test1");
      dfs.create(testFile1).close();

      dfs.setQuota(testFolder, 2L, 2 * BLOCK_SIZE);
    } finally {
      cluster.shutdown();
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
