/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.tools.DFSAdmin;
import org.apache.hadoop.hdfs.web.WebHdfsConstants;
import org.apache.hadoop.io.IOUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.server.namenode.cloud.quota.TestCloudQuotaCommands.checkContentSummary;
import static org.apache.hadoop.hdfs.server.namenode.cloud.quota.TestCloudQuotaCommands.runCommand;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * A class for testing quota-related commands
 */
@RunWith(Parameterized.class)
public class TestCloudMultipleFilesSmallerThanOneBlock {

  public static final Log LOG = LogFactory.getLog(TestCloudMultipleFilesSmallerThanOneBlock.class);
  
  static String testBucketPrefix = "hops-test-TCMFSTOB";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudMultipleFilesSmallerThanOneBlock(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /**
   * Like the previous test but create many files. This covers bugs where
   * the quota adjustment is incorrect but it takes many files to accrue
   * a big enough accounting error to violate the quota.
   */
  @Test
  public void testMultipleFilesSmallerThanOneBlock() throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
  
    Configuration conf = new HdfsConfiguration();
    final int BLOCK_SIZE = 6 * 1024;
    conf.setInt(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCK_SIZE);
    // Make it relinquish locks. When run serially, the result should
    // be identical.
    conf.setInt(DFSConfigKeys.DFS_CONTENT_SUMMARY_LIMIT_KEY, 2);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_QUOTA_UPDATE_INTERVAL_KEY, 1000);
    //Cloud setup
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, 0);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(3)
            .storageTypes(CloudTestHelper.genStorageTypes(3)).build();
    cluster.waitActive();
    FileSystem fs = cluster.getFileSystem();
  
    // set cloud storage policy on root dir
    fs.setStoragePolicy(new Path("/"),
        HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
    
    DFSAdmin admin = new DFSAdmin(conf);

    final String nnAddr = conf.get(DFSConfigKeys.DFS_NAMENODE_HTTP_ADDRESS_KEY);
    final String webhdfsuri = WebHdfsConstants.WEBHDFS_SCHEME + "://" + nnAddr;
    System.out.println("webhdfsuri=" + webhdfsuri);
    final FileSystem webhdfs = new Path(webhdfsuri).getFileSystem(conf);
    
    try {
      Path dir = new Path("/test");
      boolean exceededQuota = false;
      ContentSummary c;
      // 1kb file
      // 6kb block
      // 60kb quota
      final short replication = 1;
      final int FILE_SIZE = 1024;
      final int QUOTA_SIZE = 10 * (int) fs.getDefaultBlockSize(dir);
      assertEquals(6 * 1024, fs.getDefaultBlockSize(dir));
      assertEquals(60 * 1024, QUOTA_SIZE);
  
      assertEquals("Cloud files have a fixed replication factor of 1 ",
          1, replication);
      
      // Create the dir and set the quota. We need to enable the quota before
      // writing the files as setting the quota afterwards will over-write
      // the cached disk space used for quota verification with the actual
      // amount used as calculated by INode#spaceConsumedInTree.
      assertTrue(fs.mkdirs(dir));
      runCommand(admin, false, "-setSpaceQuota", Integer.toString(QUOTA_SIZE),
          dir.toString());

      // We can create at most 55 files because block allocation is
      // conservative and initially assumes a full block is used, so we
      // need to leave at least replication * BLOCK_SIZE free space when
      // allocating the last block
      for (int i = 0; i < 55; i++) {
        Path file = new Path("/test/test" + i);
        DFSTestUtil.createFile(fs, file, FILE_SIZE, (short) 3, 1L);
        DFSTestUtil.waitReplication(fs, file, (short) 3);
        // HOP - Wait for asynchronous quota updates to be applied
        Thread.sleep(1000);
        //Cloud replication should always be 1
        assertEquals(replication, fs.getFileStatus(file).getReplication());
      }

      // Should account for all 55 files (almost QUOTA_SIZE)
      c = fs.getContentSummary(dir);
      checkContentSummary(c, webhdfs.getContentSummary(dir));
      assertEquals("Invalid space consumed", 55 * FILE_SIZE * replication,
          c.getSpaceConsumed());
      assertEquals("Invalid space consumed",
          QUOTA_SIZE - (55 * FILE_SIZE * replication),
          replication * (fs.getDefaultBlockSize(dir) - FILE_SIZE));

      // Now check that trying to create another file violates the quota
      Path file = new Path("/test/test55");
      FSDataOutputStream out = fs.create(file, (short) 3);
      try {
        out.write(new byte[FILE_SIZE]);
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        out.close();
        DFSTestUtil.waitReplication(fs, file, (short) 3);
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      assertTrue("Quota not exceeded", exceededQuota);
    } finally {
      cluster.shutdown();
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
