/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.tools.DFSAdmin;
import org.apache.hadoop.io.IOUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;

import static org.apache.hadoop.hdfs.server.namenode.cloud.quota.TestCloudQuotaCommands.runCommand;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * A class for testing quota-related commands
 */
@RunWith(Parameterized.class)
public class TestCloudDiskspaceForSFWhenMovedToCloud {

  public static final Log LOG = LogFactory.getLog(TestCloudDiskspaceForSFWhenMovedToCloud.class);
  
  static String testBucketPrefix = "hops-test-TCDSFSFWMC";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudDiskspaceForSFWhenMovedToCloud(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void testDiskspaceForSFWhenMovedToCloud1() throws
      Exception{
    testDiskspaceForSmallFilesWhenMovedToCloud((short) 1);
  }
  
  @Test
  public void testDiskspaceForSFWhenMovedToCloud2() throws
      Exception{
    testDiskspaceForSmallFilesWhenMovedToCloud((short) 2);
  }
  
  private void testDiskspaceForSmallFilesWhenMovedToCloud(final short dbReplication) throws
      Exception{
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    
    //Cloud replication factor is always 1
    final short replication = 1;
    
    final Configuration conf = new HdfsConfiguration();
    conf.setInt(DFSConfigKeys.DFS_DB_REPLICATION_FACTOR, dbReplication);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_QUOTA_UPDATE_INTERVAL_KEY, 1000);
    final int MAX_SMALL_FILE_SIZE = 64 * 1024;
    final int BLOCK_SIZE = 2 * MAX_SMALL_FILE_SIZE;
    
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, MAX_SMALL_FILE_SIZE);
    conf.setInt(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCK_SIZE);
  
    //Cloud setup
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    final MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(3)
            .storageTypes(CloudTestHelper.genStorageTypes(3)).build();
    
    final FileSystem fs = cluster.getFileSystem();
    assertTrue("Not a HDFS: " + fs.getUri(),
        fs instanceof DistributedFileSystem);
    final DistributedFileSystem dfs = (DistributedFileSystem) fs;
  
    // set cloud storage policy on root dir
    dfs.setStoragePolicy(new Path("/"),
        HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
    
    DFSAdmin admin = new DFSAdmin(conf);
    
    try{
      final int smallfileLen = 1024;
      final long spaceQuota = BLOCK_SIZE * replication * 2;
      
      //create a test directory
      final Path parent = new Path("/test");
      assertTrue(dfs.mkdirs(parent));
      
      String[] args = new String[]{"-setQuota", "5", parent.toString()};
      runCommand(admin, args, false);
      
      // set diskspace quota to 3 * filesize
      runCommand(admin, false, "-setSpaceQuota", Long.toString(spaceQuota),
          parent.toString());
      
      final Path childDir0 = new Path(parent, "data0");
      dfs.mkdirs(childDir0);
      
      final Path datafile0 = new Path(childDir0, "datafile0");
      
      //create datafile0
      DFSTestUtil.createFile(fs, datafile0, smallfileLen, replication, 0);
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      
      // replication factor for small files will alway be dbReplication
      assertEquals(dbReplication, dfs.getFileStatus(datafile0).getReplication());
      
      // count -q /test
      ContentSummary c = DFSTestUtil.getContentSummary(dfs,parent);
      assertEquals(c.getFileCount() + c.getDirectoryCount(), 3);
      assertEquals(c.getQuota(), 5);
      assertEquals(c.getSpaceConsumed(), smallfileLen * dbReplication);
      assertEquals(c.getSpaceQuota(), spaceQuota);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      assertEquals(0, c.getTypeConsumed(StorageType.CLOUD));
  
      boolean exceededQuota = false;
      // append to the file data less than MAX_SMALL_FILE_SIZE
      OutputStream out = fs.append(datafile0);
      
      try {
        out.write(new byte[smallfileLen]);
        out.close();
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      assertFalse("Quota exceeded", exceededQuota);
      
      // replication factor for small files will alway be dbReplication
      assertEquals(dbReplication, dfs.getFileStatus(datafile0).getReplication());
      
      c = DFSTestUtil.getContentSummary(dfs,parent);
      assertEquals(c.getFileCount() + c.getDirectoryCount(), 3);
      assertEquals(c.getQuota(), 5);
      assertEquals(c.getSpaceConsumed(), 2 * smallfileLen * dbReplication);
      assertEquals(c.getSpaceQuota(), spaceQuota);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      assertEquals(0, c.getTypeConsumed(StorageType.CLOUD));
  
      // append more than the small file max size to move to normal file with
      // blocks
      out = fs.append(datafile0);
      try {
        out.write(new byte[BLOCK_SIZE]);
        out.close();
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      assertFalse("Quota exceeded", exceededQuota);
      
      // replication factor should be equal to cloud replication once
      // moved to block
      assertEquals(replication, dfs.getFileStatus(datafile0).getReplication());
      
      c = DFSTestUtil.getContentSummary(dfs,parent);
      assertEquals(c.getFileCount() + c.getDirectoryCount(), 3);
      assertEquals(c.getQuota(), 5);
      assertEquals(c.getSpaceConsumed(), (2 * smallfileLen  + BLOCK_SIZE)*
          replication);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      
      assertEquals(c.getSpaceQuota(), spaceQuota);
      
      //append to the limit of the quota
      //Cloud files have variable sized blocks, that means the append call
      //will keep the last block (2*smallfilLen) as is and create a new block
      //which would violate the quota when checking for adding block
      final long adjustedSpaceQuota = spaceQuota + 2 * replication * smallfileLen;
      runCommand(admin, false, "-setSpaceQuota", Long.toString(adjustedSpaceQuota),
          parent.toString());
      
      out = fs.append(datafile0);
      try {
        out.write(new byte[BLOCK_SIZE-2*smallfileLen]);
        out.close();
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      assertFalse("Quota exceeded", exceededQuota);
      
      c = DFSTestUtil.getContentSummary(dfs,parent);
      assertEquals(c.getFileCount() + c.getDirectoryCount(), 3);
      assertEquals(c.getQuota(), 5);
      assertEquals(c.getSpaceConsumed(), spaceQuota);
      assertEquals(c.getSpaceQuota(), adjustedSpaceQuota);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      
      //anymore appends should fail
      out = fs.append(datafile0);
      try {
        out.write(new byte[1]);
        out.close();
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      
      assertTrue("Quota exceeded", exceededQuota);
      
      //increase replication, should not fail because of quota since cloud
      // files have fixed replication factor
      try{
        dfs.setReplication(datafile0, (short)(replication + 1));
        exceededQuota = false;
      }catch (QuotaExceededException e){
        exceededQuota = true;
      }
      
      assertFalse("Quota exceeded", exceededQuota);
      
      assertEquals(replication, dfs.getFileStatus(datafile0).getReplication());
      
      c = DFSTestUtil.getContentSummary(dfs,parent);
      assertEquals(c.getFileCount() + c.getDirectoryCount(), 3);
      assertEquals(c.getQuota(), 5);
      assertEquals(c.getSpaceConsumed(), spaceQuota);
      assertEquals(c.getSpaceQuota(), adjustedSpaceQuota);
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      
    }finally {
      cluster.shutdown();
    }
  }
  
  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
