/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.quota;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.QuotaExceededException;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.tools.DFSAdmin;
import org.apache.hadoop.hdfs.web.WebHdfsConstants;
import org.apache.hadoop.io.IOUtils;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.server.namenode.cloud.quota.TestCloudQuotaCommands.checkContentSummary;
import static org.apache.hadoop.hdfs.server.namenode.cloud.quota.TestCloudQuotaCommands.runCommand;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * A class for testing quota-related commands
 */
@RunWith(Parameterized.class)
public class TestCloudBlkAllocAdjustUsage {

  public static final Log LOG = LogFactory.getLog(TestCloudBlkAllocAdjustUsage.class);
  
  static String testBucketPrefix = "hops-test-TCBAAU";
  
  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }
  
  CloudProvider defaultCloudProvider;
  public TestCloudBlkAllocAdjustUsage(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }
  
  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /**
   * Violate a space quota using files of size < 1 block. Test that block
   * allocation conservatively assumes that for quota checking the entire
   * space of the block is used.
   */
  @Test
  public void testBlkAllocAdjustUsage() throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
  
    Configuration conf = new HdfsConfiguration();
    final int BLOCK_SIZE = 6 * 1024;
    conf.setInt(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLOCK_SIZE);
    //Cloud setup
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.setInt(DFSConfigKeys.DFS_DB_FILE_MAX_SIZE_KEY, 0);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(3)
            .storageTypes(CloudTestHelper.genStorageTypes(3)).build();
    cluster.waitActive();
    FileSystem fs = cluster.getFileSystem();
    
    // set cloud storage policy on root dir
    fs.setStoragePolicy(new Path("/"),
        HdfsConstants.CLOUD_STORAGE_POLICY_NAME);
    
    DFSAdmin admin = new DFSAdmin(conf);

    final String nnAddr = conf.get(DFSConfigKeys.DFS_NAMENODE_HTTP_ADDRESS_KEY);
    final String webhdfsuri = WebHdfsConstants.WEBHDFS_SCHEME + "://" + nnAddr;
    System.out.println("webhdfsuri=" + webhdfsuri);
    final FileSystem webhdfs = new Path(webhdfsuri).getFileSystem(conf);

    try {
      Path dir = new Path(
          "/folder1/folder2/folder3/folder4/folder5/folder6/folder7/folder8/folder9/folder10");
      Path file1 = new Path(dir, "test1");
      Path file2 = new Path(dir, "test2");
      boolean exceededQuota = false;
      // Cloud files have replication factor of 1
      final short replication = 1;
      final int QUOTA_SIZE = replication * BLOCK_SIZE; // total space usage including
      // repl.
      final int FILE_SIZE = BLOCK_SIZE / 2;
      ContentSummary c;
      
      // Create the directory and set the quota
      assertTrue(fs.mkdirs(dir));
      runCommand(admin, false, "-setSpaceQuota", Integer.toString(QUOTA_SIZE),
          dir.toString());
      runCommand(admin, false, "-setSpaceQuota", Integer.toString(QUOTA_SIZE),
          "/folder1/folder2");
      runCommand(admin, false, "-setSpaceQuota", Integer.toString(QUOTA_SIZE),
          "/folder1/folder2/folder3/folder4/folder5");
      runCommand(admin, false, "-setSpaceQuota", Integer.toString(QUOTA_SIZE),
          "/folder1/folder2/folder3/folder4/folder5/folder6/folder7/folder8");

      // Creating a file should use half the quota
      DFSTestUtil.createFile(fs, file1, FILE_SIZE, (short) 3, 1L);
      DFSTestUtil.waitReplication(fs, file1, (short) 3);
      DFSTestUtil.waitForQuotaUpdatesToBeApplied();
      
      // cloud files replication is always 1
      assertEquals(1, replication);
      assertEquals(replication,fs.getFileStatus(file1).getReplication());
      
      c = fs.getContentSummary(dir);
      checkContentSummary(c, webhdfs.getContentSummary(dir));
      assertEquals("Quota is half consumed", QUOTA_SIZE / 2,
          c.getSpaceConsumed());
      assertEquals(c.getSpaceConsumed(), c.getTypeConsumed(StorageType.CLOUD));
      assertEquals(0, c.getTypeConsumed(StorageType.DB));
      assertEquals(0, c.getTypeConsumed(StorageType.DISK));
      
      // We can not create the 2nd file because even though the total spaced
      // used by two files (2 * 3 * 512/2) would fit within the quota (3 * 512)
      // when a block for a file is created the space used is adjusted
      // conservatively (3 * block size, ie assumes a full block is written)
      // which will violate the quota (3 * block size) since we've already 
      // used half the quota for the first file.
      FSDataOutputStream out = fs.create(file2, (short) 3);
      try {
        out.write(new byte[FILE_SIZE]);
        DFSTestUtil.waitForQuotaUpdatesToBeApplied();
        out.close();
      } catch (QuotaExceededException e) {
        exceededQuota = true;
        IOUtils.closeStream(out);
      }
      assertTrue("Quota not exceeded", exceededQuota);
    } finally {
      cluster.shutdown();
    }
  }


  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
