/*
 * Copyright (C) 2022 Hopsworks AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.operations;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudS3Encryption;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.PartRef;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.UploadID;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;

import static org.junit.Assert.fail;


@RunWith(Parameterized.class)
public class TestCloudAllOperations {
  static final Log LOG = LogFactory.getLog(TestCloudAllOperations.class);

  static String testBucketPrefix = "hops-test-TS3E";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudAllOperations(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Test
  public void TestAllOps() throws IOException {

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      Configuration conf = new HdfsConfiguration();

      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      //---------------- S3 Config ----------------
      conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_ENABLE_KEY, false);
      conf.set(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_TYPE_KEY,
        CloudS3Encryption.SSE_KMS.toString());
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_BUCKET_KEY_ENABLE_KEY, true);

      //---------------- AZURE Config ----------------
      conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);

      //---------------- GCS Config ----------------
      conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);

      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      String objKey = CloudHelper.getBlockKey(1000, new Block(1, 1, 1, bucket));
      String newKey = CloudHelper.getBlockKey(1000, new Block(2, 2, 2, bucket));

      File objFile = new File("/tmp/" + UUID.randomUUID());
      FileOutputStream fw = new FileOutputStream(objFile);
      fw.write(new byte[1024 *   10]);
      fw.close();
      File downFile = new File("/tmp/testfile");

      HashMap<String, String> metadata = new HashMap<>();
      metadata.put("GEN_STAMP", Long.toString(0));
      metadata.put("OBJECT_SIZE", Long.toString(0));

      metadata = null;

      List<String> buckets = new ArrayList();
      buckets.add(bucket);

      CloudPersistenceProvider cloud =
        (CloudPersistenceProvider) CloudPersistenceProviderFactory.getCloudClient(conf);

      try {
        cloud.checkAllBuckets(buckets);
        LOG.info("PASS: Check buckets ");
      } catch (Exception e) {
        fail("FAIL: Check buckets " + e);
      }

      try {
        cloud.format(buckets);
        LOG.info("PASS: format ");
      } catch (Exception e) {
        fail("FAIL: format. " + e);
      }

      boolean exists = false;
      //check if obj exists
      try {
        exists = cloud.objectExists(bucket, objKey);
        LOG.info("PASS: Check obj exists ");
      } catch (Exception e) {
        fail("FAIL: Check obj exists " + e);
      }

      if (!exists) {
        try {
          cloud.uploadObject(bucket, objKey, objFile, metadata);
          LOG.info("PASS:  upload objcet ");
        } catch (Exception e) {
          fail("FAIL:  upload objcet " + e);
        }
      }

      //overwrite
      try {
        cloud.uploadObject(bucket, objKey, objFile, metadata);
        LOG.info("PASS:  overwrite obj ");
      } catch (Exception e) {
        fail("FAIL:  overwrite obj " + e);
      }

      //read
      try {
        cloud.downloadObject(bucket, objKey, downFile);
        LOG.info("PASS:  read obj ");
      } catch (Exception e) {
        fail("FAIL:  read obj " + e);
      }

      try {
        cloud.getAll("/", buckets);
        LOG.info("PASS:  listing the bucket objects ");
      } catch (Exception e) {
        fail("FAIL:  listing the bucket objects " + e);
      }


      try {
        cloud.getUserMetaData(bucket, objKey);
        LOG.info("PASS:  get obj metadata ");
      } catch (Exception e) {
        fail("FAIL:  get obj  metadata " + e);
      }

      try {
        cloud.getObjectSize(bucket, objKey);
        LOG.info("PASS:  get obj size ");
      } catch (Exception e) {
        fail("FAIL:  get obj  size " + e);
      }


      try {
        cloud.getAllHopsFSDirectories(buckets);
        LOG.info("PASS:  get all dirs ");
      } catch (Exception e) {
        fail("FAIL:  get all dirs " + e);
      }

      //rename
      try {
        cloud.renameObject(bucket, bucket, objKey, newKey);
        cloud.renameObject(bucket, bucket, newKey, objKey);
        LOG.info("PASS:  rename  ");
      } catch (Exception e) {
        if (!(e instanceof UnsupportedOperationException &&
          defaultCloudProvider.name().compareTo(CloudProvider.AZURE.name()) == 0)) {
          fail("FAIL:  rename " + e);
        }
      }

      //copy obj
      try {
        cloud.copyObject(bucket, bucket, objKey, newKey, null);
        LOG.info("PASS:  copy obj  ");
      } catch (Exception e) {
        if ((e instanceof UnsupportedOperationException &&
          defaultCloudProvider.name().compareTo(CloudProvider.AZURE.name()) == 0)) {
          cloud.uploadObject(bucket, newKey, objFile, null); //create a copy for delete operation
        } else {
          fail("FAIL:  copy obj " + e);
        }
      }


      //copy obj
      try {
        cloud.deleteObject(bucket, newKey);
        LOG.info("PASS:  delete  ");
      } catch (Exception e) {
        fail("FAIL:  delete " + e);
      }

      //multipart
      UploadID uid = null;
      try {
        uid = cloud.startMultipartUpload(bucket, newKey, null);
        LOG.info("PASS:  start multipart  ");
      } catch (Exception e) {
        fail("FAIL:  start multipart " + e);
      }

      //upload part
      ArrayList<PartRef> prefs = new ArrayList<PartRef>();
      try {
        prefs.add(cloud.uploadPart(bucket, newKey, uid, 1, objFile, 0, objFile.length()));
        LOG.info("PASS:  upload part   ");
      } catch (Exception e) {
        fail("FAIL:  upload part " + e);
      }

      //finalize part
      try {
        cloud.finalizeMultipartUpload(bucket, newKey, uid, prefs);
        LOG.info("PASS:  finalize part   ");
      } catch (Exception e) {
        fail("FAIL:  finalize part " + e);
      }

      //finalize part
      try {
        cloud.deleteObject(bucket, newKey);
        LOG.info("PASS:  delete   ");
      } catch (Exception e) {
        fail("FAIL:  delete " + e);
      }


      // testing abort
      try {
        uid = cloud.startMultipartUpload(bucket, newKey, null);
        LOG.info("PASS:  start multipart  ");
      } catch (Exception e) {
        fail("FAIL:  start multipart " + e);
      }

      //upload part
      prefs.clear();
      try {
        prefs.add(cloud.uploadPart(bucket, newKey, uid, 1, objFile, 0, objFile.length()));
        LOG.info("PASS:  upload part   ");
      } catch (Exception e) {
        fail("FAIL:  upload part " + e);
      }

      //abort
      try {
        cloud.abortMultipartUpload(bucket, newKey, uid);
        LOG.info("PASS:  abort multipart   ");
      } catch (Exception e) {
        fail("FAIL:  abort multipart " + e);
      }

      if (defaultCloudProvider.name().compareTo(CloudProvider.AZURE.name()) != 0) {
        //list multipart
        try {
          cloud.listMultipartUploads(buckets, "/");
          LOG.info("PASS:  list multipart upload   ");
        } catch (Exception e) {
          fail("FAIL:  list multipart upload " + e);
        }
      }

      //delete block
      try {
        cloud.deleteObject(bucket, objKey);
        LOG.info("PASS:  delete   ");
      } catch (Exception e) {
        fail("FAIL:  delete " + e);
      }

      //restore block
      try {
        if (!cloud.restoreDeletedBlock(bucket, objKey)) {
          LOG.info("FAIL:  restore did not work ");
        } else
          LOG.info("PASS:  restore   ");
      } catch (Exception e) {
        fail("FAIL:  restore " + e);
      }

      //is versionin
      try {
        if (!cloud.isVersioningSupported(bucket)) {
          LOG.info("FAIL:  is version supported is false  ");
        } else
          LOG.info("PASS:  versioning supported   ");
      } catch (Exception e) {
        fail("FAIL:  versioning supported " + e);
      }

      //overwrite to create a new version
      try {
        cloud.uploadObject(bucket, objKey, objFile, null);
        LOG.info("PASS:  upload objcet ");
      } catch (Exception e) {
        fail("FAIL:  upload objcet " + e);
      }

      //delete all version
      try {
        cloud.deleteOldVersions(bucket, objKey);
        LOG.info("PASS:  delete old versions   ");
      } catch (Exception e) {
        fail("FAIL:  delete old version " + e);
      }


      try {
        cloud.deleteAllVersions(bucket, objKey);
        LOG.info("PASS:  delete all versions   ");
      } catch (Exception e) {
        // azure uses soft delete where the blocks are automatically deleted after the
        // retention period for the block expires. You can not delete block that has snapshots
        if (!(e.getMessage().contains("This operation is not permitted because the blob has " +
          "snapshots") && defaultCloudProvider.name().compareTo(CloudProvider.AZURE.name()) == 0)) {
          fail("FAIL:  delete all version " + e);
        }
      }
    } catch (Exception e) {
      fail(e.toString());
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
