/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.multipartupload;

import com.lc.repackaged.com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudReadAndWrite {

  static final Log LOG = LogFactory.getLog(TestCloudReadAndWrite.class);
  static String testBucketPrefix = "hops-test-TCRAW";

  @Before
  public void setup() {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudReadAndWrite(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Test
  public void TestSimpleConcurrentReadAndWrite() throws IOException {
    testConcurrentWrite(true);
  }

  @Test
  public void TestSimpleReadAndWrite() throws IOException {
    testConcurrentWrite(false);
  }

  public void testConcurrentWrite(boolean multipart) throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 32 * 1024 * 1024;
      final int FILESIZE = 2 * BLKSIZE;

      final String FILE_NAME1 = "/dir/TEST-FLIE1";
      final int NUM_DN = 3;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD, multipart);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();
      CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int numFiles = 1;
      int fileSize = BLKSIZE - (1024 * 1024);
      FSDataOutputStream out[] = new FSDataOutputStream[numFiles];
      byte[] data = new byte[fileSize];
      for (int i = 0; i < numFiles; i++) {
        out[i] = dfs.create(new Path("/dir/file" + i), (short) 1);
        out[i].write(data);
      }

      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        if (multipart) {
          assert cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                  CloudHelper.ROOT_PREFIX).size() == numFiles;
        } else {
          assert cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                  CloudHelper.ROOT_PREFIX).size() == 0;
        }
      }

      for (int i = 0; i < numFiles; i++) {
        out[i].close();
      }

      Thread.sleep(5000);
      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        int count =
                cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        assertTrue("Expecting 0 multipler uploads: Got: " + count, count == 0);
      }

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
