/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.multipartupload;

import com.lc.repackaged.com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudMultipartUploadAndSync {

  static final Log LOG = LogFactory.getLog(TestCloudMultipartUploadAndSync.class);
  static String testBucketPrefix = "hops-test-TCMUAS";

  @Before
  public void setup() {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudMultipartUploadAndSync(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Test
  public void TestCloudMultipartUploadAndSync() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 32 * 1024 * 1024;
      final int NUM_DN = 3;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      int cloudMultipartSize = 5 * 1024 * 1024;
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, cloudMultipartSize);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, cloudMultipartSize);

      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      DistributedFileSystem dfs = cluster.getFileSystem();
      dfs.mkdirs(new Path("/dir"));

      int dataWritten = 0;
      byte buffer[] = new byte[cloudMultipartSize + 64 * 1024];
      FSDataOutputStream out = dfs.create(new Path("/dir/file"));


      // write some data and make sure that multipart upload is being used for the current
      // block
      out.write(buffer);
      dataWritten += buffer.length;
      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        Thread.sleep(10000);
        int uploads = cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        assertTrue("Expected: 1. Got " + uploads,
                uploads == 1);
      }

      // call hsync
      // multipart upload will be cancelled and the current block
      // will be uploaded to the cloud
      // first block ~ 5MB
      out.hsync();
      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        int uploads = cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        assertTrue("Expected: 0. Got " + uploads, uploads == 0);
      }
      long read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + dataWritten, read == dataWritten);


      //write some more data
      //second block.
      out.write(buffer);  // 5MB
      dataWritten += buffer.length;
      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        int uploads = cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        assertTrue("Expected: 0. Got " + uploads, uploads == 0);
      }
      read = readData(cluster, "/dir/file");
      assert read <= dataWritten;


      //write enough data to complete the current block and allocate a new block
      byte remaining[] = new byte[BLKSIZE - (buffer.length) + 64 * 1024];
      out.write(remaining);
      dataWritten += remaining.length;
      read = readData(cluster, "/dir/file");
      assert read <= dataWritten;

      //write some more data and make sure that multipart upload is being used
      //for the new block
      out.write(buffer);
      dataWritten += buffer.length;
      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        Thread.sleep(10000);
        int uploads = cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX ).size();
        assertTrue("Expected: 1. Got " + uploads, uploads == 1);
      }
      read = readData(cluster, "/dir/file");
      assert read <= dataWritten;


      //write some more data an call hsync
      out.write(buffer);
      dataWritten += buffer.length;
      out.hsync();
      Thread.sleep(5000);
      read = readData(cluster, "/dir/file");
      assertTrue("Got: " + read + " Expected: " + dataWritten, read == dataWritten);

      if (defaultCloudProvider != CloudProvider.AZURE && defaultCloudProvider != CloudProvider.GCS) {
        int uploads = cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        assertTrue("Expected: 0. Got " + uploads, uploads == 0);
      }
      out.close();

      CloudTestHelper.matchMetadata(conf);
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  int readData(MiniDFSCluster cluster, String file) throws IOException {
    byte[] somedata = new byte[1024];
    DistributedFileSystem newdfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
    FSDataInputStream in = newdfs.open(new Path(file));
    int totRead = 0;
    int read = 0;
    do {
      read = in.read(somedata);
      if (read != -1) {
        totRead += read;
      }
    } while (read != -1);

    return totRead;
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
