/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.mover;

import io.hops.exception.StorageException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.TablesDef;
import io.hops.metadata.hdfs.dal.SQLDataAccess;
import io.hops.metadata.hdfs.dal.SQLResultSetHandler;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.LogVerificationAppender;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.balancer.ExitStatus;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.mover.Mover;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.util.ToolRunner;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

@RunWith(Parameterized.class)
public class TestCloudupdateTS {
  static final Log LOG = LogFactory.getLog(TestCloudupdateTS.class);

  static String testBucketPrefix = "hops-test-TCUTS";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudupdateTS(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void testupdateTS() throws Exception {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    int datanodes = 2;
    int filePerDir = 5;
    int retriesForMover = 1;
    final int FILE_SIZE = 1 * 1024 * 1024;
    final int BLKSIZE = 1 * 1024 * 1024;

    final Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.DFS_DN_CLOUD_BYPASS_CACHE_KEY, false);
    conf.setInt(DFSConfigKeys.DFS_DATANODE_BALANCE_MAX_NUM_CONCURRENT_MOVES_KEY, 20);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 98);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    final LogVerificationAppender appender = new LogVerificationAppender();
    Logger.getRootLogger().addAppender(appender);

    final MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf)
            .numDataNodes(datanodes)
            .storageTypes( new StorageType[]{StorageType.DISK, StorageType.CLOUD})
            .build();
    try {

      cluster.waitActive();

      // Cloud enabled setup. Safe mode will be skipped.
      assertEquals(1, appender.countLinesWithMessage("HopsFS skipping safe mode"));

      final DistributedFileSystem dfs = cluster.getFileSystem();
      Path dir1 = new Path("/dir1");
      dfs.mkdirs(dir1);
      dfs.setStoragePolicy(dir1, "HOT");

      List<Path> files = new ArrayList<Path>();
      for (int i = 0; i < filePerDir; i++) {
        writeFile(dfs,  (new Path(dir1, "file"+i)).toString(), FILE_SIZE, (short)2);
      }

      dfs.setStoragePolicy(dir1, "CLOUD");

      //update time stamps
      //restart and move blocks to cloud
      long ts = System.currentTimeMillis();
      updateTimeStamps(ts);
      int count = countChangedRows(ts);
      assertEquals(count, filePerDir*FILE_SIZE/BLKSIZE );


      cluster.restartNameNode();
      cluster.restartDataNodes();
      waitDNCount(cluster, 2);

      // make sure that nn goes through safemode
      callMover(conf, retriesForMover);

      for (int i = 0; i < filePerDir; i++) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks((new Path(dir1, "file"+i)).toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          assertTrue(StorageType.CLOUD == storageType);
        }
      }

      ProvidedBlocksChecker pbc =
              cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();
      long brCount = pbc.getProvidedBlockReportsCount();
      ProvidedBlocksChecker.scheduleBlockReportNow();
      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, brCount + 1);

      CloudTestHelper.matchMetadata(conf);
    } finally {
      cluster.shutdown();
    }
  }

  public void callMover(Configuration conf, int retriesForMover) throws Exception {
    int rc = 0;
    String args[] = new String[]{"-p", "/"};
    for (int i = 0; i < retriesForMover; i++) {
      rc = ToolRunner.run(conf, new Mover.Cli(), args);
      if (rc == ExitStatus.SUCCESS.getExitCode()) {
        break;
      } else if (rc == ExitStatus.IN_PROGRESS.getExitCode()) {
        continue;
      } else {
        Assert.assertEquals("Movement to CLOUD should be successfull", 0, rc);
      }
    }
    Assert.assertEquals("Movement to CLOUD should be successfull Code:" + rc,
      ExitStatus.SUCCESS.getExitCode(), rc);
  }


  public int countChangedRows(long ts) throws IOException {
    String query =
            "select count(*) from " + TablesDef.INodeTableDef.TABLE_NAME + ", " +
                    TablesDef.BlockInfoTableDef.TABLE_NAME + " WHERE " +
                    TablesDef.INodeTableDef.TABLE_NAME + "." + TablesDef.INodeTableDef.ID + " = " +
                    TablesDef.BlockInfoTableDef.TABLE_NAME + "." + TablesDef.BlockInfoTableDef.INODE_ID + " AND " +
                    TablesDef.INodeTableDef.TABLE_NAME + "." + TablesDef.INodeTableDef.MODIFICATION_TIME + " = " + ts + " and " +
                    TablesDef.INodeTableDef.TABLE_NAME + "." + TablesDef.INodeTableDef.ACCESS_TIME + " = " + ts + " " + "and " +
                    TablesDef.BlockInfoTableDef.TABLE_NAME + "." + TablesDef.BlockInfoTableDef.TIME_STAMP + " = " + ts + " ";

    LightWeightRequestHandler handler = new LightWeightRequestHandler(HDFSOperationType.TEST) {
      @Override
      public Object performTask() throws IOException {
        SQLDataAccess da = (SQLDataAccess) HdfsStorageFactory
                .getDataAccess(SQLDataAccess.class);
        int ret = (int) da.executeQuery(query, new SQLResultSetHandler<Integer>() {
          @Override
          public Integer handle(ResultSet result) throws SQLException, StorageException {
            if (!result.next()) {
              throw new StorageException("Unable to get count");
            }
            return result.getInt(1);
          }
        });
        return  ret;
      }
    };

    return (int)handler.handle();
  }

  public int updateTimeStamps(long ts) throws IOException {
    LightWeightRequestHandler handler = new LightWeightRequestHandler(HDFSOperationType.TEST) {
      @Override
      public Object performTask() throws IOException {
        SQLDataAccess da = (SQLDataAccess) HdfsStorageFactory
                .getDataAccess(SQLDataAccess.class);
        String updateInodeTSQuery = "UPDATE "+ TablesDef.INodeTableDef.TABLE_NAME+ " SET "+
                TablesDef.INodeTableDef.ACCESS_TIME +" = "+ts+", "+
                TablesDef.INodeTableDef.MODIFICATION_TIME +" = "+ts+" ";
        String updateBlockTSQuery = "UPDATE "+ TablesDef.BlockInfoTableDef.TABLE_NAME+ " SET "+
                TablesDef.BlockInfoTableDef.TIME_STAMP +" = "+ts+" ";
        da.executeUpdate(updateInodeTSQuery);
        da.executeUpdate(updateBlockTSQuery);
        return  0;
      }
    };

    return (int)handler.handle();
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
