/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.mover;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.balancer.ExitStatus;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeDescriptor;
import org.apache.hadoop.hdfs.server.blockmanagement.DatanodeStorageInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.HashBuckets;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.TestBlockReport2;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.mover.Mover;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.BlockReport;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.util.ToolRunner;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudMoverMedBlkRept {
  static final Log LOG = LogFactory.getLog(TestCloudMoverMedBlkRept.class);

  static String testBucketPrefix = "hops-test-TCMMBR";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudMoverMedBlkRept(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void testMoverMedBlkRept() throws Exception {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HashBuckets.class).setLevel(Level.DEBUG);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    int datanodes = 2;
    int treeDepth = 10;
    final int FILE_SIZE = 1 * 1024 * 1024;
    final int BLKSIZE = 1 * 1024 * 1024;
    final Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.DFS_DN_CLOUD_BYPASS_CACHE_KEY, false);
    conf.setInt(DFSConfigKeys.DFS_DATANODE_BALANCE_MAX_NUM_CONCURRENT_MOVES_KEY, 20);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 98);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    // dead datanodes
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 1000);
    conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 2L);
    conf.setInt(DFSConfigKeys.DFS_NAMENODE_REPLICATION_PENDING_TIMEOUT_SEC_KEY, 2);
    conf.setInt(DFSConfigKeys.DFS_CLIENT_SOCKET_TIMEOUT_KEY, 5000);

    final MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf)
            .numDataNodes(datanodes)
            .storageTypes(
                    new StorageType[]{StorageType.DISK, StorageType.CLOUD})
            .build();
    try {
      cluster.waitActive();
      final DistributedFileSystem dfs = cluster.getFileSystem();
      Path dir = new Path("/dir");
      dfs.mkdirs(dir);
      dfs.setStoragePolicy(dir, "HOT");

      List<Path> files = new ArrayList<Path>();
      Path curDir = new Path(dir.toString());
      for (int i = 0; i < treeDepth; i++) {
        curDir = new Path(curDir, "depth" + i);
        files.add(new Path(curDir, "file_at_depth_" + i));
      }

      // write to DISK
      for (Path file : files) {
        writeFile(dfs, file.toString(), FILE_SIZE, (short)2);
      }

      //verify before movement
      for (Path file : files) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks(file.toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          assertTrue(StorageType.DISK == storageType);
        }
      }

      dfs.setStoragePolicy(dir, "CLOUD");

      //stopping all datanodes
      cluster.stopDataNode(1);
      waitDNCount(cluster, 1);

      //restart NN and remaining datanode.
      cluster.restartNameNode();
      cluster.restartDataNode(0);
      waitDNCount(cluster, 1);

      LOG.info("Performing block report in parallel with Mover");
      List<Map<DatanodeStorage, BlockReport>> allBrs = new ArrayList<>();
      BlockManager bm = cluster.getNamesystem(0).getBlockManager();
      DataNode dn = cluster.getDataNodes().get(0);
      Map<DatanodeStorage, BlockReport> dnBr = TestBlockReport2.getDNBR(cluster, dn,
              conf.getInt(DFSConfigKeys.DFS_NUM_BUCKETS_KEY, DFSConfigKeys.DFS_NUM_BUCKETS_DEFAULT));
      allBrs.add(dnBr);

      MoverThread mover = new MoverThread(dfs, conf);
      mover.start();

      Thread.sleep(5000);

      for (Map<DatanodeStorage, BlockReport> dnBR : allBrs) {

        for (Map.Entry<DatanodeStorage, BlockReport> datanodeStorageBlockReportEntry : dnBR.entrySet()) {
          DatanodeDescriptor datanode = cluster.getNamesystem().getBlockManager().getDatanodeManager().getDatanode(dn.getDatanodeId());
          DatanodeStorageInfo storageInfo = datanode.getStorageInfo(datanodeStorageBlockReportEntry.getKey().getStorageID());

          BlockManager.ReportStatistics stats = bm.processReport(storageInfo,
                  datanodeStorageBlockReportEntry.getValue());
        }
      }
      mover.join();

      assertTrue(cluster.getNamesystem().getMissingBlocksCount() == 0);
      assertTrue(cluster.getNamesystem().getCorruptReplicaBlocks() == 0);

      CloudTestHelper.matchMetadata(conf);
    } finally {
      cluster.shutdown();
    }
  }

  class MoverThread extends Thread {
    DistributedFileSystem dfs;
    Configuration conf;

    public MoverThread(DistributedFileSystem dfs, Configuration conf) {
      this.dfs = dfs;
      this.conf = conf;
    }

    @Override
    public void run() {
      try {
        // move to CLOUD
        int tries = 1;
        int rc = 0;
        String args[] = new String[]{"-p", "/"};
        for (int i = 0; i < tries; i++) {
          rc = ToolRunner.run(conf, new Mover.Cli(),
                  args);
          if (rc == ExitStatus.SUCCESS.getExitCode()) {
            break;
          } else if (rc == ExitStatus.IN_PROGRESS.getExitCode()) {
            continue;
          } else {
            Assert.assertEquals("Movement to CLOUD should be successfull", 0, rc);
          }
        }
        Assert.assertEquals("Movement to CLOUD should be successfull Code:" + rc,
                ExitStatus.SUCCESS.getExitCode(), rc);
      } catch (Exception e) {
        fail("Mover failed");
      } finally {
      }
    }

  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
