/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.mover;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.LogVerificationAppender;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.balancer.ExitStatus;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.mover.Mover;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.util.ToolRunner;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

@RunWith(Parameterized.class)
public class TestCloudMoverAndSafeMode {
  static final Log LOG = LogFactory.getLog(TestCloudMoverAndSafeMode.class);

  static String testBucketPrefix = "hops-test-TCMASM";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudMoverAndSafeMode(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /*
  Safe Mode should not be skipped if there are blocks on disk
   */
  @Test
  public void testMoverAndSafeMode() throws Exception {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    int datanodes = 2;
    int filePerDir = 5;
    int retriesForMover = 1;
    final int FILE_SIZE = 1 * 1024 * 1024;
    final int BLKSIZE = 1 * 1024 * 1024;
    final Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.DFS_DN_CLOUD_BYPASS_CACHE_KEY, false);
    conf.setInt(DFSConfigKeys.DFS_DATANODE_BALANCE_MAX_NUM_CONCURRENT_MOVES_KEY, 20);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 98);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    final LogVerificationAppender appender = new LogVerificationAppender();
    Logger.getRootLogger().addAppender(appender);

    final MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf)
            .numDataNodes(datanodes)
            .storageTypes(
                    new StorageType[]{StorageType.DISK, StorageType.CLOUD})
            .build();
    try {

      cluster.waitActive();

      // Cloud enabled setup. Safe mode will be skipped.
      assertEquals(1, appender.countLinesWithMessage("HopsFS skipping safe mode"));

      final DistributedFileSystem dfs = cluster.getFileSystem();
      Path dir1 = new Path("/dir1");
      Path dir2 = new Path("/dir2");
      dfs.mkdirs(dir1);
      dfs.mkdirs(dir2);
      dfs.setStoragePolicy(dir1, "HOT");
      dfs.setStoragePolicy(dir2, "HOT");

      List<Path> files = new ArrayList<Path>();
      for (int i = 0; i < filePerDir; i++) {
        writeFile(dfs,  (new Path(dir1, "file"+i)).toString(), FILE_SIZE, (short)2);
        writeFile(dfs,  (new Path(dir2, "file"+i)).toString(), FILE_SIZE, (short)2);
      }

      dfs.setStoragePolicy(dir1, "CLOUD");

      assertEquals(0, appender.countLinesWithMessage("HopsFS entering safe mode"));

      cluster.restartNameNode();
      cluster.restartDataNodes();
      waitDNCount(cluster, 2);

      // make sure that nn goes through safemode
      callMover(conf, retriesForMover);

      for (int i = 0; i < filePerDir; i++) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks((new Path(dir1, "file"+i)).toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          assertTrue(StorageType.CLOUD == storageType);
        }
      }

      assertEquals(1, appender.countLinesWithMessage("HopsFS entering safe mode"));

      for (int i = 0; i < filePerDir; i++) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks((new Path(dir2, "file"+i)).toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          assertTrue(StorageType.DISK == storageType);
        }
      }

      dfs.setStoragePolicy(dir2, "CLOUD");
      cluster.restartNameNode();
      cluster.restartDataNodes();
      waitDNCount(cluster, 2);

      //  make sure that nn goes through safemode
      assertEquals(2, appender.countLinesWithMessage("HopsFS entering safe mode"));
      callMover(conf, retriesForMover);

      for (int i = 0; i < filePerDir; i++) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks((new Path(dir2, "file"+i)).toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          assertTrue(StorageType.CLOUD == storageType);
        }
      }


      cluster.restartNameNode();
      cluster.restartDataNodes();
      waitDNCount(cluster, 2);

      // make sure that nn DOES NOT go through safemode
      assertEquals(2, appender.countLinesWithMessage("HopsFS entering safe mode"));

      CloudTestHelper.matchMetadata(conf);
    } finally {
      cluster.shutdown();
    }
  }

  public void callMover(Configuration conf, int retriesForMover) throws Exception {
    int rc = 0;
    String args[] = new String[]{"-p", "/"};
    for (int i = 0; i < retriesForMover; i++) {
      rc = ToolRunner.run(conf, new Mover.Cli(), args);
      if (rc == ExitStatus.SUCCESS.getExitCode()) {
        break;
      } else if (rc == ExitStatus.IN_PROGRESS.getExitCode()) {
        continue;
      } else {
        Assert.assertEquals("Movement to CLOUD should be successfull", 0, rc);
      }
    }
    Assert.assertEquals("Movement to CLOUD should be successfull Code:" + rc,
            ExitStatus.SUCCESS.getExitCode(), rc);
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
