/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.mover;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.LastUpdatedContentSummary;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.server.balancer.ExitStatus;
import org.apache.hadoop.hdfs.server.mover.Mover;
import org.apache.hadoop.util.ToolRunner;
import org.junit.Assert;

import java.util.ArrayList;
import java.util.List;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.datanode.TestBlockReport2.sendAndCheckBR;

public class MoverHelper {
  static final Log LOG = LogFactory.getLog(MoverHelper.class);

  public void testScheduleBlockWithinSameNode(String testname, CloudProvider defaultCloudProvider,
                                              String testBucketPrefix,
                                              int datanodes,
                                              short replication) throws Exception {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int FILE_SIZE = 5 * 1024 * 1024;
    final int BLKSIZE = 1 * 1024 * 1024;
    final Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setBoolean(DFSConfigKeys.DFS_DN_CLOUD_BYPASS_CACHE_KEY, true);
    conf.setInt(DFSConfigKeys.DFS_DATANODE_BALANCE_MAX_NUM_CONCURRENT_MOVES_KEY, 20);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    final MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf)
      .numDataNodes(datanodes)
      .storageTypes(
        new StorageType[]{StorageType.DISK, StorageType.CLOUD})
      .build();
    try {
      cluster.waitActive();
      int treeDepth = 5;
      final DistributedFileSystem dfs = cluster.getFileSystem();
      Path dir = new Path("/dir");
      dfs.mkdirs(dir);
      dfs.setStoragePolicy(dir, "HOT");

      List<Path> files = new ArrayList<Path>();
      Path curDir = new Path(dir.toString());
      for (int i = 0; i < treeDepth; i++) {
        curDir = new Path(curDir, "depth" + i);
        files.add(new Path(curDir, "file_at_depth_" + i));
      }

      // write to DISK
      for (Path file : files) {
        writeFile(dfs, file.toString(), FILE_SIZE, replication);
        verifyFile(dfs, file.toString(), FILE_SIZE);
      }

      //verify before movement
      for (Path file : files) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks(file.toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          Assert.assertTrue(StorageType.DISK == storageType);
        }
      }

      //Some buckets are expected to mismatch because the initial block reporting is
      //running in parallel with file writes.
      sendAndCheckBR(0, datanodes, cluster, cluster.getNamesystem().getBlockPoolId(),
        files.size() * conf.getInt(DFSConfigKeys.DFS_REPLICATION_KEY, DFSConfigKeys.DFS_REPLICATION_DEFAULT),
        conf.getInt(DFSConfigKeys.DFS_NUM_BUCKETS_KEY, DFSConfigKeys.DFS_NUM_BUCKETS_DEFAULT));

      // move to CLOUD
      dfs.setStoragePolicy(dir, "CLOUD");
      int tries = 1;
      int rc = 0;
      for(int i = 0; i < tries; i++){
        int index=0;
        String args[] = new String[files.size()+1];
        args[index++] = "-p";
        for(Path path:files){
          args[index++] = path.toString();
        }

        rc = ToolRunner.run(conf, new Mover.Cli(),
          args);
        if(rc == ExitStatus.SUCCESS.getExitCode()){
          break;
        } else if(rc == ExitStatus.IN_PROGRESS.getExitCode()){
          continue;
        } else {
          Assert.assertEquals("Movement to CLOUD should be successfull", 0, rc);
        }
      }
      Assert.assertEquals("Movement to CLOUD should be successfull Code:"+rc,
        ExitStatus.SUCCESS.getExitCode(), rc);

      // Wait till namenode notified
      Thread.sleep(3000);
      //verify
      for (Path file : files) {
        LocatedBlock lb = dfs.getClient().getLocatedBlocks(file.toString(), 0).get(0);
        StorageType[] storageTypes = lb.getStorageTypes();
        for (StorageType storageType : storageTypes) {
          Assert.assertTrue("Storage type for "+file+" has not changed",
            StorageType.CLOUD == storageType);
        }
        verifyFile(dfs, file.toString(), FILE_SIZE);
      }

      CloudTestHelper.matchMetadata(conf);
      sendAndCheckBR(0, datanodes, cluster, cluster.getNamesystem().getBlockPoolId(), 0,
        conf.getInt(DFSConfigKeys.DFS_NUM_BUCKETS_KEY, DFSConfigKeys.DFS_NUM_BUCKETS_DEFAULT));

      checkSpaceConsumedOnRootDir(dfs, FILE_SIZE * files.size());

    } finally {
      cluster.shutdown();
    }
  }

  void checkSpaceConsumedOnRootDir(DistributedFileSystem dfs,
                                           long spaceConsumed) throws Exception{
    Path rootDir = new Path("/");
    LastUpdatedContentSummary summary =
      dfs.getLastUpdatedContentSummary(rootDir);

    Assert.assertEquals(spaceConsumed, summary.getSpaceConsumed());

    ContentSummary cSummary = dfs.getContentSummary(rootDir);

    Assert.assertEquals(spaceConsumed, cSummary.getSpaceConsumed());
    Assert.assertEquals(spaceConsumed, cSummary.getTypeConsumed(StorageType.CLOUD));
    Assert.assertEquals(0, cSummary.getTypeConsumed(StorageType.DISK));
  }
}
