/*
 * Copyright (C) 2021 LogicalClocks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.incrementalBR;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataNodeTestUtils;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNodeRpcServer;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.ReceivedDeletedBlockInfo;
import org.apache.hadoop.hdfs.server.protocol.StorageReceivedDeletedBlocks;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;

@RunWith(Parameterized.class)
public class TestCloudIncrementalBlockReport {

  static final Log LOG = LogFactory.getLog(TestCloudIncrementalBlockReport.class);
  static String testBucketPrefix = "hops-test-TCIBR";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudIncrementalBlockReport(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Test(timeout = 60000)
  public void TestIBR() throws IOException {
    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(TestCloudIncrementalBlockReport.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.ERROR);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 32 * 1024 * 1024;
      final int NUM_DN = 1;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      CloudPersistenceProvider cloud = null;
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      final FSNamesystem fsNamesystem = cluster.getNameNode().getNamesystem();
      final FSNamesystem fsNamesystemSpy = Mockito.spy(fsNamesystem);
      NameNodeRpcServer rpcServer = (NameNodeRpcServer) cluster.getNameNode().getRpcServer();
      rpcServer.setFSNamesystem(fsNamesystemSpy);

      List<Long> incorrectBlock = new ArrayList<>();
      List<Long> correctBlock = new ArrayList<>();
      correctBlock.add(1L);
      correctBlock.add(2L);
      incorrectBlock.add(3L);
      incorrectBlock.add(4L);

      ReceivedDeletedBlockInfo cBlockInfo1 =
              new ReceivedDeletedBlockInfo(new Block(correctBlock.get(0), 1000, 1, bucket),
                      ReceivedDeletedBlockInfo.BlockStatus.RECEIVED_BLOCK, null);
      ReceivedDeletedBlockInfo cBlockInfo2 =
              new ReceivedDeletedBlockInfo(new Block(correctBlock.get(1), 1000, 1, bucket),
                      ReceivedDeletedBlockInfo.BlockStatus.RECEIVED_BLOCK, null);

      // this will cause Unsupported operation Exception.
      ReceivedDeletedBlockInfo icBlockInfo1 =
              new ReceivedDeletedBlockInfo(new Block(incorrectBlock.get(0), 1000, 1, bucket),
                      ReceivedDeletedBlockInfo.BlockStatus.DELETED_BLOCK, null);

      ReceivedDeletedBlockInfo icBlockInfo2 =
              new ReceivedDeletedBlockInfo(new Block(incorrectBlock.get(1), 1000, 1, bucket),
                      ReceivedDeletedBlockInfo.BlockStatus.DELETED_BLOCK, null);

      boolean first[] = new boolean[1];
      first[0] = true;
      boolean fail[] = new boolean[1];
      fail[0] = false;
      Answer delayer = new Answer() {
        @Override
        public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
          DatanodeRegistration nodeReg = (DatanodeRegistration) invocationOnMock.getArguments()[0];
          StorageReceivedDeletedBlocks r = (StorageReceivedDeletedBlocks) invocationOnMock.getArguments()[1];
          LOG.info("IBR size "+r.getBlocks().length);

          if (first[0]){
            first[0] = false;
          } else {
            for(ReceivedDeletedBlockInfo rdbi: r.getBlocks()){
              if(!incorrectBlock.contains(rdbi.getBlock().getBlockId())){
                LOG.info("Unexpected block");
                if(!fail[0]){
                  fail[0] = true;
                }
              }
            }
          }
          return invocationOnMock.callRealMethod();
        }
      };

      Mockito.doAnswer(delayer).when(fsNamesystemSpy).processIncrementalBlockReport(any(), any());


      // the first IBR will contain both correct and incorrect blocks.
      // incorrect blocks will throw unsupported operation exception. the datanode will retry only
      // the failed blocks


      DataNode dn = cluster.getDataNodes().get(0);
      String storageID = dn.getFSDataset().getVolumes().get(0).getStorageID();
      DataNodeTestUtils.injectIBR(dn, icBlockInfo1, storageID, false);
      DataNodeTestUtils.injectIBR(dn, icBlockInfo2, storageID, false);
      DataNodeTestUtils.injectIBR(dn, cBlockInfo1, storageID, false);
      DataNodeTestUtils.injectIBR(dn, cBlockInfo2, storageID, true);

      Thread.sleep(3000);

      if(fail[0]){
        fail();
      }

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
