/*
 * Copyright (C) 2020 LogicalClocks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.gcs;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.PartRef;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.UploadID;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.*;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutionException;

import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudGCSUserPays {

  static final Log LOG = LogFactory.getLog(TestCloudGCSUserPays.class);
  static String testBucketPrefix = "hops-test-TGCSUP";


  static Collection CloudProviders = Arrays.asList(new Object[][]{
          {CloudProvider.GCS}
  });

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudGCSUserPays(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*10);

  @Rule
  public Timeout timeout = Timeout.seconds(60*60);

  @Ignore
  @Test
  public void TestAllOperations() throws IOException, InterruptedException, ExecutionException {

    try {

      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
      conf.setBoolean(DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_KEY, true);
      conf.set(DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_PROJECT_KEY, "hopssalman");
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      CloudPersistenceProviderGCSImpl cloud = getConnector(conf);
      List<String> buckets = CloudHelper.getBucketsFromConf(conf);

      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix, conf);
      String bucket = buckets.get(0);
      String key1 = CloudHelper.getBlockKey(1000, new Block(1,1,1, bucket));
      String key2 = CloudHelper.getBlockKey(1000, new Block(2,2,2, bucket));

      File file1 = new File("/tmp/" + UUID.randomUUID());
      File file2 = new File("/tmp/" + UUID.randomUUID());
      FileOutputStream fw = new FileOutputStream(file1);
      fw.write(new byte[1024 * 1024 * 10]);
      fw.close();

      Map<String, String> metadata = new HashMap<>();

      // testing all functions in CloudPersistenceProviderGCSImpl
      cloud.format(buckets);

      // simple copy
      cloud.uploadObject(bucket, key1, file1, metadata);
      cloud.downloadObject(bucket, key1, file2);
      cloud.deleteObject(bucket, key1);

      // multipart upload
      cloud.startMultipartUpload(bucket, key1, metadata);
      UploadID uploadID = null;
      List<PartRef> partRefs = new ArrayList();
      partRefs.add(cloud.uploadPart(bucket, key1, uploadID, 1, file1, 0,
              1024 * 1024 * 5));
      partRefs.add(cloud.uploadPart(bucket, key1, uploadID, 2, file1, 1024 * 1024 * 5,
              1024 * 1024 * 10));
      cloud.finalizeMultipartUpload(bucket, key1, uploadID, partRefs);
      cloud.downloadObject(bucket, key1, file2);
      assert file2.length() == 10 * 1024 * 1024;
      assert cloud.getObjectSize(bucket, key1) == 10 * 1024 * 1024;

      // listings
      cloud.getAllHopsFSDirectories(buckets);
      cloud.listAllVersions(bucket, key1);
      cloud.listMultipartUploads(buckets, key1);
      cloud.getAll(CloudHelper.ROOT_PREFIX, buckets);

      // other
      cloud.checkAllBuckets(buckets);
      cloud.copyObject(bucket, bucket, key1, key2, metadata);
      cloud.deleteObject(bucket, key2);
      cloud.renameObject(bucket, bucket, key1, key2);
      assert cloud.objectExists(bucket, key1) == false;
      assert cloud.objectExists(bucket, key2) == true;
      assert cloud.isVersioningSupported(bucket) == true;


    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
    }
  }

  private CloudPersistenceProviderGCSImpl getConnector(Configuration conf) throws IOException {
    CloudPersistenceProviderGCSImpl cloud = (CloudPersistenceProviderGCSImpl) CloudPersistenceProviderFactory.getCloudClient(conf);
    return cloud;
  }

  //@AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix, CloudProviders);
  }
}
