/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.files;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

@RunWith(Parameterized.class)
public class TestCloudStoragePolicy  {

  static final Log LOG = LogFactory.getLog(TestCloudStoragePolicy.class);

  static String testBucketPrefix = "hops-test-TCSP";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudStoragePolicy(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /*
  If cloud is enabled then the user must not be able to override the storage policy
   */
  @Test
  public void TestCloudStoragePolicyCloudEnabled() throws IOException {
    testStoragePolicy(true);
  }

  @Test
  public void TestCloudStoragePolicyCloudDisabled() throws IOException {
    testStoragePolicy(false);
  }

  public void testStoragePolicy(boolean cloudEnabled) throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      final int NUM_DN = 5;
      final int BLKSIZE = 128 * 1024;
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, cloudEnabled);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      conf.setBoolean(DFSConfigKeys.DFS_DISABLE_NON_CLOUD_STORAE_POLICIES, true);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();
      Exception exception = null;
      try {
        dfs.setStoragePolicy(new Path("/"), "HOT");
      } catch (IOException e) {
        // exception was expected
        exception = e;
      }

      if(cloudEnabled) {
        assert exception != null;
      } else {
        assert exception == null;
      }

      exception = null;
      try {
        dfs.setStoragePolicy(new Path("/"), "CLOUD");
      } catch (IOException e) {
        // exception was expected
        exception = e;
      }

      if(cloudEnabled) {
        assert exception == null;
      } else {
        assert exception != null;
      }


    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
