/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.files;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.*;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockStoragePolicy;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.UUID;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudFormatNonEmptyBucket {

  static final Log LOG = LogFactory.getLog(TestCloudFormatNonEmptyBucket.class);

  static String testBucketPrefix = "hops-test-TCFNEB";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudFormatNonEmptyBucket(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60 * 15);

  @Rule
  public Timeout timeout = Timeout.seconds(60 * 15);

  @Test
  public void TestCloudFormatNonEmptyBucket() throws IOException {

    MiniDFSCluster cluster = null;

    // create the bucket before NN starts and put some HopsFS related data in it.
    // the format operation should fail
    // If CID file exists
    LOG.info("________________ Run 1 ________________");
    try {
      final int NUM_DN = 1;
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      createDummyCIDFile(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      fail("Format should have failed as CID file already exists");
    } catch (Exception e) {
      if (!e.getMessage().contains("Cannot format")) {
        fail(e.getMessage());
      }
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // create the bucket before NN starts and put some HopsFS related data in it.
    // the format operation should fail
    // If blocks exists
    LOG.info("________________ Run 2 ________________");
    try {
      final int NUM_DN = 1;
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      createDummyFSObject(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      fail("Format should have failed as blocks already exists");
    } catch (Exception e) {
      if (!e.getMessage().contains("Cannot format")) {
        fail(e.getMessage());
      }
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // format fails as the bucket does not exist
    LOG.info("________________ Run 3 ________________");
    try {
      final int NUM_DN = 1;
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());

      // this set the bucket name and creats a bucket
      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
      // this deletes the bucket
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      fail("Format should have failed as bucket does not exists");
    } catch (Exception e) {
      if (!e.getMessage().contains("Cannot format")) {
        fail(e.getMessage());
      }
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // format works if the is non hopsfs data
    LOG.info("________________ Run 4 ________________");
    try {
      final int NUM_DN = 1;
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());

      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
      createNonHopsFSObjects(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      // make sure that the non hopsfs data exists
      if (!nonHopsFSObjectsExists(conf)) {
        fail("Non hopsfs data is deleted by format");
      }
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // test force
    LOG.info("________________ Run 5 ________________");
    try {
      final int NUM_DN = 1;
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      // set the configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());

      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
      createDummyCIDFile(conf);
      createNonHopsFSObjects(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN))
              .format(true).startupOption(HdfsServerConstants.StartupOption.FORCE).build();
      cluster.waitActive();

      // make sure that the non hopsfs data exists
      if (!nonHopsFSObjectsExists(conf)) {
        fail("Non hopsfs data is deleted by format");
      }

      CloudTestHelper.matchMetadata(conf);
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }

  public void createDummyCIDFile(Configuration conf) throws IOException {
    String bucket = CloudHelper.getBucketsFromConf(conf).get(0);
    CloudPersistenceProvider cloudConnector =
            CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.setCID(bucket, "test-cid");
    cloudConnector.shutdown();
  }

  public void createDummyFSObject(Configuration conf) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);

    File tmp = new File(System.getProperty("java.io.tmpdir"));
    File file = new File(tmp, UUID.randomUUID().toString());
    FileWriter outblk = new FileWriter(file);
    outblk.write("hello");
    outblk.close();

    String bucket = CloudHelper.getBucketsFromConf(conf).get(0);
    Block blk = new Block(1, 0, 1, bucket);
    String blkKey = CloudHelper.getBlockKey(prefixSize, blk);
    String metaKey = CloudHelper.getMetaFileKey(prefixSize, blk);

    HashMap<String, String> metadata = new HashMap<>();
    CloudPersistenceProvider cloudConnector =
            CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.uploadObject(bucket, blkKey, file, metadata);
    cloudConnector.uploadObject(bucket, metaKey, file, metadata);
    cloudConnector.shutdown();

    file.delete();
  }

  public void createNonHopsFSObjects(Configuration conf) throws IOException {
    File tmp = new File(System.getProperty("java.io.tmpdir"));
    File file = new File(tmp, UUID.randomUUID().toString());
    FileWriter outblk = new FileWriter(file);
    outblk.write("hello");
    outblk.close();

    String bucket = CloudHelper.getBucketsFromConf(conf).get(0);

    HashMap<String, String> metadata = new HashMap<>();
    CloudPersistenceProvider cloudConnector =
            CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.uploadObject(bucket, "someobj", file, metadata);
    cloudConnector.uploadObject(bucket, "/tmp/someobj", file, metadata);
    cloudConnector.uploadObject(bucket, "/backup/someobj", file, metadata);
    cloudConnector.uploadObject(bucket, "/hopsfs/someobj", file, metadata);
    cloudConnector.shutdown();

    file.delete();
  }

  public boolean nonHopsFSObjectsExists(Configuration conf) throws IOException {

    String bucket = CloudHelper.getBucketsFromConf(conf).get(0);

    HashMap<String, String> metadata = new HashMap<>();
    CloudPersistenceProvider cloudConnector =
            CloudPersistenceProviderFactory.getCloudClient(conf);

    boolean exists = true;
    if (!cloudConnector.objectExists(bucket, "someobj") ||
            !cloudConnector.objectExists(bucket, "/tmp/someobj") ||
            !cloudConnector.objectExists(bucket, "/backup/someobj") ||
            !cloudConnector.objectExists(bucket, "/hopsfs/someobj")) {
      exists = false;
    }
    cloudConnector.shutdown();
    return exists;
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
