/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.files;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;

@RunWith(Parameterized.class)
public class TestCloudDisabled  {

  static final Log LOG = LogFactory.getLog(TestCloudDisabled.class);

  static String testBucketPrefix = "hops-test-TCD";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudDisabled(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void TestCloudDisabled() throws Exception {
    MiniDFSCluster cluster = null;
    String connectString = "";
    String awsCredentialProfilesFile = "";
    String awsAccessKeyId = "";
    String awsSecretAccessKey = "";
    String googleApplicaitonCredentials = "";

    try {

      connectString = System.getenv("AZURE_STORAGE_CONNECTION_STRING");
      awsCredentialProfilesFile = System.getenv("AWS_CREDENTIAL_PROFILES_FILE");
      awsAccessKeyId = System.getenv("AWS_ACCESS_KEY_ID");
      awsSecretAccessKey = System.getenv("AWS_SECRET_ACCESS_KEY");
      googleApplicaitonCredentials = System.getenv("GOOGLE_APPLICATION_CREDENTIALS");

      Map<String, String> envs = new HashMap<>();
      if (connectString != null) {
        envs.put("AZURE_STORAGE_CONNECTION_STRING", "");
      }
      if (awsCredentialProfilesFile != null) {
        envs.put("AWS_CREDENTIAL_PROFILES_FILE", "");
      }
      if (awsAccessKeyId != null) {
        envs.put("AWS_ACCESS_KEY_ID", "");
      }
      if (awsSecretAccessKey != null) {
        envs.put("AWS_SECRET_ACCESS_KEY", "");
      }
      if (googleApplicaitonCredentials != null) {
        envs.put("GOOGLE_APPLICATION_CREDENTIALS", "");
      }

      setEnv(envs);


      final int BLKSIZE = 128 * 1024;
      final int FILESIZE = 2 * BLKSIZE;

      final String FILE_NAME1 = "/dir/TEST-FLIE1";
      final int NUM_DN = 3;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, false);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      conf.set(DFSConfigKeys.DFS_CLOUD_AWS_S3_REGION, "some-region");

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));
      writeFile(dfs, FILE_NAME1, FILESIZE);
      verifyFile(dfs, FILE_NAME1, FILESIZE);
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
      //put back the environment variables
      Map<String, String> envs = new HashMap<>();
      if (connectString != null) {
        envs.put("AZURE_STORAGE_CONNECTION_STRING", connectString);
      }
      if (awsCredentialProfilesFile != null) {
        envs.put("AWS_CREDENTIAL_PROFILES_FILE", awsCredentialProfilesFile);
      }

      if (awsAccessKeyId != null) {
        envs.put("AWS_ACCESS_KEY_ID", awsAccessKeyId);
      }

      if (awsSecretAccessKey != null) {
        envs.put("AWS_SECRET_ACCESS_KEY", awsSecretAccessKey);
      }

      if (googleApplicaitonCredentials != null) {
        envs.put("GOOGLE_APPLICATION_CREDENTIALS", googleApplicaitonCredentials);
      }

      setEnv(envs);
    }
  }

  //https://stackoverflow.com/questions/318239/how-do-i-set-environment-variables-from-java
  private void setEnv(Map<String, String> newenv) throws Exception {
    try {
      Class<?> processEnvironmentClass = Class.forName("java.lang.ProcessEnvironment");
      Field theEnvironmentField = processEnvironmentClass.getDeclaredField("theEnvironment");
      theEnvironmentField.setAccessible(true);
      Map<String, String> env = (Map<String, String>) theEnvironmentField.get(null);
      env.putAll(newenv);
      Field theCaseInsensitiveEnvironmentField = processEnvironmentClass.getDeclaredField("theCaseInsensitiveEnvironment");
      theCaseInsensitiveEnvironmentField.setAccessible(true);
      Map<String, String> cienv = (Map<String, String>) theCaseInsensitiveEnvironmentField.get(null);
      cienv.putAll(newenv);
    } catch (NoSuchFieldException e) {
      Class[] classes = Collections.class.getDeclaredClasses();
      Map<String, String> env = System.getenv();
      for (Class cl : classes) {
        if ("java.util.Collections$UnmodifiableMap".equals(cl.getName())) {
          Field field = cl.getDeclaredField("m");
          field.setAccessible(true);
          Object obj = field.get(env);
          Map<String, String> map = (Map<String, String>) obj;
          map.clear();
          map.putAll(newenv);
        }
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
