/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.files;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.*;
import org.apache.hadoop.hdfs.server.blockmanagement.HopsFSRestore;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.*;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudCID {

  static final Log LOG = LogFactory.getLog(TestCloudCID.class);
  static String testBucketPrefix = "hopsfs-testing-TCCID";

  @Before
  public void setup() {
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudCID(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60 * 15);

  @Rule
  public Timeout timeout = Timeout.seconds(60 * 15);

  // test backward compatibility
  @Test
  public void TestCloudCIDBC() throws IOException {
    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.TRACE);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int BLKSIZE = 8 * 1024 * 1024;
    final int FILESIZE = 1 * BLKSIZE;
    final int NUM_DN = 2;
    final String FILE1 = "/dir/file1";

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    LOG.info("________________ Run 1 ________________");
    MiniDFSCluster cluster = null;
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem(0);
      //write a file to the cloud
      writeFile(dfs, FILE1, FILESIZE);
      CloudTestHelper.matchMetadata(conf);
    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // delete the CID file form the bucket
    // the NNs should recreat the CID file
    CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloud.deleteObject(bucket, CloudHelper.CID_FILE);
    cloud.shutdown();

    LOG.info("________________ Run 2 ________________");
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem(0);

      //old file is there
      verifyFile(dfs, FILE1, FILESIZE);
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }


  // test bad CID. cluster should fail to start if CID does not match
  @Test
  public void TestCloudBadCID() throws IOException {
    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.TRACE);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int BLKSIZE = 8 * 1024 * 1024;
    final int FILESIZE = 1 * BLKSIZE;
    final int NUM_DN = 2;
    final String FILE1 = "/dir/file1";

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    LOG.info("________________ Run 1 ________________");
    MiniDFSCluster cluster = null;
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem(0);
      //write a file to the cloud
      writeFile(dfs, FILE1, FILESIZE);
      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    //change CID file
    CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloud.deleteObject(CloudHelper.getBucketsFromConf(conf).get(0), CloudHelper.CID_FILE);
    cloud.setCID(CloudHelper.getBucketsFromConf(conf).get(0), "test-cid");
    cloud.shutdown();

    LOG.info("________________ Run 2 ________________");
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(false).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem(0);

      //old file is there
      verifyFile(dfs, FILE1, FILESIZE);
      fail("Cluster should have failed");
    } catch (Exception e) {
      // cluster failure expected
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }

  // format should fail if bucket is not empty
  @Test
  public void TestCloudCIDFormat() throws IOException {
    Logger.getRootLogger().setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HopsFSRestore.class).setLevel(Level.TRACE);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    final int BLKSIZE = 8 * 1024 * 1024;
    final int FILESIZE = 1 * BLKSIZE;
    final int NUM_DN = 2;
    final String FILE1 = "/dir/file1";

    // set the configuration
    Configuration conf = new HdfsConfiguration();
    conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
    conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE, 5 * 1024 * 1024);
    conf.setLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD, 5 * 1024 * 1024);
    conf.setBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY, true);
    conf.setBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY, true);
    String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

    LOG.info("________________ Run 1 ________________");
    MiniDFSCluster cluster = null;
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .nnTopology(MiniDFSNNTopology.simpleHOPSTopology(2))
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      DistributedFileSystem dfs = cluster.getFileSystem(0);
      //write a file to the cloud
      writeFile(dfs, FILE1, FILESIZE);
      CloudTestHelper.matchMetadata(conf);
    } catch (Exception e) {
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }

    // format should fail
    LOG.info("________________ Run 2 ________________");
    try {
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).wipeDNDirs(true).format(true).build();
      cluster.waitActive();
      fail("Cluster should have failed");
    } catch (Exception e) {
      // cluster failure expected
    } finally {
      if (cluster != null) {
        cluster.shutdown();
        cluster = null;
      }
    }
  }


  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
