/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.failures;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.impl.Log4JLogger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSCluster.DataNodeProperties;
import org.apache.hadoop.hdfs.client.HdfsDataInputStream;
import org.apache.hadoop.hdfs.client.HdfsDataOutputStream;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.BlockReport;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.io.IOUtils;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import static junit.framework.TestCase.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudDNFailures {
  static final Log LOG = AppendTestUtil.LOG;
  static String testBucketPrefix = "hops-test-TCDNFS";

  @Before
  public void setup() {
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.WARN);
  }

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudDNFailures(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  static final String DIR =
          "/" + TestCloudDNFailures.class.getSimpleName() + "/";

  {
    ((Log4JLogger) CloudPersistenceProviderS3Impl.LOG).getLogger().setLevel(Level.ALL);
  }


  /**
   * If the clients and datanodes all die while writing
   *
   * @throws IOException
   */
  @Test
  public void TestCloudDNFailures() throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.WARN);
    MiniDFSCluster cluster = null;
    final int BLKSIZE = 64 * 1024 * 1024;
    final int NUM_DN = 3;

    try {
      // Set Configuration
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, NUM_DN);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 1000 );
      // dead datanodes
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 2L);
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_REPLICATION_PENDING_TIMEOUT_SEC_KEY, 2);
      conf.setInt(DFSConfigKeys.DFS_CLIENT_SOCKET_TIMEOUT_KEY, 5000);

      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      // Start Cluster
      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      cluster.setLeasePeriod(3 * 1000, 10 * 1000);

      // DFS
      DistributedFileSystem dfs = cluster.getFileSystem();
      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      // Save Datanode properties. Need this for DN restart
      Object dnProps[] = cluster.getDataNodeProperties().toArray();

      int numWorker = 3;
      final Future[] futures = new Future[numWorker];
      final SlowWriter[] slowWriters = new SlowWriter[numWorker];
      ExecutorService es = Executors.newFixedThreadPool(numWorker);
      for (int i = 0; i < numWorker; i++) {
        slowWriters[i] = new SlowWriter(dfs, new Path("/dir", "file" + i), 1000L,
                1 * 1024 * 1024);
        slowWriters[i].setMaxDataToWrite(2*BLKSIZE);
        slowWriters[i].setCloseStram(false);
        futures[i] = es.submit(slowWriters[i]);
      }

      Thread.sleep(5000);

      dfs.getClient().getLeaseRenewer().interruptAndJoin();
      dfs.getClient().abort();

      //stopping all datanodes
      for (int i = 0; i < NUM_DN; i++) {
        cluster.stopDataNode(0);
        LOG.info("Stopped a DN at port " + ((DataNodeProperties)dnProps[i]).ipcPort);
      }

      waitDNCount(cluster, 0);

      try{
        for (Future f : futures) {
          f.get();
        }
      } catch (Exception e){
      }

      Thread.sleep(10000);
      ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      long brCount = pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();

      long expected = brCount+1;
      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, expected);
      assertTrue("Exptected "+expected+". Got: " + ret, expected == ret);

      assert  cluster.getNamesystem().getNumDeadDataNodes() == NUM_DN;
      assert cluster.getNamesystem().getMissingBlocksCount() == 0;

      CloudTestHelper.matchMetadata(conf);

      LOG.info("Restarting DN");
      // now restart the datanode
      for(Object dn : dnProps){
        cluster.restartDataNode((DataNodeProperties)dn);
      }
      cluster.waitActive();
      waitDNCount(cluster, NUM_DN);

      cluster.triggerBlockReports();

      Thread.sleep(10000);

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  public static class SlowWriter implements Callable {
    private final Path filepath;
    private HdfsDataOutputStream out;
    private final long sleepms;
    private final int writeSize;
    private volatile boolean running = true;
    private final DistributedFileSystem fs;
    private int fileSize;
    private long maxDataToWrite = Long.MAX_VALUE;
    private boolean closeStram = true;

    public SlowWriter(DistributedFileSystem fs, Path filepath, final long sleepms,
                      final int writeSize) {
      this.filepath = filepath;
      this.sleepms = sleepms;
      this.writeSize = writeSize;
      this.fs = fs;
    }

    @Override
    public Object call() throws Exception {
      running = true;
      slowWrite();
      if(closeStram){
        IOUtils.closeStream(out);
      }
      return null;
    }

    public void setCloseStram(boolean closeStram){
      this.closeStram = closeStram;
    }

    public void setMaxDataToWrite(long maxDataToWrite) {
      this.maxDataToWrite = maxDataToWrite;
    }

    public int getFileSize() {
      return fileSize;
    }

    public Path getFilepath() {
      return filepath;
    }

    void slowWrite() {
      try {
        out = (HdfsDataOutputStream) fs.create(filepath);
        Thread.sleep(sleepms);
        while (running) {
          HopsFilesTestHelper.writeData(out, fileSize, writeSize);
          fileSize += writeSize;
          LOG.info(filepath + " written  " + fileSize);
          if (fileSize >= maxDataToWrite) {
            return;
          }
          Thread.sleep(sleepms);
        }
      } catch (InterruptedException e) {
        LOG.info(filepath + " interrupted:" + e);
      } catch (IOException e) {
        LOG.warn(filepath + " error:" + e);
        throw new RuntimeException(filepath.toString(), e);
      } finally {
      }
    }

    public void verify() throws IOException {
      HopsFilesTestHelper.verifyFile(fs, filepath.toString(), fileSize);
    }

    void stop() {
      running = false;
    }
  }

  public static void waitDNCount(MiniDFSCluster cluster, int count) throws InterruptedException,
    IOException {
    waitDNConnectedTONN(cluster, count);
    waitDNFullyInitialized(cluster, count);
  }

  public static void waitDNFullyInitialized(MiniDFSCluster cluster, int count) throws InterruptedException {
    for (int i = 0; i < 60; i++) {
      ArrayList<DataNode> dns = cluster.getDataNodes();
      boolean pendingNode = false;
      assert dns.size() == count;
      for( DataNode dn : dns){
        if( !dn.isDatanodeFullyStarted()) {
          pendingNode = true;
          LOG.info("Waiting dn is not fully started ");
          Thread.sleep(1000);
        }
      }

      if (!pendingNode) {
        return;
      }
    }
    fail("not all datanodes have fully started yet");
  }

  public static void waitDNConnectedTONN(MiniDFSCluster cluster, int count) throws InterruptedException,
    IOException {
    for (int i = 0; i < 60; i++) {
      if (count == cluster.getNamesystem().getNumLiveDataNodes()) {
        return;
      }
      LOG.info("Waiting for datanode count to reach " + count+ " Current DN Count: "+cluster.getNamesystem().getNumLiveDataNodes());
      Thread.sleep(1000);
    }
    fail("fail to read datanode count. Expecting: " + count + " Got: " + cluster.getNamesystem().getNumLiveDataNodes()
            + " Nodes: "+cluster.getNamesystem().getLiveNodes());
  }

  private void checkDNBR(MiniDFSCluster cluster, DataNode dn) {
    if (!dn.isDatanodeUp())
      fail("Datanode is not running");
    String poolId = cluster.getNamesystem().getBlockPoolId();
    Map<DatanodeStorage, BlockReport> br = dn.getFSDataset().getBlockReports(poolId);
    for (BlockReport reportedBlocks : br.values()) {
      if(reportedBlocks.getNumberOfBlocks() != 0){
        Iterator<Block> itr = reportedBlocks.blockIterable().iterator();
        while(itr.hasNext()){
          LOG.info("Block stored on DN is "+ itr.next());
        }
        fail(dn + " should have no blocks on disk. ");
      }
    }
  }

  static void sleepSeconds(final int waittime) throws InterruptedException {
    LOG.info("Wait " + waittime + " seconds");
    Thread.sleep(waittime * 1000L);
  }


  static class SlowReader implements Callable {
    private final Path filepath;
    private HdfsDataInputStream in;
    private final long sleepms;
    private final int bufferSize;
    private volatile boolean running = true;
    private final DistributedFileSystem fs;
    private final int fileSize;
    private int totRead;

    SlowReader(DistributedFileSystem fs, Path filepath, final long sleepms,
               final int bufferSize, int fileSize) {
      this.filepath = filepath;
      this.sleepms = sleepms;
      this.bufferSize = bufferSize;
      this.fs = fs;
      this.fileSize = fileSize;
    }

    @Override
    public Object call() throws Exception {
      running = true;
      slowRead();
      IOUtils.closeStream(in);
      return null;
    }

    void slowRead() {
      try {
        in = (HdfsDataInputStream) fs.open(filepath);
        Thread.sleep(sleepms);
        for (; running; ) {
          byte[] buffer = new byte[bufferSize];
          int ret = in.read(buffer);
          if (ret > 0) {
            totRead += ret;
            LOG.info(filepath + " read  " + (int) ((double) totRead / (double) fileSize * 100) + "% " +
              "bytes");
            Thread.sleep(sleepms);
          } else {
            assert fileSize == totRead;
            return;
          }
        }
      } catch (InterruptedException e) {
        LOG.info(filepath + " interrupted:" + e);
      } catch (IOException e) {
        throw new RuntimeException(filepath.toString(), e);
      } finally {
      }
    }

    void stop() {
      running = false;
    }

  }

  static void startNewDataNode(boolean enableCloud, int num, MiniDFSCluster cluster,
                                Configuration conf) throws IOException {
    //start new datanodes
    StorageType[][] styps = null;
    if (enableCloud) {
      styps = CloudTestHelper.genStorageTypes(num);
    }
    cluster.startDataNodes(conf, num, styps, true, null, null,
      null, null, null, false, false, false, null);
    cluster.waitActive();
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
