/*
 * Copyright (C) 2024 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.failures;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlock;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.*;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Map;
import java.util.Random;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudClientReportBadBlocks {

  static final Log LOG = LogFactory.getLog(TestCloudClientReportBadBlocks.class);
  static String testBucketPrefix = "hops-test-TCC";
  static String bucket;


  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudClientReportBadBlocks(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60 * 15);

  @Rule
  public Timeout timeout = Timeout.seconds(60 * 15);

  @Before
  public void setup() {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudTestHelper.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProvider.class).setLevel(Level.DEBUG);
  }

  Configuration getConf() {
    Configuration conf = new Configuration();
    return conf;
  }

  /*
  When reading a block if a client encounters CRC errors then it informs the NN.
  The NN marks the block as corrupt.
  I have observed that in the "app" cluster the client
  encountered a CRC error while reading a healthy block, that is the block in the cloud
  was correct. It may have happened due to faulty CRC logic in file-read or due
  to network issue or disk issues.

  When a client reports such a block then the NN will verify the CRC of the block.
  If the block is correct then the NN will issue request to the DN to delete the block
  from the cache. The block is marked corrupt if the block actually has CRC errors
   */
  @Test
  public void TestWrongCorruption1() throws IOException {
    test(true);
  }

  @Test
  public void TestWrongCorruption2() throws IOException {
    test(false);
  }

  public void test(boolean temporaryCorruption) throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);

    MiniDFSCluster cluster = null;
    CloudPersistenceProvider cloudConnector = null;
    try {

      final int BLKSIZE = 2 * 1024 * 1024;
      final int NUM_DN = 3;

      Configuration conf = getConf();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);
      bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      FSDataOutputStream out = dfs.create(new Path("/dir/file"), (short) 1);
      byte[] data = new byte[BLKSIZE];

      int num_blocks = 3;
      for (int i = 0; i < num_blocks; i++) {
        out.write(data);
      }
      out.close();

      LocatedBlocks locations =
              dfs.dfs.getNamenode().getBlockLocations("/dir/file", 0, Long.MAX_VALUE);
      assertEquals(num_blocks, locations.locatedBlockCount());

      FSNamesystem ns = cluster.getNamesystem();

      LocatedBlock[] blocks = new LocatedBlock[0];
      blocks = locations.getLocatedBlocks().toArray(blocks);

      //corrupt blocks
      if (!temporaryCorruption) {
        for (int i = 0; i < blocks.length; i++) {
          corruptTheBlocks(conf, cloudConnector, blocks[i].getBlock());
        }
      }

      ns.reportBadBlocks(blocks);

      Thread.sleep(1000 * 10);

      FSDataInputStream in = dfs.open(new Path("/dir/file"));
      for (int i = 0; i < num_blocks; i++) {
        in.read(data);
      }
      in.close();

      if (temporaryCorruption) {
        CloudTestHelper.matchMetadata(conf);
      } else {
        assert CloudTestHelper.findAllUnderReplicatedBlocks().size() != 0;
      }
    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cloudConnector != null) {
        cloudConnector.shutdown();
      }
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  void corruptTheBlocks(Configuration conf,
                        CloudPersistenceProvider cloudConnector,
                        ExtendedBlock blk) throws IOException {

    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    String blockKey = CloudHelper.getBlockKey(prefixSize, blk.getLocalBlock());

    File tmp = new File(System.getProperty("java.io.tmpdir"));
    File blockFile = new File(tmp, blk.getBlockName());
    cloudConnector.downloadObject(bucket, blockKey, blockFile);
    Map<String, String> md = cloudConnector.getUserMetaData(bucket, blockKey);

    //corrupt the file
    long size = blockFile.length();
    byte rand_data[] = new byte[(int) size];
    Random rand = new Random(System.currentTimeMillis());
    rand.nextBytes(rand_data);
    FileOutputStream fos = new FileOutputStream(blockFile, false);
    fos.write(rand_data);
    fos.close();

    // upload the bad file
    cloudConnector.uploadObject(bucket, blockKey, blockFile, md);

    blockFile.delete();
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
