/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.failures;

import org.apache.commons.logging.Log;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.AppendTestUtil;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSCluster.DataNodeProperties;
import org.apache.hadoop.hdfs.client.HdfsClientConfigKeys;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderAzureImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.protocol.BlockReport;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import static junit.framework.TestCase.assertTrue;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.sleepSeconds;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class CloudFileDNFailureWhileWriting {
  static final Log LOG = AppendTestUtil.LOG;
  static final String DIR =
    "/" + CloudFileDNFailureWhileWriting.class.getSimpleName() + "/";
  @Before
  public void setup() {
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderAzureImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.WARN);
  }

  public void TestFileWriteOnDataNodeFailure(String testname, CloudProvider defaultCloudProvider,
                                             String testBucketPrefix,
                                             final int numThreads) throws Exception {
    final Configuration conf = new HdfsConfiguration();
    boolean enableCloud = true;
    final long BLK_SIZE = 1 * 1024 * 1024;
    final int NUM_DN = 5;

    ExecutorService es = Executors.newFixedThreadPool(numThreads);
    if (enableCloud) {
      CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, NUM_DN);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
    }
    conf.setInt(HdfsClientConfigKeys.BlockWrite.RETRIES_KEY, NUM_DN);
    conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);

    // sometimes the NN constantly return dead datanodes for addBlock Request
    // Increasing the retry count
    conf.getInt(HdfsClientConfigKeys.BlockWrite.RETRIES_KEY, 3 * HdfsClientConfigKeys.BlockWrite.RETRIES_DEFAULT);

    conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
    //if a datanode fails then the unfinished block report entry will linger for some time
    //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
    //able to block report. Reducing the BR Max process time to quickly reclaim
    //unfinished block reports
    conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5*1000);
    conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);

    final MiniDFSCluster.Builder builder = new MiniDFSCluster.Builder(conf).format(true).
            numDataNodes(NUM_DN);
    if (enableCloud) {
      builder.storageTypes(CloudTestHelper.genStorageTypes(NUM_DN));
    }
    final MiniDFSCluster cluster = builder.build();

    try {
      cluster.waitActive();
      final DistributedFileSystem fs = cluster.getFileSystem();
      final Path dir = new Path(DIR);

      int livenodes = cluster.getNamesystem().getNumLiveDataNodes();
      assertTrue("Expecting " + NUM_DN + " Got: " + livenodes,
              cluster.getNamesystem().getNumLiveDataNodes() == NUM_DN);

      List<DataNodeProperties> dnpList = cluster.getDataNodeProperties();
      DataNodeProperties dnProps[] = new DataNodeProperties[NUM_DN];
      assert dnpList.size() == NUM_DN;
      for (int i = 0; i < dnpList.size(); i++) {
        dnProps[i] = dnpList.get(i);
      }

      //stopping all datanodes except one
      for (int i = 0; i < NUM_DN - 1; i++) {
        cluster.stopDataNode(0);
        LOG.info("Stopped a DN at port " + dnProps[i].ipcPort);
      }

      waitDNCount(cluster, 1);


      final TestCloudDNFailures.SlowWriter[] slowWriters =
        new TestCloudDNFailures.SlowWriter[numThreads];
      final Future[] futures = new Future[numThreads];
      for (int i = 0; i < numThreads; i++) {
        //create slow writers in different speed
        slowWriters[i] = new TestCloudDNFailures.SlowWriter(fs, new Path(dir, "file" + i),
          (i + 1) * 1000L,
                64 * 1024); //write
        futures[i] = es.submit(slowWriters[i]);
      }

      // sleep to make sure that the client is writing to the first DN
      sleepSeconds(10);


      int numfailures = NUM_DN - 1;
      assert numfailures <= dnProps.length;
      for (int i = 0; i < numfailures; i++) {
        LOG.info("Starting a datanode: " + dnProps[i]);
        cluster.restartDataNode(dnProps[i]);
        cluster.waitActive();
        waitDNCount(cluster, 2);
        LOG.info("Stopping a datanode");
        DataNodeProperties dnProp = cluster.stopDataNode(0);
        LOG.info("Stoped a datanode: " + dnProp);
        waitDNCount(cluster, 1);
        sleepSeconds(3); // write some more data
      }
      sleepSeconds(10); // write some more data

      LOG.info("Waiting for the threads to stop");
      for (int i = 0; i < slowWriters.length; i++) {
        slowWriters[i].stop();
      }

      for (Future f : futures) {
        try {
          f.get();
        } catch (Exception e){
          LOG.warn(e);
          fail(e.getMessage());
        }
      }

      for (int i = 0; i < slowWriters.length; i++) {
        slowWriters[i].verify();
      }

      //stop all datanodes
      assert cluster.getNamesystem().getNumLiveDataNodes() == 1;
      cluster.stopDataNode(0);

      LOG.info("HopsFS-Cloud. starting all the datanodes");
      //now start all datanodes. make sure that after the
      //block reporting is done no garbage "rbw" files are
      //left on the DNs
      for (DataNodeProperties dnp : dnProps) {
        cluster.restartDataNode(dnp);
      }
      cluster.waitActive();
      waitDNCount(cluster, dnProps.length);

      //make sure that all block reports are done and stray blocks have been deleted
      sleepSeconds(10);
      checkForStrayDNBlocks(cluster, NUM_DN);

    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  private void checkForStrayDNBlocks(MiniDFSCluster cluster, int numDataNodes) {

    for (int i = 0; i < numDataNodes; i++) {
      LOG.info("Checking DataNode Index: " + i);
      DataNode dn = cluster.getDataNodes().get(i);
      checkDNBR(cluster, dn);
    }
  }

  private void checkDNBR(MiniDFSCluster cluster, DataNode dn) {
    if (!dn.isDatanodeUp())
      fail("Datanode is not running");
    String poolId = cluster.getNamesystem().getBlockPoolId();
    Map<DatanodeStorage, BlockReport> br = dn.getFSDataset().getBlockReports(poolId);
    for (BlockReport reportedBlocks : br.values()) {
      if(reportedBlocks.getNumberOfBlocks() != 0){
        Iterator<Block> itr = reportedBlocks.blockIterable().iterator();
        while(itr.hasNext()){
          LOG.info("Block stored on DN is "+ itr.next());
        }
        fail(dn + " should have no blocks on disk. ");
      }
    }
  }
}
