/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.cache;

import io.hops.metadata.hdfs.entity.ProvidedBlockCacheLoc;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.fail;


@RunWith(Parameterized.class)
public class TestCloudCacheDNFailure {

  static final Log LOG = LogFactory.getLog(TestCloudCacheDNFailure.class);
  static String testBucketPrefix = "hops-test-TCCDNF";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudCacheDNFailure(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /*
  Test the cached block is moved to another datanode if the datanode dies
   */
  @Test
  public void TestCloudCacheDNFailure() throws IOException {

    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      final int BLK_SIZE = 128 * 1024;
      final int BLK_PER_FILE = 1;
      final int FILESIZE = BLK_PER_FILE * BLK_SIZE;
      final int NUM_DN = 5;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);

      // dead datanodes
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 2000);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 2L);
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_REPLICATION_PENDING_TIMEOUT_SEC_KEY, 2);
      conf.setInt(DFSConfigKeys.DFS_CLIENT_SOCKET_TIMEOUT_KEY, 5000);

      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      HopsFilesTestHelper.writeFile(dfs, "/dir/file", FILESIZE);
      CloudTestHelper.matchMetadata(conf);
      Map<Long, ProvidedBlockCacheLoc> locs =     CloudTestHelper.findCacheLocations();
      assert locs.size() == 1;
      int oldSid = locs.get(0L).getStorageID();

      String poolId = cluster.getNamesystem().getBlockPoolId();
      int dnToKill = -1;
      for (int i = 0; i < NUM_DN; i++) {
         int cachedFiles =
                 ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset()).
                         getCloudVolume().getBlockPoolSlice(poolId).
                         getProvidedBlocksCacheCleaner().getCachedFilesCount();
        if (cachedFiles!=0) {
          dnToKill = i;
          LOG.info("Killing  "+cluster.getDataNodes().get(i)
                  .getFSDataset().getVolumes().get(0).getBasePath());
          break;
        }
      }

      cluster.stopDataNode(dnToKill);
      waitDNCount(cluster, NUM_DN - 1);

      // cache location is not changed yet
      locs =     CloudTestHelper.findCacheLocations();
      assert locs.size() == 1;
      assert oldSid == locs.get(0L).getStorageID();

      // read the file and the cache location should have changed by now
      verifyFile(dfs, "/dir/file", FILESIZE);
      locs =     CloudTestHelper.findCacheLocations();
      assert locs.size() == 1;
      assert oldSid != locs.get(0L).getStorageID();

    } catch (Exception e) {
      LOG.info(e, e);
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
