/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.blockreport;

import com.google.common.collect.Lists;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.BlockReport;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;
import java.util.Map;

import static junit.framework.TestCase.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudRBWBR {

  static final Log LOG = LogFactory.getLog(TestCloudRBWBR.class);
  static String testBucketPrefix = "hops-test-TCRBWBR";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudRBWBR(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  public void TestCloudRBWBR() throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 64 * 1024 * 1024;
      final int NUM_DN = 1;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);

      CloudTestHelper.createRandomBucket(conf,  testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      ProvidedBlocksChecker pbc =
              cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      assert pbc.getProvidedBlockReportsCount() == 0;
      pbc.scheduleBlockReportNow();

      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, 1);
      assertTrue("Exptected 1. Got: " + ret, 1 == ret);

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      FSDataOutputStream out = (FSDataOutputStream) dfs.create(new Path("/dir/file"), (short) 1);
      byte[] data = new byte[BLKSIZE + BLKSIZE / 2]; // 1 1/2 blocks
      out.write(data);

      String poolId = cluster.getNamesystem().getBlockPoolId();
      Map<DatanodeStorage, BlockReport> brs =
              cluster.getDataNodes().get(0).getFSDataset().getBlockReports(poolId);

      //there should be only one block in the BR for CLOUD volume
      for (DatanodeStorage storage : brs.keySet()) {
        BlockReport br = brs.get(storage);
        if (storage.getStorageType() == StorageType.CLOUD) {
          assert br.getNumberOfBlocks() == 1;
        } else {
          assert br.getNumberOfBlocks() == 0;
        }
      }

      CloudTestHelper.matchMetadata(conf, CloudTestHelper.ExpectedErrors.EXPECTING_UCB);

      CloudPersistenceProvider cloudConnector =
              CloudPersistenceProviderFactory.getCloudClient(conf);
      Map<BlockIDAndGSTuple, CloudBlock> cloudBlocksMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
      assert cloudBlocksMap.size() == 1;

      cluster.getDataNodes().get(0).scheduleAllBlockReport(0);
      Thread.sleep(10000);

      out.close();

      brs = cluster.getDataNodes().get(0).getFSDataset().getBlockReports(poolId);

      //there should be only one block in the BR for CLOUD volume
      for (DatanodeStorage storage : brs.keySet()) {
        BlockReport br = brs.get(storage);
        if (storage.getStorageType() == StorageType.CLOUD) {
          assert br.getNumberOfBlocks() == 0;
        } else {
          assert br.getNumberOfBlocks() == 0;
        }
      }

      cloudBlocksMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
      assert cloudBlocksMap.size() == 2;

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}





