/*
 * Copyright (C) 2022 HopsWorks.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.blockreport;

import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static junit.framework.TestCase.assertTrue;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudFailedMultipartUploads {

  static final Log LOG = LogFactory.getLog(TestCloudFailedMultipartUploads.class);
  static String testBucketPrefix = "hops-test-TCFMU";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudFailedMultipartUploads(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  @Test
  // Write some large files. the datanodes will use multipart api to upload
  // the block to the clould. This will result in abandoned multipart uploads.
  // The block reporting system should detect this and delete the abandoned multipart
  // upload
  //
  public void testFailedMultipartUploads() throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 32 * 1024 * 1024;
      final int NUM_DN = 2;
      final int prefixSize = 10;
      long deleteAbandonedBlocksAfter = 2* 60 * 1000;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, prefixSize);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              deleteAbandonedBlocksAfter);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_DELETE_ABANDONED_MULTIPART_FILES_AFTER,
              deleteAbandonedBlocksAfter); //two mins

      CloudTestHelper.createRandomBucket(conf,  testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      int numFiles = 10;
      int fileSize = BLKSIZE - (1024 * 1024);
      FSDataOutputStream out[] = new FSDataOutputStream[numFiles];
      byte[] data = new byte[fileSize];
      for (int i = 0; i < numFiles; i++) {
        out[i] = dfs.create(new Path("/dir/file" + i), (short) 1);
        out[i].write(data);
      }

      CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);

      //kill one datanode
      //There gotta be one open block on each datanode
      cluster.stopDataNode(0);

      for (int i = 0; i < numFiles; i++) {
        out[i].close();
      }

      long abandonStartTime = System.currentTimeMillis();


      if (defaultCloudProvider == CloudProvider.AWS || defaultCloudProvider == CloudProvider.GCS) {
        // make sure that there are open multipart blocks
        int activeMultipartUploads =
                cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                        CloudHelper.ROOT_PREFIX).size();
        LOG.info("Number of open multipart uploads are "+activeMultipartUploads);
        assertTrue("Expecting 0 multipart uploads. Got: " + activeMultipartUploads,
                activeMultipartUploads != 0);
      } else if (defaultCloudProvider == CloudProvider.AZURE){
        //nothing
      } else {
        throw new UnsupportedOperationException("Clound provider not supported");
      }

      for (int i = 0; i < numFiles; i++) {
        FileStatus stat = dfs.getFileStatus(new Path("/dir/file" + i));
        assert stat.getLen() == fileSize;
      }

      CloudTestHelper.matchMetadata(conf);

      if (defaultCloudProvider == CloudProvider.AWS || defaultCloudProvider == CloudProvider.GCS) {
        while ((System.currentTimeMillis() - abandonStartTime) < 2 * deleteAbandonedBlocksAfter) {
          if (cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet())
                  , CloudHelper.ROOT_PREFIX).size() == 0) {
            return; //pass
          }
          Thread.sleep(10 * 1000);
        }

        fail("Abandoned blocks were not cleaned by the block reporting system. Active multipart " +
                "uploads: "+cloud.listMultipartUploads(Lists.newArrayList(CloudHelper.getAllBuckets().keySet()),
                CloudHelper.ROOT_PREFIX).size());

      } else if (defaultCloudProvider == CloudProvider.AZURE) {
        //nothing
      } else {
        throw new UnsupportedOperationException("Clound provider not supported");
      }

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}





