/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.blockreport;

import com.google.common.collect.Lists;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.hdfs.server.protocol.ReceivedDeletedBlockInfo;
import org.apache.hadoop.hdfs.server.protocol.StorageReceivedDeletedBlocks;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;

import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudDanglingIBR {

  static final Log LOG = LogFactory.getLog(TestCloudDanglingIBR.class);
  static String testBucketPrefix = "hops-test-TCDBR";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudDanglingIBR(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /*
  * Handling of incremental BR of a block that doest not belong to any file
   */
  @Test
  public void TestCloudDanglingIBR() throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 64 * 1024 * 1024;
      final int NUM_DN = 1;
      final int prefixSize = 10;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, prefixSize);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);

      CloudTestHelper.createRandomBucket(conf,  testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      File file = new File(cluster.getDataDirectory()+"/tmp-blk");
      FileWriter outblk = new FileWriter(file);
      outblk.write("hello");
      outblk.close();

      String bucket = CloudHelper.getBucketsFromConf(conf).get(0);
      Block blk = new Block(1, 0, 1, bucket);
      String blkKey = CloudHelper.getBlockKey( prefixSize, blk);
      String metaKey = CloudHelper.getMetaFileKey( prefixSize, blk);

      HashMap<String, String> metadata = new HashMap<>();
      CloudPersistenceProvider cloudConnector =
              CloudPersistenceProviderFactory.getCloudClient(conf);
      cloudConnector.uploadObject(bucket, blkKey, file, metadata);
      cloudConnector.uploadObject(bucket, metaKey, file, metadata);


      String bpid = cluster.getNamesystem().getBlockPoolId();
      DatanodeRegistration nodeReg = cluster.getDataNodes().get(0).getDNRegistrationForBP(bpid);
      String storageID =
              ((CloudFsDatasetImpl)cluster.getDataNodes().get(0).getFSDataset()).getCloudVolume().getStorageID();
      DatanodeStorage datanodeStorage =
              ((CloudFsDatasetImpl)cluster.getDataNodes().get(0).getFSDataset()).getStorage(storageID);
      assert storageID != null;
      StorageReceivedDeletedBlocks[] receivedAndDeletedBlocks =
              new StorageReceivedDeletedBlocks[1];

      ReceivedDeletedBlockInfo[] blocks = new ReceivedDeletedBlockInfo[1];
      blocks[0] = new ReceivedDeletedBlockInfo(blk,
              ReceivedDeletedBlockInfo.BlockStatus.RECEIVED_BLOCK, null );
      receivedAndDeletedBlocks[0] = new StorageReceivedDeletedBlocks(datanodeStorage, blocks);

      cluster.getNameNodeRpc().blockReceivedAndDeleted(nodeReg, bpid, receivedAndDeletedBlocks);

      Thread.sleep(10000);

      assert cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet())).size() == 0;

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}





