/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.blockreport;

import com.google.common.collect.Lists;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.entity.UnderReplicatedBlock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.*;

import static junit.framework.TestCase.assertTrue;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudBlockReportCorruptBlocks {

  static final Log LOG = LogFactory.getLog(TestCloudBlockReportCorruptBlocks.class);
  static String testBucketPrefix = "hops-test-TCBRCB";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;
  public TestCloudBlockReportCorruptBlocks(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @Before
  public void setup() {
  }

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /**
   * Block report detects corrupt blocks and adds the
   * block to the URB list with corrupt priority
   * <p>
   * Write some files --> change GS of some blocks in the cloud
   * --> Block report detects corrupt blocks
   *
   * @throws IOException
   */
  @Test
  public void TestCloudBlockReportCorruptBlocks() throws IOException {
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 128 * 1024;
      final int NUM_DN = 3;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);

      CloudTestHelper.createRandomBucket(conf,  testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      ProvidedBlocksChecker pbc =
              cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      assert pbc.getProvidedBlockReportsCount() == 0;
      pbc.scheduleBlockReportNow();

      CloudBlockReportTestHelper.waitForBRCompletion(pbc, 1);
      assert pbc.getProvidedBlockReportsCount() == 1;

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      for (int i = 0; i < 1; i++) {
        writeFile(dfs, "/dir/file" + i, BLKSIZE * 10);
      }

      CloudTestHelper.matchMetadata(conf);

      CloudBlockReportTestHelper.changeGSOfCloudObjs(conf,0 , 5);
      Thread.sleep(2000);

      CloudPersistenceProvider cloudConnector =
              CloudPersistenceProviderFactory.getCloudClient(conf);
      Map<BlockIDAndGSTuple, CloudBlock> cloudBlocksMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
      Map<BlockIDAndGSTuple, BlockInfoContiguous> dbBlocksMap = new HashMap<>();
      Map<Long, UnderReplicatedBlock> corruptBlkMap = new HashMap<>();
      pbc.findAllBlocksRange(0, 1000, dbBlocksMap, corruptBlkMap);

      assert cloudBlocksMap.size() == 10;
      assert dbBlocksMap.size() == 10;
      assert corruptBlkMap.size() == 0;

      List<BlockInfoContiguous> toMissing = new ArrayList<>();
      List<ProvidedBlocksChecker.BlockToMarkCorrupt> toCorrupt = new ArrayList<>();
      List<BlockInfoContiguous> toUnCorrupt = new ArrayList<>();
      List<CloudBlock> toDelete = new ArrayList<>();
      pbc.reportDiff(dbBlocksMap, cloudBlocksMap, corruptBlkMap, toMissing, toCorrupt, toDelete,
              toUnCorrupt);

      assertTrue("Exptected 0. Got: " + toDelete.size(), toDelete.size() == 0);
      assertTrue("Exptected 0. Got: " + toMissing.size(), toMissing.size() == 0);
      assertTrue("Exptected 5. Got: " + toCorrupt.size(), toCorrupt.size() == 5);
      assertTrue("Exptected 0. Got: " + toUnCorrupt.size(), toUnCorrupt.size() == 0);

      long brCount = pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();
      CloudBlockReportTestHelper.waitForBRCompletion(pbc, brCount + 1);
      assert pbc.getProvidedBlockReportsCount() == brCount + 1;

      //check
      assert cluster.getNamesystem().getMissingBlocksCount() == 5;

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}





