/*
 * Copyright (C) 2022 HopsWorks
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdfs.server.namenode.cloud.blockreport;

import com.google.common.collect.Lists;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.HdfsVariables;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.INodeDataAccess;
import io.hops.metadata.hdfs.entity.UnderReplicatedBlock;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.lock.TransactionLocks;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguousUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderGCSImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeFile;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.*;

import static junit.framework.TestCase.assertTrue;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;

public class BlkRptMultipleBlkVersionsHelper {

  static final Log LOG = LogFactory.getLog(BlkRptMultipleBlkVersionsHelper.class);


  public void testBlkRptMultipleBlkVersions(String testname, CloudProvider defaultCloudProvider,
                                            String testBucketPrefix,
                                            boolean missingBlkFiles) throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.ALL);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(CloudPersistenceProviderGCSImpl.class).setLevel(Level.DEBUG);
    Logger.getLogger(HdfsVariables.class).setLevel(Level.DEBUG);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 1 * 1024 * 1024;
      final int NUM_DN = 5;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY,
              0);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      CloudPersistenceProvider cloud = null;
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      DistributedFileSystem dfs = cluster.getFileSystem();

      ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      assert pbc.getProvidedBlockReportsCount() == 0;
      pbc.scheduleBlockReportNow();

      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, 1);
      assertTrue("Exptected 1. Got: " + ret, 1 == ret);

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int num_files = 10;
      int blks_perfile = 2;
      for (int i = 0; i < num_files; i++) {
        writeFile(dfs, "/dir/file" + i, BLKSIZE * blks_perfile);
      }
      CloudTestHelper.matchMetadata(conf);
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == num_files * blks_perfile;

      int newGS = 1001 + blks_perfile + 1;
      CloudBlockReportTestHelper.copyBlocksWithNewGS(conf, missingBlkFiles, 0,
              num_files*blks_perfile,
              newGS);
      int got = CloudTestHelper.getAllCloudBlocks(cloud).size();
      int expecting = num_files * blks_perfile * 2;
        assertTrue("Expecing: "+expecting+" Got: "+got, expecting == got);

      Thread.sleep(3000);

      //do a block report
      Map<BlockIDAndGSTuple, CloudBlock> cloudBlocksMap = cloud.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));

      Map<BlockIDAndGSTuple, BlockInfoContiguous> dbBlocksMap = new HashMap<>();
      Map<Long, UnderReplicatedBlock> corruptBlkMap = new HashMap<>();
      pbc.findAllBlocksRange(0, 1000, dbBlocksMap, corruptBlkMap);

      assert cloudBlocksMap.size() == num_files * blks_perfile * 2;
      assert dbBlocksMap.size() == num_files * blks_perfile;

      List<BlockInfoContiguous> toMissing = new ArrayList<>();
      List<ProvidedBlocksChecker.BlockToMarkCorrupt> toCorrupt = new ArrayList<>();
      List<CloudBlock> toDelete = new ArrayList<>();
      List<BlockInfoContiguous> toUnCorrupt = new ArrayList<>();
      pbc.reportDiff(dbBlocksMap, cloudBlocksMap, corruptBlkMap, toMissing, toCorrupt, toDelete,
              toUnCorrupt);

      assertTrue("Exptected 10. Got: " + toDelete.size(), toDelete.size() == num_files * blks_perfile);
      assertTrue("Exptected 0. Got: " + toMissing.size(), toMissing.size() == 0);
      assertTrue("Exptected 0. Got: " + toCorrupt.size(), toCorrupt.size() == 0);
      assertTrue("Exptected 0. Got: " + toUnCorrupt.size(), toUnCorrupt.size() == 0);

      for(CloudBlock cb: toDelete) {
        assert cb.getBlock().getGenerationStamp() == newGS;
      }

      // what if the files are open, in that case the block report should not try to delete the
      // blocks as long as the files are open
      //
      markFilesAsUnderConstruction(num_files);

      cloudBlocksMap = cloud.getAll(CloudHelper.ROOT_PREFIX,
              Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
      dbBlocksMap.clear();
      corruptBlkMap.clear();
      pbc.findAllBlocksRange(0, 1000, dbBlocksMap, corruptBlkMap);
      toMissing = new ArrayList<>();
      toCorrupt = new ArrayList<>();
      toDelete = new ArrayList<>();
      toUnCorrupt = new ArrayList<>();
      pbc.reportDiff(dbBlocksMap, cloudBlocksMap, corruptBlkMap, toMissing, toCorrupt, toDelete,
              toUnCorrupt);

      assertTrue("Exptected 0. Got: " + toDelete.size(), toDelete.size() == 0);
      assertTrue("Exptected 0. Got: " + toMissing.size(), toMissing.size() == 0);
      assertTrue("Exptected 0. Got: " + toCorrupt.size(), toCorrupt.size() == 0);
      assertTrue("Exptected 0. Got: " + toUnCorrupt.size(), toUnCorrupt.size() == 0);

    }
    catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  public void testBlkRptMultipleBlkVer(String testname, CloudProvider defaultCloudProvider,
                                            String testBucketPrefix, boolean missingBlkFiles) throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.ALL);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 1 * 1024 * 1024;
      final int NUM_DN = 5;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY,
              0);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      CloudPersistenceProvider cloud = null;
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));

      DistributedFileSystem dfs = cluster.getFileSystem();

      ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      assert pbc.getProvidedBlockReportsCount() == 0;
      pbc.scheduleBlockReportNow();

      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, 1);
      assertTrue("Exptected 1. Got: " + ret, 1 == ret);

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int num_files = 10;
      int blks_perfile = 2;
      for (int i = 0; i < num_files; i++) {
        writeFile(dfs, "/dir/file" + i, BLKSIZE * blks_perfile);
      }
      CloudTestHelper.matchMetadata(conf);
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == num_files * blks_perfile;

      CloudBlockReportTestHelper.copyBlocksWithNewGS(conf, missingBlkFiles, 0,
              num_files*blks_perfile, 5000,
              5001);
      Thread.sleep(3000);
      int got = CloudTestHelper.getAllCloudBlocks(cloud).size();
      int expecting =num_files * blks_perfile * 3;
      assertTrue("Expecing: "+expecting+" Got: "+got, expecting == got);


      long count =  pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();
      ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, count+1);
      assertTrue("Exptected 1. Got: " + ret, count+1 == ret);

      Thread.sleep(10000); // wait for the blocks to be deleted

      got = CloudTestHelper.getAllCloudBlocks(cloud).size();
      expecting =num_files * blks_perfile;
      assertTrue("Expecing: "+expecting+" Got: "+got, expecting == got);


      for (int i = 0; i < num_files; i++) {
        verifyFile(dfs, "/dir/file" + i, BLKSIZE * blks_perfile);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  private static List<INode> markFilesAsUnderConstruction(int count) throws IOException {
    HopsTransactionalRequestHandler handler =
            new HopsTransactionalRequestHandler(HDFSOperationType.TEST) {
              @Override
              public void acquireLock(TransactionLocks locks) throws IOException {

              }

              @Override
              public Object performTask() throws IOException {


                BlockInfoDataAccess bida = (BlockInfoDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockInfoDataAccess.class);
                List< BlockInfoContiguous> allblks = bida.findAllBlocks();
                int i  = 0;
                List<BlockInfoContiguous> modifiedBlks = new ArrayList<>();
                for(BlockInfoContiguous blk : allblks){
                  BlockInfoContiguousUnderConstruction uc =
                          new BlockInfoContiguousUnderConstruction(blk, blk.getInodeId());
                  modifiedBlks.add(uc);
                }
                bida.prepare(Collections.EMPTY_LIST, Collections.EMPTY_LIST, modifiedBlks);

                INodeDataAccess ida = (INodeDataAccess) HdfsStorageFactory
                        .getDataAccess(INodeDataAccess.class);
                List<INode> inodes = ida.allINodes();
                List<INode> modifiedFiles = new ArrayList<INode>();
                i = 0;
                for(INode inode : inodes){
                  if(inode.isFile()){
                    INodeFile file = (INodeFile)inode;
                    INodeFile modifiedFile = file.toUnderConstruction("clientname", "machine");
                    modifiedFiles.add(modifiedFile);
                    i ++;
                    if(i>= count){
                      break;
                    }
                  }
                }
                ida.prepare(Collections.EMPTY_LIST, Collections.EMPTY_LIST, modifiedFiles);
                return null;
              }
            };
    return (List<INode>) handler.handle();
  }
}





