/*
 * Copyright (C) 2022 HopsWorks 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.append;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.http.impl.conn.Wire;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudReplaceDNPolicy {

  static final Log LOG = LogFactory.getLog(TestCloudReplaceDNPolicy.class);

  static String testBucketPrefix = "hops-test-TCRDNP";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudReplaceDNPolicy(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);


  /**
   * In the cloud we have
   * dfs.client.block.write.replace-datanode-on-failure.enable = false
   * dfs.client.block.write.replace-datanode-on-failure.policy = NEVER
   * <p>
   * this caused problem in pipeline recovery.
   * <p>
   * the following test are for proof
   */
  @Test
  public void TestCloudReplaceDNPolicy1() throws IOException {
    try {
      testReplaceDNPolicy(false, "NEVER");
    } catch (Exception e) {
      if (!e.getMessage().contains("Failing write. Tried pipeline recovery")) {
        fail("This test is expected to fail");
      }
    }
  }

  @Test
  public void TestCloudReplaceDNPolicy2() throws IOException {
    try {
      testReplaceDNPolicy(true, "NEVER");
    } catch (Exception e) {
      fail("This test is not expected to fail");
    }
  }

  @Test
  public void TestCloudReplaceDNPolicy3() throws IOException {
    try {
      testReplaceDNPolicy(true, "DEFAULT");
    } catch (Exception e) {
      fail("This test is not expected to fail");
    }
  }

  public void testReplaceDNPolicy(boolean enable, String policy) throws Exception {
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(Wire.class).setLevel(Level.ERROR);
    Logger.getRootLogger().setLevel(Level.INFO);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      Configuration conf = new HdfsConfiguration();
      final int BLK_SIZE = 10 * 1024 * 1024;
      final int NUM_DN = 5;

      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
      //if a datanode fails then the unfinished block report entry will linger for some time
      //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
      //able to block report. Reducing the BR Max process time to quickly reclaim
      //unfinished block reports
      conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5 * 1000);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);


      conf.setBoolean(DFSConfigKeys.DFS_CLIENT_WRITE_REPLACE_DATANODE_ON_FAILURE_ENABLE_KEY,
        enable);
      conf.set(DFSConfigKeys.DFS_CLIENT_WRITE_REPLACE_DATANODE_ON_FAILURE_POLICY_KEY,
        policy);

      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN).
        storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));

      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int initialSize = FSNamesystem.getMaxSmallFileSize() + 1;

      writeFile(dfs, "/dir/file1", initialSize);  // write to cloud

      CloudTestHelper.matchMetadata(conf);

      // find the datanode responsible for the block
      String poolId = cluster.getNamesystem().getBlockPoolId();
      int dnToKill = -1;
      for (int i = 0; i < NUM_DN; i++) {
        int cachedFiles =
          ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset()).
            getCloudVolume().getBlockPoolSlice(poolId).
            getProvidedBlocksCacheCleaner().getCachedFilesCount();
        if (cachedFiles != 0) {
          dnToKill = i;
          LOG.info("Datanode that will be killed " + cluster.getDataNodes().get(i)
            .getFSDataset().getVolumes().get(0).getBasePath());
          break;
        }
      }

      // append data
      FSDataOutputStream out = dfs.append(new Path("/dir/file1"));
      HopsFilesTestHelper.writeData(out, initialSize, BLK_SIZE);

      //now kill the datanode that has the block
      cluster.stopDataNode(dnToKill);
      out.close();
      verifyFile(cluster.getNewFileSystemInstance(0), "/dir/file1",
        initialSize + BLK_SIZE);
      waitDNCount(cluster, NUM_DN - 1);

      // trigger block report to clean the old versions of the block
      ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();
      long count = pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();
      CloudBlockReportTestHelper.waitForBRCompletion(pbc, count + 1);
      Thread.sleep(10000); //wait for DN to process the commands


      CloudTestHelper.matchMetadata(conf, CloudTestHelper.ExpectedErrors.EXPECTING_MISSING_CACHE_LOCS);
      assert CloudTestHelper.findAllBlocks().size() == 2;

    } catch (Exception e) {
      throw e;
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
