/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.append;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.http.impl.conn.Wire;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudNotEnoughDNs {

  static final Log LOG = LogFactory.getLog(TestCloudNotEnoughDNs.class);

  static String testBucketPrefix = "hops-test-TCNEDN";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudNotEnoughDNs(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*90);

  /*
  Test that the client fails gracefully if no datanodes are found to fix the pipeline
   */
  @Test
  public void TestCloudNotEnoughDNs() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.ALL);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.ALL);
    Logger.getLogger(Wire.class).setLevel(Level.ERROR);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      final int BLK_SIZE = 10 * 1024 * 1024;
      final int NUM_DN = 2;
      MiniDFSCluster.DataNodeProperties DN_PROPS1 = null;
      MiniDFSCluster.DataNodeProperties DN_PROPS2 = null;
      int dataWritten = 0;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);

      // dead datanodes
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 2000);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 2L);
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_REPLICATION_PENDING_TIMEOUT_SEC_KEY, 2);
      conf.setInt(DFSConfigKeys.DFS_CLIENT_SOCKET_TIMEOUT_KEY, 5000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY,
              1 * 1000);

      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      //set low lease recovery times for testing
      cluster.setLeasePeriod(3 * 1000, 10 * 1000);

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));
      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      HopsFilesTestHelper.writeFile(dfs, "/dir/file", 100 * 1024);
      dataWritten += 100 * 1024;

      String poolId = cluster.getNamesystem().getBlockPoolId();
      int dnToKill = -1;
      for (int i = 0; i < NUM_DN; i++) {
        int cachedFiles =
                ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset()).
                        getCloudVolume().getBlockPoolSlice(poolId).
                        getProvidedBlocksCacheCleaner().getCachedFilesCount();
        if (cachedFiles != 0) {
          dnToKill = i;
          LOG.info("Datanode that will be killed " + cluster.getDataNodes().get(i)
                  .getFSDataset().getVolumes().get(0).getBasePath());
          break;
        }
      }

      // append more data to the file and then kill the dn and the client
      int appendSize = 1024 * 1024;
      FSDataOutputStream out = dfs.append(new Path("/dir/file"));
      HopsFilesTestHelper.writeData(out, 100 * 1024, appendSize);
      dataWritten += appendSize;
      DN_PROPS1 = cluster.stopDataNode(dnToKill);
      out.close();
      HopsFilesTestHelper.verifyFile(cluster.getNewFileSystemInstance(0), "/dir/file", dataWritten);

      // try again and kill the remaining DN
      dnToKill = -1;
      for (int i = 0; i < NUM_DN - 1; i++) {
        int cachedFiles =
                ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset()).
                        getCloudVolume().getBlockPoolSlice(poolId).
                        getProvidedBlocksCacheCleaner().getCachedFilesCount();
        if (cachedFiles != 0) {
          dnToKill = i;
          LOG.info("Datanode that will be killed " + cluster.getDataNodes().get(i)
                  .getFSDataset().getVolumes().get(0).getBasePath());
          break;
        }
      }

      out = dfs.append(new Path("/dir/file"));
      try {
        HopsFilesTestHelper.writeData(out, 100 * 1024, appendSize);
        DN_PROPS2 = cluster.stopDataNode(dnToKill);
        out.close();
        dataWritten += appendSize;
        fail();
      } catch (IOException e) {
        e.printStackTrace();
      }

      //kill the client first
      dfs.getClient().getLeaseRenewer().interruptAndJoin();
      dfs.getClient().abort();
      LOG.info("Aborted the client");

      long startTime = System.currentTimeMillis();
      while (true) {
        if ((System.currentTimeMillis() - startTime) < 60 * 1000) {
          if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
            break;
          }
        }
        Thread.sleep(1000);
      }

      cluster.restartDataNode(DN_PROPS1);
      cluster.restartDataNode(DN_PROPS2);
      waitDNCount(cluster, NUM_DN);
      // this should not fail. New appended data is not visible after failure
      HopsFilesTestHelper.verifyFile(cluster.getNewFileSystemInstance(0), "/dir/file",
              dataWritten);

    } catch (Exception e) {
      LOG.info(e, e);
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
