/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.append;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.NameNodeRpcServer;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.hadoop.hdfs.server.protocol.DatanodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.ReceivedDeletedBlockInfo;
import org.apache.hadoop.hdfs.server.protocol.StorageReceivedDeletedBlocks;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;

@RunWith(Parameterized.class)
public class TestCloudIBRForAppend {

  static final Log LOG = LogFactory.getLog(TestCloudIBRForAppend.class);

  static String testBucketPrefix = "hops-test-TCIBRA";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudIBRForAppend(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*90);

  @Test
  public void TestCloudIBRForAppend() throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.INFO);

    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {

      final int BLKSIZE = 32 * 1024 * 1024;
      final int NUM_DN = 1;

      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLKSIZE);

      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_THREAD_SLEEP_INTERVAL_KEY, 1000);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BR_SUB_TASKS_SIZE_KEY, 10*5);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_KEY,
              DFSConfigKeys.DFS_CLOUD_BLOCK_REPORT_DELAY_DEFAULT);
      conf.setLong(DFSConfigKeys.DFS_NAMENODE_BLOCKID_BATCH_SIZE, 10);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY,
              1 * 1000);
      String bucket = CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN)
              .storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      CloudPersistenceProvider cloud = null;
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      final FSNamesystem fsNamesystem = cluster.getNameNode().getNamesystem();
      final FSNamesystem fsNamesystemSpy = Mockito.spy(fsNamesystem);
      NameNodeRpcServer rpcServer = (NameNodeRpcServer) cluster.getNameNode().getRpcServer();
      rpcServer.setFSNamesystem(fsNamesystemSpy);


      boolean blocked[] = new boolean[1];
      blocked[0] = false;
      Answer delayer = new Answer() {
        @Override
        public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
          DatanodeRegistration nodeReg = (DatanodeRegistration) invocationOnMock.getArguments()[0];
          StorageReceivedDeletedBlocks r = (StorageReceivedDeletedBlocks) invocationOnMock.getArguments()[1];
          LOG.info("IBR size " + r.getBlocks().length);

          for (ReceivedDeletedBlockInfo rdbi : r.getBlocks()) {
            if (rdbi.getStatus() == ReceivedDeletedBlockInfo.BlockStatus.APPENDING) {
              blocked[0] =true;
              break;
            }
          }
          while(blocked[0]) {
            LOG.info("Blocked. Blocks "+Arrays.toString(r.getBlocks()));
            Thread.sleep(1000);
          }
          return invocationOnMock.callRealMethod();
        }
      };
      Mockito.doAnswer(delayer).when(fsNamesystemSpy).processIncrementalBlockReport(any(), any());
      DistributedFileSystem dfs = cluster.getFileSystem();

      int initialSize = 1024 * 1024;
      writeFile(dfs, "/dir/file1", initialSize);

      FSDataOutputStream out = dfs.append(new Path("/dir/file1"));
      HopsFilesTestHelper.writeData(out, initialSize, initialSize);
      Thread.sleep(5000);
      //this will generate the BlockStatus.APPENDING IBR.

      // Delete the files metadata and release the IBR
      CloudBlockReportTestHelper.deleteFileMetadata("file1");
      //kill the client
      dfs.getClient().getLeaseRenewer().interruptAndJoin();
      dfs.getClient().abort();
      LOG.info("Aborted the client");

      //blk_0_1001 and blk_0_1002
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == 2;

      blocked[0] = false;
      //This will schedule deletion for blk_0_1002

      Thread.sleep(10000);
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == 1;

      ProvidedBlocksChecker pbc = cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();
      long count = pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();
      CloudBlockReportTestHelper.waitForBRCompletion(pbc, count + 1);
      Thread.sleep(10000);
      assert CloudTestHelper.getAllCloudBlocks(cloud).size() == 0;


    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
