/*
 * Copyright (C) 2022 HopsWorks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.append;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.hadoop.hdfs.server.namenode.CloudBlockReportTestHelper;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.cloud.TestClouds;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.junit.AfterClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.junit.rules.Timeout;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import java.io.IOException;
import java.util.Collection;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.fail;

@RunWith(Parameterized.class)
public class TestCloudAppendDNFailure {

  static final Log LOG = LogFactory.getLog(TestCloudAppendDNFailure.class);

  static String testBucketPrefix = "hops-test-TCADNF";

  @Parameterized.Parameters
  public static Collection<Object> configs() {
    return TestClouds.CloudProviders;
  }

  CloudProvider defaultCloudProvider = null;

  public TestCloudAppendDNFailure(CloudProvider cloudProvider) {
    this.defaultCloudProvider = cloudProvider;
  }

  @Rule
  public TestName testname = new TestName();

  @ClassRule
  public static Timeout classTimeout = Timeout.seconds(60*15);

  @Rule
  public Timeout timeout = Timeout.seconds(60*15);

  /*
   * Testing failure of datanodes
   * Append some data to the file. Before the file is closed kill the datanode.
   * Make sure that the client recovers from the failure and the file is closed
   * properly, that is, no data is lost as the client buffers the data.
   */
  @Test
  public void TestCloudAppendDNFailure1() throws IOException {
    testAppendDNFailure(2 * 1024 * 1024);
  }

  @Test
  public void TestCloudAppendDNFailure2() throws IOException {
    testAppendDNFailure(1024 * 4);
  }

  public void testAppendDNFailure(int appendSize) throws IOException {
    Logger.getRootLogger().setLevel(Level.INFO);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      Configuration conf = new HdfsConfiguration();
      final int BLK_SIZE = 32 * 1024 * 1024;
      final int NUM_DN = 3;

      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());

      // basically never clean blocks
      conf.setInt(DFSConfigKeys.DFS_DN_CLOUD_CACHE_DELETE_ACTIVATION_PRECENTAGE_KEY, 99);
      conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, NUM_DN);
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);

      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
      //if a datanode fails then the unfinished block report entry will linger for some time
      //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
      //able to block report. Reducing the BR Max process time to quickly reclaim
      //unfinished block reports
      conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5 * 1000);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN).
              storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));

      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      int initialSize = FSNamesystem.getMaxSmallFileSize() + 1;
      writeFile(dfs, "/dir/file1", initialSize);  // write to cloud
      int dataWritten = initialSize;
      CloudTestHelper.matchMetadata(conf);

      FSDataOutputStream out = dfs.append(new Path("/dir/file1"));
      HopsFilesTestHelper.writeData(out, initialSize, appendSize);
      dataWritten += appendSize;
      Thread.sleep(10000);

      //kill datanode
      String poolId = cluster.getNamesystem().getBlockPoolId();
      int dnToKill = -1;
      for (int i = 0; i < NUM_DN; i++) {
        ProvidedBlocksCacheCleaner cleaner =
                ((CloudFsDatasetImpl) (cluster.getDataNodes().get(i).getFSDataset())).
                        getCloudVolume().getBlockPoolSlice(poolId).getProvidedBlocksCacheCleaner();

        if (cleaner.getCachedFilesCount() > 0) {
          dnToKill = i;
          LOG.info("Choosing DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                  " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                  " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " to kill");
          break;
        }
      }

      if (dnToKill != -1) {
        cluster.stopDataNode(dnToKill);
        waitDNCount(cluster, NUM_DN - 1);
      }

      HopsFilesTestHelper.writeData(out, dataWritten, appendSize);
      dataWritten += appendSize;

      // write enough to cross block boundary and more
      HopsFilesTestHelper.writeData(out, dataWritten, (BLK_SIZE - dataWritten) + BLK_SIZE / 2);
      dataWritten += (BLK_SIZE - dataWritten) + BLK_SIZE / 2;
      Thread.sleep(5000);
      assert CloudTestHelper.findAllBlocks().size() == 2;

      //kill one more datanode
      dnToKill = -1;
      for (int i = 0; i < NUM_DN - 1; i++) {
        if (((CloudFsDatasetImpl) (cluster.getDataNodes().get(i).getFSDataset())).replicaExistsInVolumeMap(poolId, 2)) {
          dnToKill = i;
          LOG.info("Choosing DN: " + cluster.getDataNodes().get(i).getDatanodeUuid() +
                  " port: " + cluster.getDataNodes().get(i).getRpcPort() +
                  " dir: " + cluster.getDataNodes().get(i).getFSDataset().getVolumes().get(0).getBasePath() + " to kill");
          break;
        }
      }

      if (dnToKill != -1) {
        cluster.stopDataNode(dnToKill);
        waitDNCount(cluster, NUM_DN - 2);
      }

      HopsFilesTestHelper.writeData(out, dataWritten, BLK_SIZE / 2);
      dataWritten += BLK_SIZE / 2;
      out.close();
      verifyFile(dfs, "/dir/file1", dataWritten);
      assert CloudTestHelper.findAllBlocks().size() == 2;

      //do a block report to clean all the blocks left behind
      ProvidedBlocksChecker pbc =
              cluster.getNamesystem().getBlockManager().getProvidedBlocksChecker();

      long count = pbc.getProvidedBlockReportsCount();
      pbc.scheduleBlockReportNow();
      long ret = CloudBlockReportTestHelper.waitForBRCompletion(pbc, count + 1);
      Thread.sleep(5000);

      CloudTestHelper.matchMetadata(conf, CloudTestHelper.ExpectedErrors.EXPECTING_MISSING_CACHE_LOCS);
      verifyFile(dfs, "/dir/file1", dataWritten);

      CloudTestHelper.matchMetadata(conf);

    } catch (Exception e) {
      e.printStackTrace();
      fail(e.getMessage());
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @AfterClass
  public static void CleanUp() throws IOException {
    TestClouds.DeleteAllBuckets(testBucketPrefix);
  }
}
