/*
 * Copyright (C) 2022 HopsWorks 
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.namenode.cloud.append;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.CloudTestHelper;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.HopsFilesTestHelper;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ProvidedBlocksCacheCleaner;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderS3Impl;
import org.apache.http.impl.conn.Wire;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import static org.apache.hadoop.hdfs.HopsFilesTestHelper.verifyFile;
import static org.apache.hadoop.hdfs.HopsFilesTestHelper.writeFile;
import static org.apache.hadoop.hdfs.server.namenode.cloud.failures.TestCloudDNFailures.waitDNCount;
import static org.junit.Assert.assertTrue;

public class MultipleFailuresHelper {

  static final Log LOG = LogFactory.getLog(MultipleFailuresHelper.class);

  public void testMultipleFailures(String testname, CloudProvider defaultCloudProvider,
                                   String testBucketPrefix,
                                   final boolean clientFails, final boolean dnFails,
                                   final int initBlkSize) throws Exception {
    Logger.getRootLogger().setLevel(Level.DEBUG);
    Logger.getLogger(CloudFsDatasetImpl.class).setLevel(Level.ALL);
    Logger.getLogger(CloudPersistenceProviderS3Impl.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksChecker.class).setLevel(Level.DEBUG);
    Logger.getLogger(ProvidedBlocksCacheCleaner.class).setLevel(Level.ERROR);
    Logger.getLogger(Wire.class).setLevel(Level.ERROR);
    Logger.getLogger(BlockManager.class).setLevel(Level.DEBUG);
    Logger.getRootLogger().setLevel(Level.INFO);
    CloudTestHelper.purgeCloudData(defaultCloudProvider, testBucketPrefix);
    MiniDFSCluster cluster = null;
    try {
      Configuration conf = new HdfsConfiguration();
      final int BLK_SIZE = 128 * 1024 * 1024;
      final int NUM_DN = 5;

      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_STORE_SMALL_FILES_IN_DB_KEY, false);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, defaultCloudProvider.name());
      conf.setLong(DFSConfigKeys.DFS_BLOCK_SIZE_KEY, BLK_SIZE);
      conf.setInt(DFSConfigKeys.DFS_NAMENODE_HEARTBEAT_RECHECK_INTERVAL_KEY, 500);
      //if a datanode fails then the unfinished block report entry will linger for some time
      //before it is reclaimed. Untill the entry is reclaimed other datanodes will not be
      //able to block report. Reducing the BR Max process time to quickly reclaim
      //unfinished block reports
      conf.setLong(DFSConfigKeys.DFS_BR_LB_MAX_BR_PROCESSING_TIME, 5 * 1000);
      conf.setInt(DFSConfigKeys.DFS_BR_LB_MAX_CONCURRENT_BR_PER_NN, NUM_DN);
      conf.setLong(DFSConfigKeys.DFS_HEARTBEAT_INTERVAL_KEY, 1L);
      conf.setLong(DFSConfigKeys.DFS_CLOUD_MARK_BLOCKS_CORRUPT_OR_MISSING_AFTER_KEY, 0);
      conf.setBoolean(DFSConfigKeys.DFS_CLOUD_CONCURRENT_UPLOAD, false);
      CloudTestHelper.createRandomBucket(conf, testBucketPrefix, testname);
      setDatanodeFailurePolicy(conf);

      cluster = new MiniDFSCluster.Builder(conf).numDataNodes(NUM_DN).
        storageTypes(CloudTestHelper.genStorageTypes(NUM_DN)).format(true).build();
      cluster.waitActive();
      cluster.setLeasePeriod(30 * 1000, 35 * 1000);

      DistributedFileSystem dfs = cluster.getFileSystem();

      dfs.mkdirs(new Path("/dir"));

      dfs.setStoragePolicy(new Path("/dir"), "CLOUD");

      writeFile(dfs, "/dir/file1", initBlkSize);  // write to cloud

      CloudTestHelper.matchMetadata(conf);


      int appendSize = 1024 * 4;
      int times = clientFails ? 5 : NUM_DN;
      for (int run = 1; run < times; run++) {

        // append data
        FSDataOutputStream out = dfs.append(new Path("/dir/file1"));
        // the block is closed if hsync is called and the block size
        // is greater than 1 MB. We have to call hsync twice here
        // because the first hsync might close the block. The subsequent
        // hsync will open a new block.
        HopsFilesTestHelper.writeData(out, initBlkSize, appendSize);
        out.hsync();
        HopsFilesTestHelper.writeData(out, initBlkSize, appendSize);
        out.hsync();

        if (dnFails) {
          // find the datanode responsible for the block
          String poolId = cluster.getNamesystem().getBlockPoolId();
          int dnToKill = -1;
          for (int i = 0; i < cluster.getDataNodes().size(); i++) {
            int activeReplicas =
              ((CloudFsDatasetImpl) cluster.getDataNodes().get(i).getFSDataset()).getOpenReplicasCount(poolId);
            if (activeReplicas != 0) {
              dnToKill = i;
              LOG.info("Datanode that will be killed " + cluster.getDataNodes().get(i)
                .getFSDataset().getVolumes().get(0).getBasePath());
              break;
            }
          }

          assert dnToKill != -1;
          LOG.info("Run " + run + " Killing DN: " + dnToKill);
          cluster.stopDataNode(dnToKill);
          waitDNCount(cluster, NUM_DN - run);
        }

        if (clientFails) {
          // kill the client
          dfs.getClient().getLeaseRenewer().interruptAndJoin();
          dfs.getClient().abort();
          LOG.info("Aborted the client");
          dfs = (DistributedFileSystem) cluster.getNewFileSystemInstance(0);
        } else {
          out.close();
        }

        // wait for lease recovery
        long startTime = System.currentTimeMillis();
        while (true) {
          if ((System.currentTimeMillis() - startTime) < 120 * 1000) {
            if (cluster.getNamesystem().getLeaseManager().countLease() == 0) {
              break;
            }
          }
          Thread.sleep(1000);
          LOG.info("Waiting for lease recovery to finish");
        }

        assertTrue("The NN should have recoverd the lease for the file ",
          cluster.getNamesystem().getLeaseManager().countLease() == 0);

        // this should not fail.
        verifyFile(dfs, "/dir/file1",
          initBlkSize + (appendSize * (2 * run)));

      }

    } catch (Exception e) {
      throw e;
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  private void setDatanodeFailurePolicy(Configuration conf) {
    conf.setBoolean(DFSConfigKeys.DFS_CLIENT_WRITE_REPLACE_DATANODE_ON_FAILURE_ENABLE_KEY,
      true);
    conf.set(DFSConfigKeys.DFS_CLIENT_WRITE_REPLACE_DATANODE_ON_FAILURE_POLICY_KEY,
      "NEVER");
  }
}
