package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.collect.Lists;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.dal.INodeDataAccess;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.ProvidedBlocksChecker;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class CloudBlockReportTestHelper {
  static final Log LOG = LogFactory.getLog(CloudBlockReportTestHelper.class);

  public static long waitForBRCompletion(ProvidedBlocksChecker pbc, long count) throws IOException {
    try {
      long waitForFirstBR = 300;
      long value = -1;
      do {
        value = pbc.getProvidedBlockReportsCount();
        if (value == count) {
          value = count;
          break;
        }

        LOG.info("HopsFS-Cloud. BR waiting for block report counter to increase. Current count: "+value);
        Thread.sleep(1000);
        waitForFirstBR--;
      } while (waitForFirstBR > 0);

      if (value != count) {
        return -1;
      }

      waitForFirstBR = 300;
      do {
        if (pbc.getAllTasks().size() == 0 && !pbc.isBRInProgress()) {
          return value;
        }
        LOG.info("HopsFS-Cloud. BR waiting for block report tasks to finish");
        Thread.sleep(1000);
      } while (--waitForFirstBR > 0);

    } catch (InterruptedException e) {

    }
    return -1;
  }

  public static void changeGSOfCloudObjs(Configuration conf, int startLoc, int count) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
    Map<BlockIDAndGSTuple, CloudBlock> objMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
            Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
    List<BlockIDAndGSTuple> sortedList = new ArrayList<>(objMap.keySet());
    Collections.sort(sortedList);
    int corrupted = 0;

    for(int i = startLoc; i < sortedList.size(); i++){
      BlockIDAndGSTuple key = sortedList.get(i);
      CloudBlock blk = objMap.get(key);
      if (blk.isPartiallyListed()) {
        continue;
      }
      String srcBucket = blk.getBlock().getCloudBucket();
      String dstBucket = srcBucket;

      Block b = blk.getBlock();
      String srcBlkKey = CloudHelper.getBlockKey(prefixSize, b);
      String srcMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);

      BlockIDAndGSTuple tuple = CloudHelper.getIDAndGSFromKey(srcBlkKey);
      long newGS = tuple.getGs()+100;
      b.setGenerationStampNoPersistance(newGS);
      String dstBlkKey = CloudHelper.getBlockKey(prefixSize, b);
      String dstMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);

      cloudConnector.renameObject(srcBucket, dstBucket, srcBlkKey, dstBlkKey);
      cloudConnector.renameObject(srcBucket, dstBucket, srcMetaKey, dstMetaKey);

      if (++corrupted >= count) {
        return;
      }
    }

  }

  public static void changeBlkIDsOfCloudObjs(Configuration conf, int startLoc, int count,
                                             int idIncrement) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
    Map<BlockIDAndGSTuple, CloudBlock> objMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
            Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
    List<BlockIDAndGSTuple> sortedList = new ArrayList<>(objMap.keySet());
    Collections.sort(sortedList);
    int corrupted = 0;

    for(int i = startLoc; i < sortedList.size(); i++){
      BlockIDAndGSTuple key = sortedList.get(i);
      CloudBlock blk = objMap.get(key);
      if (blk.isPartiallyListed()) {
        continue;
      }
      String srcBucket = blk.getBlock().getCloudBucket();
      String dstBucket = srcBucket;

      Block b = blk.getBlock();
      String srcBlkKey = CloudHelper.getBlockKey(prefixSize, b);
      String srcMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);

      BlockIDAndGSTuple tuple = CloudHelper.getIDAndGSFromKey(srcBlkKey);
      long newID = tuple.getBlockID()+idIncrement;
      b.setBlockIdNoPersistance(newID);
      String dstBlkKey = CloudHelper.getBlockKey(prefixSize, b);
      String dstMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);

      cloudConnector.renameObject(srcBucket, dstBucket, srcBlkKey, dstBlkKey);
      cloudConnector.renameObject(srcBucket, dstBucket, srcMetaKey, dstMetaKey);

      if (++corrupted >= count) {
        return;
      }
    }

  }

  public static void deleteMetaObjects(Configuration conf, int startLoc,  int count) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
    Map<BlockIDAndGSTuple, CloudBlock> objMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
            Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
    List<BlockIDAndGSTuple> sortedList = new ArrayList<>(objMap.keySet());
    Collections.sort(sortedList);

    int corrupted = 0;
    for(int i = startLoc; i < sortedList.size(); i++){
      BlockIDAndGSTuple key = sortedList.get(i);
      CloudBlock blk = objMap.get(key);
      String srcBucket = blk.getBlock().getCloudBucket();
      Block b = blk.getBlock();
      String srcMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);
      cloudConnector.deleteObject(srcBucket, srcMetaKey);
      if (++corrupted >= count) {
        return;
      }
    }
  }

  public static void deleteBlocksAndMetaObjs(Configuration conf, int count) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
    Map<BlockIDAndGSTuple, CloudBlock> objMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
            Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
    int corrupted = 0;
    for (CloudBlock blk : objMap.values()) {
      String srcBucket = blk.getBlock().getCloudBucket();
      Block b = blk.getBlock();
      String srcMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);
      String srcBlockKey = CloudHelper.getBlockKey(prefixSize, b);
      cloudConnector.deleteObject(srcBucket, srcMetaKey);
      cloudConnector.deleteObject(srcBucket, srcBlockKey);
      if (++corrupted >= count) {
        return;
      }
    }
  }


  public static void copyBlocksWithNewGS(Configuration conf, boolean missingBlkFile, int startLoc,
                                         int count,
                                         int ...newGSs) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    CloudPersistenceProvider cloudConnector = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloudConnector.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
    Map<BlockIDAndGSTuple, CloudBlock> objMap = cloudConnector.getAll(CloudHelper.ROOT_PREFIX,
            Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
    List<BlockIDAndGSTuple> sortedList = new ArrayList<>(objMap.keySet());
    Collections.sort(sortedList);

    int corrupted = 0;
    for(int i = startLoc; i < sortedList.size(); i++){
      BlockIDAndGSTuple key = sortedList.get(i);
      CloudBlock blk = objMap.get(key);
      String srcBucket = blk.getBlock().getCloudBucket();
      Block b = blk.getBlock();
      String srcMetaKey = CloudHelper.getMetaFileKey(prefixSize, b);
      String srcBlkKey = CloudHelper.getBlockKey(prefixSize, b);

      Map<String, String> metadata = cloudConnector.getUserMetaData(b.getCloudBucket(), srcMetaKey);
      for(int gs: newGSs){
        Block newBlock = new Block(b);
        newBlock.setGenerationStampNoPersistance(gs);
        String newSrcMetaKey = CloudHelper.getMetaFileKey(prefixSize, newBlock);
        String newSrcBlkKey = CloudHelper.getBlockKey(prefixSize, newBlock);

        if (!missingBlkFile) {
          copyObject(cloudConnector, newBlock.getCloudBucket(), newBlock.getCloudBucket(),
            srcBlkKey, newSrcBlkKey, metadata);
        }
        copyObject(cloudConnector, newBlock.getCloudBucket(), newBlock.getCloudBucket(),
          srcMetaKey, newSrcMetaKey, metadata);
      }

      if (++corrupted >= count) {
        LOG.info("Created copies of "+ corrupted +" blocks with new GS");
        return;
      }
    }
  }

  public static void copyObject(CloudPersistenceProvider cloudConnector, String srcBucket,
                                String dstBucket, String srcKey, String dstKey,
                                Map<String, String> newObjMetadata) throws IOException {
    // Download the block and create a new copy of it
    Path tmpFile = Files.createTempFile("hopsfs_block_", "");
    cloudConnector.downloadObject(srcBucket, srcKey, tmpFile.toAbsolutePath().toFile());
    cloudConnector.uploadObject(dstBucket, dstKey, tmpFile.toAbsolutePath().toFile(), newObjMetadata);
    tmpFile.toAbsolutePath().toFile().delete();
  }

  public static List<INode> deleteFileMetadata(final String name) throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                INodeDataAccess ida = (INodeDataAccess) HdfsStorageFactory
                        .getDataAccess(INodeDataAccess.class);
                BlockInfoDataAccess bda = (BlockInfoDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockInfoDataAccess.class);
                List<INode> inodes = ida.findINodes(name);
                assert inodes.size() == 1;
                ida.deleteInode(name);
                bda.deleteBlocksForFile(inodes.get(0).getId());
                return null;
              }
            };
    return (List<INode>) handler.handle();
  }

}

