/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs;

import io.hops.common.INodeUtil;
import io.hops.exception.StorageException;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.dal.BlockInfoDataAccess;
import io.hops.metadata.hdfs.entity.INodeIdentifier;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.HopsTransactionalRequestHandler;
import io.hops.transaction.handler.LightWeightRequestHandler;
import io.hops.transaction.lock.LockFactory;
import io.hops.transaction.lock.TransactionLockTypes;
import io.hops.transaction.lock.TransactionLocks;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.fs.Options.Rename;
import org.apache.hadoop.hdfs.protocol.LocatedBlocks;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockManager;
import org.apache.hadoop.hdfs.server.namenode.NameNodeAdapter;
import org.junit.Test;

import java.io.*;

import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;

import static org.junit.Assert.*;

public class TestDFSRename {
  static int countLease(MiniDFSCluster cluster) throws IOException {
    return NameNodeAdapter.getLeaseManager(cluster.getNamesystem())
        .countLease();
  }
  
  final Path dir = new Path("/test/rename/");

  void list(FileSystem fs, String name) throws IOException {
    FileSystem.LOG.info("\n\n" + name);
    for (FileStatus s : fs.listStatus(dir)) {
      FileSystem.LOG.info("" + s.getPath());
    }
  }

  static void createFile(FileSystem fs, Path f) throws IOException {
    DataOutputStream a_out = fs.create(f);
    a_out.writeBytes("something");
    a_out.close();
  }
  
  @Test
  public void testRename() throws Exception {
    Configuration conf = new HdfsConfiguration();
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(2).build();
    try {
      FileSystem fs = cluster.getFileSystem();
      assertTrue(fs.mkdirs(dir));
      
      { //test lease
        Path a = new Path(dir, "a");
        Path aa = new Path(dir, "aa");
        Path b = new Path(dir, "b");

        createFile(fs, a);
        
        //should not have any lease
        assertEquals(0, countLease(cluster));

        DataOutputStream aa_out = fs.create(aa);
        aa_out.writeBytes("something");

        //should have 1 lease
        assertEquals(1, countLease(cluster));
        list(fs, "rename0");
        fs.rename(a, b);
        list(fs, "rename1");
        aa_out.writeBytes(" more");
        aa_out.close();
        list(fs, "rename2");

        //should not have any lease
        assertEquals(0, countLease(cluster));
      }

      { // test non-existent destination
        Path dstPath = new Path("/c/d");
        assertFalse(fs.exists(dstPath));
        assertFalse(fs.rename(dir, dstPath));
      }
      
      { // dst cannot be a file or directory under src
        // test rename /a/b/foo to /a/b/c
        Path src = new Path("/a/b");
        Path dst = new Path("/a/b/c");

        createFile(fs, new Path(src, "foo"));
        
        // dst cannot be a file under src
        assertFalse(fs.rename(src, dst));
        
        // dst cannot be a directory under src
        assertFalse(fs.rename(src.getParent(), dst.getParent()));
      }
      
      { // dst can start with src, if it is not a directory or file under src
        // test rename /test /testfile
        Path src = new Path("/testPrefix");
        Path dst = new Path("/testPrefixfile");

        createFile(fs, src);
        assertTrue(fs.rename(src, dst));
      }
      
      { // dst should not be same as src test rename /a/b/c to /a/b/c
        Path src = new Path("/a/b/c");
        createFile(fs, src);
        assertTrue(fs.rename(src, src));
        assertFalse(fs.rename(new Path("/a/b"), new Path("/a/b/")));
        assertTrue(fs.rename(src, new Path("/a/b/c/")));
      }
      fs.delete(dir, true);
    } finally {
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }
  
  /**
   * Check the blocks of dst file are cleaned after rename with overwrite
   */
  @Test(timeout = 120000)
  public void testRenameWithOverwrite() throws Exception {
    final short replFactor = 2;
    final long blockSize = 512;
    Configuration conf = new Configuration();
    MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).
        numDataNodes(replFactor).build();
    DistributedFileSystem dfs = cluster.getFileSystem();
    try {
      
      long fileLen = blockSize*3;
      String src = "/foo/src";
      String dst = "/foo/dst";
      Path srcPath = new Path(src);
      Path dstPath = new Path(dst);
      
      DFSTestUtil.createFile(dfs, srcPath, fileLen, replFactor, 1);
      DFSTestUtil.createFile(dfs, dstPath, fileLen, replFactor, 1);
      
      LocatedBlocks lbs = NameNodeAdapter.getBlockLocations(
          cluster.getNameNode(), dst, 0, fileLen);
      BlockManager bm = NameNodeAdapter.getNamesystem(cluster.getNameNode()).
          getBlockManager();
      assertTrue(getStoredBlockTX(bm, lbs.getLocatedBlocks().get(0).getBlock().
          getLocalBlock()) != null);
      BlockInfoContiguous block = getStoredBlockTX(bm, lbs.getLocatedBlocks().get(0).getBlock().
          getLocalBlock());
      dfs.rename(srcPath, dstPath, Rename.OVERWRITE);
      BlockInfoContiguous b = getBlock(block.getBlockId(), block.getInodeId());
      assertTrue(b == null);
    } finally {
      if (dfs != null) {
        dfs.close();
      }
      if (cluster != null) {
        cluster.shutdown();
      }
    }
  }

  @Test
  public void testRenameFileOverFile() throws Exception {
    final boolean format = true;

    Configuration conf = new HdfsConfiguration();
    conf.set(DFSConfigKeys.DFS_NDC_ENABLED_KEY, "true");
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(2).nameNodePort(8020).format(format).build();
    DistributedFileSystem dfs = cluster.getFileSystem();
    FileSystem.setDefaultUri(conf, "hdfs://localhost:8020");

    try {
      dfs.mkdirs(new Path("./A/B"));
      dfs.mkdirs(new Path("./A/C"));

      // It should be able to overwrite!
      dfs.create(new Path("./A/B/file")).close();
      dfs.create(new Path("./A/C/file")).close();
      dfs.rename(new Path("./A/B/file"), new Path("./A/C/file"), Rename.OVERWRITE);

      // The old rename method should not work as the file already exists!
      dfs.create(new Path("./A/B/file"), false).close();
      dfs.rename(new Path("./A/B/file"), new Path("./A/C/file"));
      dfs.getFileStatus(new Path("./A/B/file"));
    } finally {
      cluster.shutdown();
    }
  }

  @Test
  public void testRenameOverADifferentType() throws Exception {
    final boolean format = true;

    Configuration conf = new HdfsConfiguration();
    conf.set(DFSConfigKeys.DFS_NDC_ENABLED_KEY, "true");
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(2).nameNodePort(8020).format(format).build();
    DistributedFileSystem dfs = cluster.getFileSystem();
    FileSystem.setDefaultUri(conf, "hdfs://localhost:8020");

    try {
      dfs.mkdirs(new Path("./A/B"));
      dfs.mkdirs(new Path("./A/C/file"));
      dfs.create(new Path("./A/B/file")).close();
      // It should not work as "file" in C is a directory!
      Exception thrown = assertThrows(
          Exception.class,
          () -> dfs.rename(new Path("./A/B/file"), new Path("./A/C/file"), Rename.OVERWRITE)
      );
      assertTrue(thrown.getMessage().contains("must both be directories"));

      // It shouldn't work the other way, too!
      thrown = assertThrows(
          Exception.class,
          () -> dfs.rename(new Path("./A/C/file"), new Path("./A/B/file"), Rename.OVERWRITE)
      );
      assertTrue(thrown.getMessage().contains("must both be directories"));

    } finally {
      cluster.shutdown();
    }
  }

  @Test
  public void testRenameDir() throws Exception {
    final boolean format = true;
    Configuration conf = new HdfsConfiguration();
    conf.set(DFSConfigKeys.DFS_NDC_ENABLED_KEY, "true");
    MiniDFSCluster cluster =
        new MiniDFSCluster.Builder(conf).numDataNodes(2).nameNodePort(8020).format(format).build();
    DistributedFileSystem dfs = cluster.getFileSystem();
    FileSystem.setDefaultUri(conf, "hdfs://localhost:8020");

    try {
      dfs.mkdirs(new Path("./A/B"));
      dfs.mkdirs(new Path("./A/C"));
      dfs.mkdirs(new Path("./A/D"));
      dfs.create(new Path("./A/B/fileB")).close();
      dfs.create(new Path("./A/C/fileC")).close();
      dfs.create(new Path("./A/D/fileD")).close();

      // The old rename function should create A/D/B
      dfs.rename(new Path("./A/B"), new Path("./A/D"));
      dfs.getFileStatus(new Path("./A/D/B/fileB"));

      // The old rename function should not work as A/D/B already exists!
      dfs.mkdirs(new Path("./A/B"));
      dfs.create(new Path("./A/B/fileB2")).close();
      dfs.rename(new Path("./A/B"), new Path("./A/D"));
      dfs.getFileStatus(new Path("./A/B/fileB2"));

      // This should return error as A/D is not empty
      Exception thrown = assertThrows(
          Exception.class,
          () -> dfs.rename(new Path("./A/C"), new Path("./A/D"), Rename.OVERWRITE)
      );
      assertTrue(thrown.getMessage().contains("not empty"));

      // This should write all content of /A/C/ into A/E/ as /A/E/ is empty
      dfs.mkdirs(new Path("./A/E"));
      dfs.rename(new Path("./A/C"), new Path("./A/E"), Rename.OVERWRITE);
      dfs.getFileStatus(new Path("./A/E/fileC"));
    } finally {
      cluster.shutdown();
    }
  }

  BlockInfoContiguous getStoredBlockTX(BlockManager bm, Block block) throws IOException {
    HopsTransactionalRequestHandler handler =
            new HopsTransactionalRequestHandler(HDFSOperationType.TEST) {
      INodeIdentifier inodeIdentifier =  null;
      @Override
      public void setUp() throws StorageException {
        inodeIdentifier = INodeUtil.resolveINodeFromBlock(block);
      }

      @Override
      public void acquireLock(TransactionLocks locks) throws IOException {
        LockFactory lf = LockFactory.getInstance();
        locks.add(lf.getIndividualINodeLock(TransactionLockTypes.INodeLockType.WRITE, inodeIdentifier, true))
                .add(lf.getIndividualBlockLock(block.getBlockId(), inodeIdentifier));
      }

      @Override
      public Object performTask() throws IOException {

        return bm.getStoredBlock(block);
      }
    };
    return  (BlockInfoContiguous)handler.handle();
  }


  private BlockInfoContiguous getBlock(final long blockId, final long inodeId) throws IOException{
    return (BlockInfoContiguous) new LightWeightRequestHandler(HDFSOperationType.TEST) {
      @Override
      public Object performTask() throws IOException {
        BlockInfoDataAccess da = (BlockInfoDataAccess) HdfsStorageFactory.getDataAccess(BlockInfoDataAccess.class);
        return da.findById(blockId, inodeId);
      }
    }.handle();
  }
}
