package org.apache.hadoop.hdfs;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.StorageClass;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.models.AccessTier;
import com.lc.repackaged.com.google.cloud.storage.BlobId;
import com.lc.repackaged.com.google.cloud.storage.BlobInfo;
import com.lc.repackaged.com.google.cloud.storage.Storage;
import com.lc.repackaged.com.google.common.collect.Lists;
import io.hops.metadata.HdfsStorageFactory;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import io.hops.metadata.hdfs.dal.*;
import io.hops.metadata.hdfs.entity.*;
import io.hops.transaction.handler.HDFSOperationType;
import io.hops.transaction.handler.LightWeightRequestHandler;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CloudProvider;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguous;
import org.apache.hadoop.hdfs.server.blockmanagement.BlockInfoContiguousUnderConstruction;
import org.apache.hadoop.hdfs.server.blockmanagement.PendingBlockInfo;
import org.apache.hadoop.hdfs.server.blockmanagement.ReplicaUnderConstruction;
import org.apache.hadoop.hdfs.server.common.CloudHelper;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProvider;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.CloudFsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud.CloudPersistenceProviderFactory;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.junit.rules.TestName;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.ExecutionException;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

public class CloudTestHelper {
  static final Log LOG = LogFactory.getLog(CloudTestHelper.class);

  private static List<INode> findAllINodes() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                INodeDataAccess da = (INodeDataAccess) HdfsStorageFactory
                        .getDataAccess(INodeDataAccess.class);
                return da.allINodes();
              }
            };
    return (List<INode>) handler.handle();
  }


  public static Map<BlockIDAndGSTuple, BlockInfoContiguous> findAllBlocks() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                Map<BlockIDAndGSTuple, BlockInfoContiguous> blkMap = new HashMap<>();
                BlockInfoDataAccess da = (BlockInfoDataAccess) HdfsStorageFactory
                        .getDataAccess(BlockInfoDataAccess.class);

                List<BlockInfoContiguous> blocks = da.findAllBlocks();
                for (BlockInfoContiguous blk : blocks) {
                  blkMap.put(new BlockIDAndGSTuple(blk.getBlockId(), blk.getGenerationStamp()), blk);
                }
                return blkMap;
              }
            };
    return (Map<BlockIDAndGSTuple, BlockInfoContiguous>) handler.handle();
  }

  public static Map<Long, ProvidedBlockCacheLoc> findCacheLocations() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                ProvidedBlockCacheLocDataAccess da = (ProvidedBlockCacheLocDataAccess) HdfsStorageFactory
                        .getDataAccess(ProvidedBlockCacheLocDataAccess.class);

                Map<Long, ProvidedBlockCacheLoc> blkMap = da.findAll();
                return blkMap;
              }
            };
    return (Map<Long, ProvidedBlockCacheLoc>) handler.handle();
  }


  public static List<Replica> findAllReplicas() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                ReplicaDataAccess da = (ReplicaDataAccess) HdfsStorageFactory
                        .getDataAccess(ReplicaDataAccess.class);
                return da.findAll();
              }
            };
    return (List<Replica>) handler.handle();
  }


  private static List<ReplicaUnderConstruction> findAllReplicasUC() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                ReplicaUnderConstructionDataAccess da =
                        (ReplicaUnderConstructionDataAccess) HdfsStorageFactory
                                .getDataAccess(ReplicaUnderConstructionDataAccess.class);
                return da.findAll();
              }
            };
    return (List<ReplicaUnderConstruction>) handler.handle();
  }


  private static List<PendingBlockInfo> findAllPendingBlocks() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                PendingBlockDataAccess da = (PendingBlockDataAccess) HdfsStorageFactory
                        .getDataAccess(PendingBlockDataAccess.class);
                return da.findAll();
              }
            };
    return (List<PendingBlockInfo>) handler.handle();
  }

  private static List<CorruptReplica> findAllCorruptReplicas() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                CorruptReplicaDataAccess da = (CorruptReplicaDataAccess) HdfsStorageFactory
                        .getDataAccess(CorruptReplicaDataAccess.class);
                return da.findAll();
              }
            };
    return (List<CorruptReplica>) handler.handle();
  }

  private static List<ExcessReplica> findAllExcessReplicas() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                ExcessReplicaDataAccess da = (ExcessReplicaDataAccess) HdfsStorageFactory
                        .getDataAccess(ExcessReplicaDataAccess.class);
                return da.findAll();
              }
            };
    return (List<ExcessReplica>) handler.handle();
  }

  private static List<InvalidatedBlock> findAllInvalidatedBlocks() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                InvalidateBlockDataAccess da = (InvalidateBlockDataAccess) HdfsStorageFactory
                        .getDataAccess(InvalidateBlockDataAccess.class);
                return da.findAll();
              }
            };
    return (List<InvalidatedBlock>) handler.handle();
  }

  public static List<UnderReplicatedBlock> findAllUnderReplicatedBlocks() throws IOException {
    LightWeightRequestHandler handler =
            new LightWeightRequestHandler(HDFSOperationType.TEST) {
              @Override
              public Object performTask() throws IOException {
                UnderReplicatedBlockDataAccess da = (UnderReplicatedBlockDataAccess) HdfsStorageFactory
                        .getDataAccess(UnderReplicatedBlockDataAccess.class);
                return da.findAll();
              }
            };
    return (List<UnderReplicatedBlock>) handler.handle();
  }

  private static boolean match(Map<BlockIDAndGSTuple, CloudBlock> cloudView,
                               Map<BlockIDAndGSTuple, BlockInfoContiguous> dbView, boolean expectingUCB) {
    if (cloudView.size() != dbView.size()) {
      List cv = new ArrayList(cloudView.values());
      Collections.sort(cv);
      List dbv = new ArrayList(dbView.values());
      Collections.sort(dbv);
      LOG.info("HopsFS-Cloud Cloud Blocks " + Arrays.toString(cv.toArray()));
      LOG.info("HopsFS-Cloud DB Blocks " + Arrays.toString(dbv.toArray()));
    }

    long dbBlockClount = dbView.size();
    if(expectingUCB){
      for(BlockInfoContiguous blk : dbView.values()){
        if(blk instanceof  BlockInfoContiguousUnderConstruction){
          dbBlockClount--;
        }
      }
    }

    if( cloudView.size() != dbBlockClount) {
      LOG.info("DB View: "+Arrays.toString(dbView.keySet().toArray()));
      LOG.info("Cloud View: "+Arrays.toString(cloudView.keySet().toArray()));
      assertTrue("number of blocks did not match." +
                      " DB Size: " + dbBlockClount + " Cloud Size: " + cloudView.size(),
              cloudView.size() == dbBlockClount);
    }

    for (BlockIDAndGSTuple blkID : dbView.keySet()) {
      BlockInfoContiguous dbBlock = dbView.get(blkID);

      if(expectingUCB){
        if(dbBlock instanceof BlockInfoContiguousUnderConstruction){
          continue;
        }
      }

      CloudBlock cloudBlock = cloudView.get(blkID);

      assert !cloudBlock.isPartiallyListed();
      assert cloudBlock != null && dbBlock != null;
      assert cloudBlock.getBlock().getCloudBucket().compareToIgnoreCase(dbBlock.getCloudBucket()) == 0;
      assert cloudBlock.getBlock().getGenerationStamp() == dbBlock.getGenerationStamp();
      assertTrue("Size Mismatch. Cloud Size: "+cloudBlock.getBlock().getNumBytes()+" DB: "+dbBlock.getNumBytes(),
              cloudBlock.getBlock().getNumBytes() == dbBlock.getNumBytes());

      assert dbBlock.getBlockUCState() == HdfsServerConstants.BlockUCState.COMPLETE;
    }

    return true;
  }

  public enum ExpectedErrors {
    EXPECTING_UCB, // Expecting under construction blocks
    EXPECTING_MISSING_CACHE_LOCS, // Expecting the cache locations check to fail
    NONE
  }

  static boolean containsError(ExpectedErrors error, ExpectedErrors... expectedErrors) {
    for (ExpectedErrors e : expectedErrors) {
      if (e == error) {
        return true;
      }
    }
    return false;
  }

  public static boolean matchMetadata(Configuration conf) throws IOException {
    return matchMetadata(conf, ExpectedErrors.NONE );
  }

  public static boolean matchMetadata(Configuration conf, ExpectedErrors... expectedErrors) throws IOException {
    int prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    HashSet<ExpectedErrors> expectedErrorsSet = new HashSet<>();

    LOG.info("HopsFS-Cloud. CloudTestHelper. Checking Metadata");
    CloudPersistenceProvider cloud = null;

    try {
      cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
      cloud.checkAllBuckets(CloudHelper.getBucketsFromConf(conf));
      Map<BlockIDAndGSTuple, CloudBlock> cloudView = getAllCloudBlocks(cloud);
      Map<BlockIDAndGSTuple, BlockInfoContiguous> dbView = findAllBlocks();

      List sortDBBlocks = new ArrayList();
      sortDBBlocks.add(dbView.values());
      Collections.sort(sortDBBlocks);
      List sortCloudBlocks = new ArrayList();
      sortCloudBlocks.add(cloudView.values());
      Collections.sort(sortCloudBlocks);

      LOG.info("HopsFS-Cloud. DB View: " + sortDBBlocks);
      LOG.info("HopsFS-Cloud. Cloud View: " + sortCloudBlocks);

      LOG.info("HopsFS-Cloud. Matching Cloud and DB view");
      match(cloudView, dbView, containsError(ExpectedErrors.EXPECTING_UCB, expectedErrors));

      //block cache mapping
      Map<Long, ProvidedBlockCacheLoc> cacheLoc = findCacheLocations();
      if(!containsError(ExpectedErrors.EXPECTING_MISSING_CACHE_LOCS, expectedErrors)){
        assertTrue("Expecting size of the blocks and cache location locations to be same."+
                        "Blocks: " +dbView.size()+" Cache: "+cacheLoc.size(),
                cacheLoc.size() == dbView.size());
      }

      for (Block blk : dbView.values()) {
        if (blk instanceof BlockInfoContiguousUnderConstruction && containsError(ExpectedErrors.EXPECTING_UCB, expectedErrors)) {
          continue;
        }

        LOG.info("HopsFS-Cloud. Checking Block: " + blk);
        String bucket = blk.getCloudBucket();
        String blockKey = CloudHelper.getBlockKey(prefixSize, blk);
        String metaKey = CloudHelper.getMetaFileKey(prefixSize, blk);

        assert cloud.objectExists(bucket, blockKey) == true;
        assert cloud.objectExists(bucket, metaKey) == true;
        assertTrue("Expected " + blk.getNumBytes() + " Got: " + cloud.getObjectSize(bucket, blockKey),
                cloud.getObjectSize(bucket, blockKey) == blk.getNumBytes());

        Map<String, String> metadata = cloud.getUserMetaData(bucket, blockKey);
        assertTrue("No metadata expected. Got "+metadata.size(), metadata.size() == 0);

        metadata = cloud.getUserMetaData(bucket, metaKey);
        assert metadata.size() == 4;
        assertTrue("Expected: " + blk.getNumBytes() +
                        " Got: " + Long.parseLong(metadata.get(CloudFsDatasetImpl.OBJECT_SIZE)),
                Long.parseLong(metadata.get(CloudFsDatasetImpl.OBJECT_SIZE)) == blk.getNumBytes());

        assertTrue("Expected: " + blk.getGenerationStamp() +
                        " Got: " + Long.parseLong(metadata.get(CloudFsDatasetImpl.GEN_STAMP)),
                Long.parseLong(metadata.get(CloudFsDatasetImpl.GEN_STAMP)) == blk.getGenerationStamp());
        if(!containsError(ExpectedErrors.EXPECTING_MISSING_CACHE_LOCS, expectedErrors)){
          assert cacheLoc.get(blk.getBlockId()) != null;
        }
      }

      assert findAllReplicas().size() == 0;
      if (!containsError(ExpectedErrors.EXPECTING_UCB, expectedErrors))
        assert findAllReplicasUC().size() == 0;
      assert findAllExcessReplicas().size() == 0;
      assert findAllPendingBlocks().size() == 0;
      assert findAllCorruptReplicas().size() == 0;
      assert findAllInvalidatedBlocks().size() == 0;
      assert findAllUnderReplicatedBlocks().size() == 0;

    } finally {
      if (cloud != null) {
        cloud.shutdown();
      }
    }
    return true;
  }

  public static Map<BlockIDAndGSTuple, CloudBlock> getAllCloudBlocks(CloudPersistenceProvider cloud)
          throws IOException {
    return cloud.getAll(CloudHelper.ROOT_PREFIX, Lists.newArrayList(CloudHelper.getAllBuckets().keySet()));
  }

  public static StorageType[][] genStorageTypes(int numDataNodes) {

    StorageType[][] types = new StorageType[numDataNodes][];
    for (int i = 0; i < numDataNodes; i++) {
      types[i] = new StorageType[]{StorageType.CLOUD,StorageType.DISK};
    }
    return types;
  }

  public static void purgeCloudData(CloudProvider cloudProvider, String prefix) throws IOException {
    try {
      LOG.info("HopsFS-Cloud. Purging all cloud data. Prefix: "+prefix+" Cloud provider: "+cloudProvider);
      Configuration conf = new HdfsConfiguration();
      conf.setBoolean(DFSConfigKeys.DFS_ENABLE_CLOUD_PERSISTENCE, true);
      conf.set(DFSConfigKeys.DFS_CLOUD_PROVIDER, cloudProvider.name());
      CloudPersistenceProvider cloudConnector =
              CloudPersistenceProviderFactory.getCloudClient(conf);
      cloudConnector.deleteAllBuckets(prefix);
      cloudConnector.shutdown();
    } catch (ExecutionException  | InterruptedException e) {
      LOG.warn(e);
    }
  }

  public static void purgeCloudData(CloudProvider cloudProvider, String prefix,
                                    Configuration config) throws IOException {
    try {
      LOG.info("HopsFS-Cloud. Purging all cloud data. Prefix: "+prefix+" Cloud provider: "+cloudProvider);
      CloudPersistenceProvider cloudConnector =
        CloudPersistenceProviderFactory.getCloudClient(config);
      cloudConnector.deleteAllBuckets(prefix);
      cloudConnector.shutdown();
    } catch (ExecutionException  | InterruptedException e) {
      LOG.warn(e);
    }
  }

  public static String createRandomBucket(Configuration conf, String prefix, TestName name) throws IOException {
    return createRandomBucket(conf, prefix, name.getMethodName());
  }

  public static String createRandomBucket(Configuration conf, String prefix,
                                          String methodName) throws IOException {
    Date date = new Date();
    String cloudProvider = conf.get(DFSConfigKeys.DFS_CLOUD_PROVIDER,
            DFSConfigKeys.DFS_CLOUD_PROVIDER_DEFAULT);

    if(methodName.contains("[")){
      methodName = methodName.substring(0, methodName.indexOf("["));
    }

    String bucket = prefix + "-" + methodName +
            "-" + date.getHours() + date.getMinutes() + date.getSeconds();

    if(cloudProvider.compareToIgnoreCase(CloudProvider.AZURE.name()) == 0){
      conf.set(DFSConfigKeys.AZURE_CONTAINER_KEY, bucket.toLowerCase());
      LOG.info("Test Azure container is "+bucket);
    } else  if(cloudProvider.compareToIgnoreCase(CloudProvider.AWS.name()) == 0){
      conf.set(DFSConfigKeys.S3_BUCKET_KEY, bucket.toLowerCase());
      LOG.info("Test S3 bucket is "+bucket);
    } else  if(cloudProvider.compareToIgnoreCase(CloudProvider.GCS.name()) == 0){
      conf.set(DFSConfigKeys.GCS_BUCKET_KEY, bucket.toLowerCase());
      LOG.info("Test GCS bucket is "+bucket);
    }

    CloudPersistenceProvider cloud = CloudPersistenceProviderFactory.getCloudClient(conf);
    cloud.createBucket(bucket.toLowerCase());
    cloud.shutdown();

    return bucket.toLowerCase();
  }


  public static void moveToGlacier(AmazonS3Client s3, String bucket, String objectKey, File object,
                            Map<String, String> metadata) throws IOException {
    try {
      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey);
      }

      long startTime = System.currentTimeMillis();
      PutObjectRequest putReq = new PutObjectRequest(bucket,
              objectKey, object);
      putReq.setStorageClass(StorageClass.Glacier);

      // Upload a file as a new object with ContentType and title specified.
      ObjectMetadata objMetadata = new ObjectMetadata();
      objMetadata.setContentType("plain/text");
      objMetadata.setUserMetadata(metadata);
      putReq.setMetadata(objMetadata);

      s3.putObject(putReq);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException e) {
      throw new IOException(e);
    } catch (SdkClientException e) {
      throw new IOException(e);
    }
  }

  public static void moveToArchive(BlobServiceClient blobClient, String container, String objectKey,
                                   File object,
                                   Map<String, String> metadata) throws IOException {
    try {
      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Container: " + container + " Object Key: " + objectKey);
      }
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);
      bc.uploadFromFile(object.getAbsolutePath(), true);
      bc.setMetadata(metadata);

      bc.setAccessTier(AccessTier.ARCHIVE);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException e) {
      throw new IOException(e);
    } catch (SdkClientException e) {
      throw new IOException(e);
    }
  }

  public static void moveToColdLine(Storage storage, String bucket,
                                    String objectKey,
                                    File file,
                                    Map<String, String> metadata) throws IOException {
    try {
      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Move to cold line. Bucket: " + bucket + " Object Key: " + objectKey);
      }
      long startTime = System.currentTimeMillis();
      BlobId blobId = BlobId.of(bucket, objectKey);

      BlobInfo blobInfo = BlobInfo.newBuilder(blobId).
              setStorageClass(com.lc.repackaged.com.google.cloud.storage.StorageClass.COLDLINE).
              setMetadata(metadata).build();
      storage.create(blobInfo, Files.readAllBytes(Paths.get(file.getAbsolutePath())));

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException e) {
      throw new IOException(e);
    } catch (SdkClientException e) {
      throw new IOException(e);
    }
  }
}
