/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.protocol;

import com.google.common.base.Joiner;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.hdfs.protocol.RemovedBlock;

import java.util.ArrayList;
import java.util.Collection;

/**
 * BlockDeletionCommand is an instruction to a data-node to
 * delete the specified blocks.
 * Using this you can delete blocks stored on the datanode
 * and in the cloud.
 */
@InterfaceAudience.Private
@InterfaceStability.Evolving
public class BlockDeletionCommand extends DatanodeCommand {
  RemovedBlock[] deletedBlocks;
  String blockPoolId;

  public BlockDeletionCommand(String poolId, RemovedBlock[] blocks) {
    super(DatanodeProtocol.DNA_INVALIDATE);
    this.deletedBlocks = blocks;
    this.blockPoolId = poolId;
  }

  public String getBlockPoolId() {
    return blockPoolId;
  }

  /**
   * Return the list of deleted blocks.
   */
  public RemovedBlock[] getDeletedBlocks() {
    return deletedBlocks;
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder();
    sb.append("BlockDeletionCommand(\n  ");
    Joiner.on("\n  ").appendTo(sb, deletedBlocks);
    sb.append("\n)");
    return sb.toString();
  }
}
