/*
 * Copyright (C) 2020 Logical Clocks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.SdkClientException;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.client.builder.ExecutorFactory;
import com.amazonaws.regions.Regions;
import com.amazonaws.retry.PredefinedRetryPolicies;
import com.amazonaws.retry.RetryPolicy;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.*;
import com.amazonaws.services.s3.transfer.Download;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Sets;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;

import java.io.*;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

public class CloudPersistenceProviderS3Impl implements CloudPersistenceProvider {

  @VisibleForTesting
  public static final Log LOG = LogFactory.getLog(CloudPersistenceProviderS3Impl.class);

  private final Configuration conf;
  private final AmazonS3 s3Client;
  private Regions region;
  private final int prefixSize;
  private TransferManager transfers;
  private final int bucketDeletionThreads;
  private long partSize;
  private int maxThreads;
  private long multiPartThreshold;
  private final boolean sseEnabled;
  private final boolean versioningEnabled;
  private final boolean sseBucketKeyEnable;
  private final String sseType;
  private final String sseKeyARN;
  private String endPoint;
  private String signingRegion;
  private final Boolean bucketOwnerFullControll;
  private final boolean bypassGovernanceRetention;

  public CloudPersistenceProviderS3Impl(Configuration conf) {
    this.conf = conf;

    // set only endpoint or the region param
    endPoint = conf.get(DFSConfigKeys.DFS_CLOUD_AWS_ENDPOINT_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_ENDPOINT_DEFAULT);
    signingRegion = conf.get(DFSConfigKeys.DFS_CLOUD_AWS_SIGNING_REGION_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_SIGNING_REGION_DEFAULT);

    region = null;
    if (endPoint.compareToIgnoreCase("") == 0) {
      this.region = Regions.fromName(conf.get(DFSConfigKeys.DFS_CLOUD_AWS_S3_REGION,
        DFSConfigKeys.DFS_CLOUD_AWS_S3_REGION_DEFAULT));
    }

    this.bucketDeletionThreads =
            conf.getInt(DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_KEY,
                    DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_DEFAULT);
    this.prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    maxThreads = conf.getInt(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS,
            DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS_DEFAULT);
    if (maxThreads < 2) {
      LOG.warn(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS +
              " must be at least 2: forcing to 2.");
      maxThreads = 2;
    }

    // versioning
    versioningEnabled = conf.getBoolean(DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_KEY,
            DFSConfigKeys.S3_BUCKET_ENABLE_VERSIONING_DEFAULT);

    // SSE
    sseEnabled = conf.getBoolean(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_ENABLE_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_ENABLE_DEFAULT);
    sseBucketKeyEnable = conf.getBoolean(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_BUCKET_KEY_ENABLE_KEY,
        DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_BUCKET_KEY_ENABLE_DEFAULT);
    sseKeyARN = conf.get(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_KEY_ARN_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_KEY_ARN_DEFAULT);
    sseType = conf.get(DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_TYPE_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_SERVER_SIDE_ENCRYPTION_TYPE_DEFAULT);
    if (sseType.compareToIgnoreCase(CloudS3Encryption.SSE_KMS.toString()) != 0
      && sseType.compareToIgnoreCase(CloudS3Encryption.SSE_S3.toString()) != 0) {
      throw new IllegalArgumentException("Invalid Amazon S3 Encryption type");
    }

    bucketOwnerFullControll = conf.getBoolean(DFSConfigKeys.DFS_CLOUD_AWS_AMZ_ACL_BUCKET_OWNER_FULL_CONTROL_ENABLE_KEY,
      DFSConfigKeys.DFS_CLOUD_AWS_AMZ_ACL_BUCKET_OWNER_FULL_CONTROL_ENABLE_DEFAULT);
    bypassGovernanceRetention = conf.getBoolean(DFSConfigKeys.S3_BYPASS_GOVERNANCE_RETENTION,
        DFSConfigKeys.S3_BYPASS_GOVERNANCE_RETENTION_DEFAULT);
    this.s3Client = connect();
    initTransferManager();
  }
  private AmazonS3 connect() {
    LOG.info("HopsFS-Cloud. Connecting to S3. Region " + region);
    ClientConfiguration s3conf = new ClientConfiguration();
    int retryCount = conf.getInt(DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_KEY,
            DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_DEFAULT);
    RetryPolicy retryPolicy = new RetryPolicy(
            PredefinedRetryPolicies.DEFAULT_RETRY_CONDITION,  // Retry on standard conditions
            PredefinedRetryPolicies.DEFAULT_BACKOFF_STRATEGY,  // Exponential backoff
            retryCount,  // Maximum retry count
            true  // Whether to retry on request throttling errors (e.g., 503)
    );
    s3conf.setRetryPolicy(retryPolicy);
    s3conf.setMaxErrorRetry(retryCount);
    s3conf.setMaxConnections(maxThreads);
    LOG.info("Max retry " + s3conf.getMaxErrorRetry());
    AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard()
            .withClientConfiguration(s3conf);

    if (region != null) {
      LOG.info("HopsFS-Cloud. Using AWS region: "+region.toString());
      builder.withRegion(region);
    } else {
      LOG.info("HopsFS-Cloud. Using AWS endpoint: "+endPoint+". Signing Region: "+signingRegion);
      AwsClientBuilder.EndpointConfiguration epc =
        new AwsClientBuilder.EndpointConfiguration(endPoint, signingRegion);
      builder.withEndpointConfiguration(epc);
      builder.setPathStyleAccessEnabled(true);
    }

    return builder.build();
  }

  public void initTransferManager() {
    partSize = conf.getLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE,
            DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE_DEFAULT);

    if (partSize < 5 * 1024 * 1024) {
      LOG.error(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE + " must be at least 5 MB");
      partSize = 5 * 1024 * 1024;
    }

    multiPartThreshold = conf.getLong(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD,
            DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD_DEFAULT);
    if (multiPartThreshold < 5 * 1024 * 1024) {
      LOG.error(DFSConfigKeys.DFS_CLOUD_MIN_MULTIPART_THRESHOLD + " must be at least 5 MB");
      multiPartThreshold = 5 * 1024 * 1024;
    }

    transfers =
            TransferManagerBuilder.standard().withS3Client(s3Client).
                    withExecutorFactory(new ExecutorFactory() {
                      @Override
                      public ExecutorService newExecutor() {
                        return Executors.newFixedThreadPool(maxThreads);
                      }
                    }).
                    withMultipartUploadThreshold(multiPartThreshold).
                    withMinimumUploadPartSize(partSize).
                    withMultipartCopyThreshold(multiPartThreshold).
                    withMultipartCopyPartSize(partSize).build();
  }

  private void createS3Bucket(String bucketName) {
    if (!s3Client.doesBucketExistV2(bucketName)) {
      s3Client.createBucket(bucketName);
      // Verify that the bucket was created by retrieving it and checking its location.
      String bucketLocation = s3Client.getBucketLocation(new GetBucketLocationRequest(bucketName));
      LOG.info("HopsFS-Cloud. New bucket created. Name: " +
              bucketName + " Location: " + bucketLocation);
    } else {
      LOG.info("HopsFS-Cloud. Bucket already exists. Bucket Name: " + bucketName);
    }
  }

  /*
  deletes all the bucket belonging to this user.
  This is only used for testing.
   */
  public void deleteAllBuckets(String prefix) throws IOException, ExecutionException, InterruptedException {
    ExecutorService tPool = Executors.newFixedThreadPool(bucketDeletionThreads);
    try {
      List<Bucket> buckets = s3Client.listBuckets();
      LOG.info("HopsFS-Cloud. Deleting all of the buckets with prefix \""+prefix+
              "\" for this user. Number of deletion threads " + bucketDeletionThreads);
      for (Bucket b : buckets) {
        if (b.getName().startsWith(prefix.toLowerCase())) {
          emptyBucket(b.getName(), false, tPool);
          deleteBucket(b.getName());
        }
      }
    } finally {
      tPool.shutdown();
    }
  }

  @Override
  public boolean existsCID(String bucket) throws IOException {
    if (!s3Client.doesBucketExistV2(bucket)) {
      return false;
    }

    return objectExists(bucket, CloudHelper.CID_FILE);
  }

  @Override
  public void setCID(String bucket, String cid) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      if (!s3Client.doesBucketExistV2(bucket)) {
        throw new IOException("Bucket " + bucket + " does not exist");
      }
      ObjectMetadata objMetadata = new ObjectMetadata();
      objMetadata.setContentType("plain/text");

      InputStream cidStream = new ByteArrayInputStream(cid.getBytes());
      PutObjectRequest putReq = new PutObjectRequest(bucket,
              CloudHelper.CID_FILE, cidStream, objMetadata);
      setUploadHeaders(putReq, objMetadata); // set encryption params
      putReq.setMetadata(objMetadata);
      Upload upload = transfers.upload(putReq);
      upload.waitForUploadResult();

    } catch (InterruptedException e) {
      throw new InterruptedIOException(e.toString());
    } catch (AmazonServiceException e) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in setCID, CID: " +
              cid + " Error: " + e.getMessage());
      throw new IOException(e);
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException in setCID, CID: " +
              cid + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  set CID. Bucket: " + bucket
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
  }

  @Override
  public String getCID(String bucket) throws IOException {
    long startTime = System.currentTimeMillis();
    String cid = null;

    try {
      if (!s3Client.doesBucketExistV2(bucket)) {
        throw new IOException("Bucket " + bucket + " does not exist");
      }

      S3Object object = s3Client.getObject(new GetObjectRequest(bucket, CloudHelper.CID_FILE));
      InputStream objectData = object.getObjectContent();
      cid = new BufferedReader(new InputStreamReader(objectData))
              .lines().collect(Collectors.joining("\n"));
      objectData.close();
    } catch (AmazonServiceException e) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in getCID. Bucket: " +
              bucket + " Error: " + e.getMessage());
      throw new IOException(e);
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException in getCID. Bucket: " +
              bucket + " Error: " + e.getMessage());
      throw new IOException(e);
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  get CID. "
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return cid;
  }

  @Override
  public boolean isEmpty(String bucket) throws IOException {
    if (!s3Client.doesBucketExistV2(bucket)) {
      throw new IOException("Bucket "+bucket+" does not exist");
    }

    ListObjectsV2Request req = new ListObjectsV2Request().withBucketName(bucket);
    ListObjectsV2Result listResult = s3Client.listObjectsV2(req);
    if (listResult.getObjectSummaries().size() > 0) {
      return false;
    }

    if(versioningEnabled) {
      VersionListing versionList = s3Client.listVersions(
              new ListVersionsRequest().withBucketName(bucket));
      if (versionList.getVersionSummaries().size() > 0) {
        return false;
      }
    }
    return true;
  }

  @Override
  public boolean bucketExists(String bucket) throws IOException {
    return s3Client.doesBucketExistV2(bucket);
  }

  /*
  Deletes all the buckets that are used by HopsFS
   */
  @Override
  public void format(List<String> buckets) {
    ExecutorService tPool = Executors.newFixedThreadPool(bucketDeletionThreads);
    try {
      String msg = "HopsFS-Cloud. Deleting all of the buckets used by HopsFS. Number of " +
              "deletion " + "threads " + bucketDeletionThreads;
      LOG.info(msg);
      System.out.println(msg);
      for (String bucket : buckets) {
        // check that s3 bucket supports versioning if needed
        if (versioningEnabled && !isVersioningSupported(bucket)) {
          throw new IOException("Cannot format file system. Versioning is enabled. However the bucket does " +
                  "not support versioning");
        }
        emptyBucket(bucket, true, tPool);
      }
    } catch (IOException | InterruptedException | ExecutionException e) {
      LOG.warn(e);
      throw new RuntimeException(e);
    } finally {
      tPool.shutdown();
    }
  }

  @Override
  public void createBucket(String bucket) {
    createS3Bucket(bucket);
    enableVersioning(bucket);
    enableBucketEncryption(bucket);
  }

  public void deleteBucket(String bucket) {
    s3Client.deleteBucket(bucket);
  }

  @Override
  public void checkAllBuckets(List<String> buckets) throws IOException {

    final int retry = 300;  // keep trying until the newly created bucket is available
    for (String bucket : buckets) {
      boolean exists = false;
      for (int j = 0; j < retry; j++) {
        if (!s3Client.doesBucketExistV2(bucket)) {
          //wait for a sec and retry
          try {
            Thread.sleep(1000);
          } catch (InterruptedException e) {
          }
        } else {
          exists = true;
          break;
        }
      }

      if (!exists) {
        throw new IllegalStateException("S3 Bucket " + bucket + " needed for the file system " +
                "does not exists");
      } else {
        //check the bucket is writable
        UUID uuid = UUID.randomUUID();
        try {

         // create a temp file to upload
          File objFile = new File("/tmp/" + UUID.randomUUID());
          FileWriter fw = new FileWriter(objFile);
          fw.write("test string");
          fw.close();

          HashMap<String, String> metadata = new HashMap<>();
          uploadObject(bucket, uuid.toString(), objFile, metadata);
          objectExists(bucket, uuid.toString());
          deleteObject(bucket, uuid.toString());
          objFile.delete();
          LOG.info("HopsFS-Cloud. Checked bucket: "+bucket);
        } catch (Exception e) {
          throw new IllegalStateException("Write test for S3 bucket: " + bucket + " failed. " + e);
        }
      }
    }
  }

  private void emptyBucket(final String bucketName,
                           boolean onlyDeleteHopsFSData, ExecutorService tPool)
          throws ExecutionException, InterruptedException, IOException {
    final AtomicInteger deletedBlocks = new AtomicInteger(0);
    try {
      if (!s3Client.doesBucketExistV2(bucketName)) {
        throw new IOException("Cannot format file system. The bucket does not exist");
      }

      System.out.println("HopsFS-Cloud. Deleting bucket: " + bucketName);

      ListObjectsV2Request req = new ListObjectsV2Request().withBucketName(bucketName);
      ListObjectsV2Result listResult;
      do {
        listResult = s3Client.listObjectsV2(req);

        final List<Future> futures = new ArrayList<>();
        for (S3ObjectSummary s3Object : listResult.getObjectSummaries()) {
          final String objectkey = s3Object.getKey();

          if (onlyDeleteHopsFSData && !objectkey.startsWith(CloudHelper.PREFIX_STR)) {
            continue;
          }

          Callable task = new Callable<Object>() {
            @Override
            public Object call() throws Exception {
              s3Client.deleteObject(bucketName, objectkey);
              String msg = "\rDeleted Blocks: " + (deletedBlocks.incrementAndGet());
              System.out.print(msg);
              return null;
            }
          };
          futures.add(tPool.submit(task));
        }

        for (Future future : futures) {
          future.get();
        }

        req.setContinuationToken(listResult.getNextContinuationToken());
      } while (listResult.isTruncated());

      // Delete all object versions (required for versioned buckets).
      VersionListing versionList = s3Client.listVersions(
              new ListVersionsRequest().withBucketName(bucketName));
      while (true) {
        Iterator<S3VersionSummary> versionIter = versionList.getVersionSummaries().iterator();
        List<Future> futures = new ArrayList<>();
        while (versionIter.hasNext()) {
          S3VersionSummary vs = versionIter.next();

          if (onlyDeleteHopsFSData && !vs.getKey().startsWith(CloudHelper.PREFIX_STR)) {
            continue;
          }

          Callable task = new Callable<Object>() {
            @Override
            public Object call() throws Exception {

              s3Client.deleteVersion(createDeleteVersionRequest(bucketName, vs));
              String msg = "\rDeleted Versioned Blocks: " + (deletedBlocks.incrementAndGet());
              System.out.print(msg);
              return null;
            }
          };
          futures.add(tPool.submit(task));
        }

        for(Future future : futures){
          future.get();
        }

        if (versionList.isTruncated()) {
          versionList = s3Client.listNextBatchOfVersions(versionList);
        } else {
          break;
        }
      }

      // delete CID file
      if (objectExists(bucketName, CloudHelper.CID_FILE)) {
        s3Client.deleteObject(bucketName, CloudHelper.CID_FILE);
      }

      System.out.println("");
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in emptyAndDeleteS3Bucket. Bucket: " +
              bucketName + "Error: " + up.getMessage());
      // The call was transmitted successfully, but Amazon S3 couldn't process
      // it, so it returned an error response.
      up.printStackTrace();
      throw up;
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in emptyAndDeleteS3Bucket. Bucket: " +
              bucketName + "Error: " + up.getMessage());
      // Amazon S3 couldn't be contacted for a response, or the client couldn't
      // parse the response from Amazon S3.
      up.printStackTrace();
      throw up;
    }
  }

  private DeleteVersionRequest createDeleteVersionRequest(String bucket, S3VersionSummary versionSummary) {
    DeleteVersionRequest dvr = new DeleteVersionRequest(bucket, versionSummary.getKey(), versionSummary.getVersionId());
    if (bypassGovernanceRetention) {
      dvr.setBypassGovernanceRetention(true);
    }
    return dvr;
  }

  @Override
  public void uploadObject(String bucket, String objectKey, File object,
                           Map<String, String> metadata) throws IOException {
    try {
      LOG.info("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey +" " +
              " Object Size: " +objectKey.length());

      long startTime = System.currentTimeMillis();
      PutObjectRequest putReq = new PutObjectRequest(bucket,
              objectKey, object);

      // Upload a file as a new object with ContentType and title specified.
      ObjectMetadata objMetadata = new ObjectMetadata();
      objMetadata.setContentType("plain/text");
      objMetadata.setUserMetadata(metadata);

      // set encryption params
      setUploadHeaders(putReq, objMetadata);
      putReq.setMetadata(objMetadata);

      Upload upload = transfers.upload(putReq);

      upload.waitForUploadResult();

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (InterruptedException e) {
      throw new InterruptedIOException(e.toString());
    } catch (AmazonServiceException e) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in uploadObject. Bucket: " +
              bucket + " Key: " + objectKey + " File: " +
              object.getAbsolutePath() + " Error: " + e.getMessage());
      throw new IOException(e);
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException in uploadObject. Bucket: " +
              bucket + " Key: " + objectKey + " File: " +
              object.getAbsolutePath() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public int getPrefixSize() {
    return prefixSize;
  }

  @Override
  public boolean objectExists(String bucket, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      boolean exists = s3Client.doesObjectExist(bucket, objectKey);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Object Exists?. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return exists;
    } catch (AmazonServiceException e) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in objectExists. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e); // throwing runtime exception will kill DN
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException : in objectExists. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private ObjectMetadata getS3ObjectMetadata(String bucket, String objectKey)
          throws IOException {
    try {
      GetObjectMetadataRequest req = new GetObjectMetadataRequest(bucket,
              objectKey);
      ObjectMetadata s3metadata = s3Client.getObjectMetadata(req);
      return s3metadata;
    } catch (AmazonServiceException e) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in getS3ObjectMetadata. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e); // throwing runtime exception will kill DN
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException in getS3ObjectMetadata. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }


  @Override
  public Map<String, String> getUserMetaData(String bucket, String objectKey)
          throws IOException {
    long startTime = System.currentTimeMillis();
    ObjectMetadata s3metadata = getS3ObjectMetadata(bucket, objectKey);
    Map<String, String> metadata = s3metadata.getUserMetadata();

    if(LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get Object Metadata. Bucket: " + bucket + " Object Key: " + objectKey
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return metadata;
  }

  @Override
  public long getObjectSize(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    ObjectMetadata s3metadata = getS3ObjectMetadata(bucket, objectKey);
    long size = s3metadata.getContentLength();

    if(LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get Object Size. Bucket: " + bucket + " Object Key: " + objectKey
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return size;
  }

  @Override
  public void downloadObject(String bucket, String objectKey, File path) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      if (path.exists()) {
        path.delete();
      } else {
        //make sure that all parent dirs exists
        if (!path.getParentFile().exists()) {
          path.getParentFile().mkdirs();
        }
      }

      Random rand = new Random(System.currentTimeMillis());
      File tmpFile = new File(path.getAbsolutePath()+"."+rand.nextLong()+".downloading");
      if(tmpFile.exists()){
        tmpFile.delete();
      }

      Download down = transfers.download(bucket, objectKey, tmpFile);
      down.waitForCompletion();

      tmpFile.renameTo(path);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Download Object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Download Path: " + path
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException e) {
      if (e instanceof AmazonS3Exception) {
        if (e.getMessage().contains("The operation is not valid for the object's storage class")) {
          String sc = ((AmazonS3Exception) e).getAdditionalDetails().get("StorageClass");
          String message = "Unable to read block " + objectKey + ".";
          if (sc != null) {
            message += " The block has moved to storage: " + sc + ". Please restore the block to " +
                    "read the file";
          }
          throw new BlockMovedToColdStorageException(message);
        }
        LOG.info("HopsFS-Cloud: AmazonServiceException in downloadObject. Bucket: " +
                bucket + " ObjKey: " + objectKey + " File: " + path.getAbsolutePath() +
                " Error: " + e.getMessage());
      }
      throw new IOException(e); // throwing runtime exception will kill DN
    } catch (SdkClientException e) {
      LOG.info("HopsFS-Cloud: SdkClientException in downloadObject Bucket: " +
              bucket + " ObjKey: " + objectKey + " File: " + path.getAbsolutePath() +
              " Error: " + e.getMessage());
      throw new IOException(e);
    } catch (InterruptedException e) {
      throw new InterruptedIOException(e.toString());
    }
  }

  @VisibleForTesting
  @Override
  public Map<BlockIDAndGSTuple, CloudBlock> getAll(String prefix, List<String> buckets) throws IOException {
    long startTime = System.currentTimeMillis();
    Map<BlockIDAndGSTuple, CloudBlock> blocks = new HashMap<>();
    for (String bucket : buckets) {
      listBucket(bucket, prefix, blocks);
    }

    if(LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get all blocks. Buckets: " + Arrays.toString(buckets.toArray()) +
              " Prefix: " + prefix +
              " Total Blocks: " + blocks.size() +
              " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return blocks;
  }

  @Override
  public List<String> getAllHopsFSDirectories(List<String> buckets) throws IOException {
    List<String> dirs = new ArrayList<>();
    for (String bucket : buckets) {
      ListObjectsV2Request req =
              new ListObjectsV2Request().withBucketName(bucket).withDelimiter("/")
                      .withPrefix(CloudHelper.PREFIX_STR);

      ListObjectsV2Result result;
      do {
        result = s3Client.listObjectsV2(req);
        for(String dir: result.getCommonPrefixes()) {
          if (dir.contains(CloudHelper.PREFIX_STR)){
            dirs.add(dir);
          } else {
            LOG.info("HopsFS-Cloud. Ignoring "+dir+" directory. It is not HopsFS directory");
          }
        }
        String token = result.getNextContinuationToken();
        req.setContinuationToken(token);
      } while (result.isTruncated());
    }

    return dirs;
  }

  @Override
  public void deleteObject(String bucket, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      s3Client.deleteObject(bucket, objectKey);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Delete object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in deleteObject. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in deleteObject. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public void shutdown() {
    s3Client.shutdown();
    if (transfers != null) {
      transfers.shutdownNow(true);
      transfers = null;
    }
  }

  private void listBucket(String bucketName, String prefix, Map<BlockIDAndGSTuple, CloudBlock> result)
          throws IOException {
    Map<BlockIDAndGSTuple, CloudObject> blockObjs = new HashMap<>();
    Map<BlockIDAndGSTuple, CloudObject> metaObjs = new HashMap<>();

    try {
      if (!s3Client.doesBucketExistV2(bucketName)) {
        return;
      }

      assert prefix != null;

      ListObjectsV2Request req = new ListObjectsV2Request().withBucketName(bucketName).withPrefix(prefix);
      ListObjectsV2Result listResult;
      do {
        listResult = s3Client.listObjectsV2(req);

        for (S3ObjectSummary s3Object : listResult.getObjectSummaries()) {
          String key = s3Object.getKey();

          CloudObject co = new CloudObject();
          co.setBucket(s3Object.getBucketName());
          co.setKey(s3Object.getKey());
          co.setSize(s3Object.getSize());
          co.setLastModifiedTime(s3Object.getLastModified().getTime());

          BlockIDAndGSTuple idAndGS = CloudHelper.getIDAndGSFromKey(key);
          if (idAndGS != null && CloudHelper.isBlockFilename(key)) {
            blockObjs.put(idAndGS, co);
          } else if (idAndGS != null || CloudHelper.isMetaFilename(key)) {
            metaObjs.put(idAndGS, co);
          } else {
            LOG.warn("HopsFS-Cloud. File system objects are tampered. The " + key + " is not HopsFS object.");
          }
        }
        req.setContinuationToken(listResult.getNextContinuationToken());
      } while (listResult.isTruncated());
    } catch (AmazonServiceException up) {
      LOG.error("HopsFS-Cloud. Unable to list bucket: " + bucketName + " prefix: \"" + prefix + "\"" +
              " AmazonServiceException " + up);
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.error("HopsFS-Cloud. Unable to list bucket: " + bucketName + " prefix: \"" + prefix + "\"" +
              " SdkClientException " + up);
      throw new IOException(up);
    }

    mergeMetaAndBlockObjects(metaObjs, blockObjs, result);
  }

  public static void mergeMetaAndBlockObjects(Map<BlockIDAndGSTuple, CloudObject> metaObjs,
                                        Map<BlockIDAndGSTuple, CloudObject> blockObjs,
                                        Map<BlockIDAndGSTuple, CloudBlock> res) {

    Set blockKeySet = blockObjs.keySet();
    Set metaKeySet = metaObjs.keySet();
    Sets.SetView<BlockIDAndGSTuple> symDiff = Sets.symmetricDifference(blockKeySet, metaKeySet);
    Sets.SetView<BlockIDAndGSTuple> intersection = Sets.intersection(blockKeySet, metaKeySet);

    for (BlockIDAndGSTuple cloudBlockObjectKey : intersection) {
      CloudObject blockObj = blockObjs.get(cloudBlockObjectKey);
      CloudObject metaObj = metaObjs.get(cloudBlockObjectKey);

      long blockSize = blockObj.getSize();
      long genStamp = cloudBlockObjectKey.getGs();

      Block block = new Block(cloudBlockObjectKey.getBlockID(), blockSize, genStamp,
              blockObj.getBucketName());

      CloudBlock cb = new CloudBlock(block, blockObj.getLastModified());
      res.put(cloudBlockObjectKey, cb);
    }

    for (BlockIDAndGSTuple cloudObjectKey : symDiff) {
      String keyFound = "";
      String bucket = "";
      CloudBlock cb = new CloudBlock();

      CloudObject blockObj = blockObjs.get(cloudObjectKey);
      CloudObject metaObj = metaObjs.get(cloudObjectKey);

      if (blockObj != null) {
        cb.setBlockObjectFound(true);
        cb.setLastModified(blockObj.getLastModified());
        bucket = blockObj.getBucketName();
      } else if (metaObj != null) {
        cb.setMetaObjectFound(true);
        cb.setLastModified(metaObj.getLastModified());
        bucket = metaObj.getBucketName();
      }

      Block block = new Block();
      block.setBlockIdNoPersistance(cloudObjectKey.getBlockID());
      block.setGenerationStampNoPersistance(cloudObjectKey.getGs());
      block.setCloudBucketNoPersistance(bucket);
      cb.setBlock(block);
      res.put(cloudObjectKey, cb);
    }
  }

  @VisibleForTesting
  @Override
  public void renameObject(String srcBucket, String dstBucket, String srcKey,
                           String dstKey) throws IOException {
    try {
      copyObject(srcBucket, dstBucket, srcKey , dstKey, null);
      long startTime = System.currentTimeMillis();
      deleteObject(srcBucket, srcKey);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Deleting after rename object. Src Bucket: " + srcBucket +
                " Dst Bucket: " + dstBucket +
                " Src Object Key: " + srcKey +
                " Dst Object Key: " + dstKey +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      //delete the src
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in renameObject  Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in renameObject  Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public void copyObject(String srcBucket, String dstBucket, String srcKey,
                           String dstKey, Map<String, String> newObjMetadata) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      CopyObjectRequest req = new CopyObjectRequest(srcBucket, srcKey,
              dstBucket, dstKey);

      ObjectMetadata objectMetadata = new ObjectMetadata();
      setUploadHeaders(req, objectMetadata);

      if(newObjMetadata != null){
        objectMetadata.setUserMetadata(newObjMetadata);
      }

      req.setNewObjectMetadata(objectMetadata);

      CopyObjectResult res = s3Client.copyObject(req);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Copy object. Src Bucket: " + srcBucket +
                " Dst Bucket: " + dstBucket +
                " Src Object Key: " + srcKey +
                " Dst Object Key: " + dstKey +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in copyObject. Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in copyObject. Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public long getPartSize() {
    return partSize;
  }

  @Override
  public int getXferThreads(){
    return maxThreads;
  }

  @Override
  public  UploadID startMultipartUpload(String bucket, String objectKey,
                                          Map<String, String> metadata) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(bucket,
              objectKey);

      ObjectMetadata objMetadata = new ObjectMetadata();
      objMetadata.setContentType("plain/text");
      objMetadata.setUserMetadata(metadata);

      setUploadHeaders(initRequest, objMetadata);
      initRequest.setObjectMetadata(objMetadata);
      InitiateMultipartUploadResult initResponse = s3Client.initiateMultipartUpload(initRequest);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Start multipart upload. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return new S3UploadID(initResponse.getUploadId());
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in startMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in startMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public PartRef uploadPart(String bucket, String objectKey, UploadID uploadID, int partNo,
                             File file, long startPos, long endPos) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      UploadPartRequest uploadRequest = new UploadPartRequest()
              .withBucketName(bucket)
              .withKey(objectKey)
              .withUploadId(((S3UploadID)uploadID).getS3MultipartID())
              .withPartNumber(partNo)
              .withFileOffset(startPos)
              .withFile(file)
              .withPartSize(endPos - startPos);

      UploadPartResult uploadResult = s3Client.uploadPart(uploadRequest);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Upload part. Bucket: " + bucket + " Object Key: " + objectKey + " " +
                "PartNo: " + partNo + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return new S3PartRef(uploadResult.getPartETag());
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in uploadPart. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " + uploadID.toString() +
              " Part No:" + partNo + " File: " + file.getAbsolutePath() +
              " Start Pos: " + startPos + " End Pos: " + endPos + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in uploadPart. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " + uploadID.toString() +
              " Part No:" + partNo + " File: " + file.getAbsolutePath() +
              " Start Pos: " + startPos + " End Pos: " + endPos + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public void finalizeMultipartUpload(String bucket, String objectKey, UploadID uploadID,
                                      List<PartRef> refs) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      List<PartETag> partETags = new ArrayList();
      for (PartRef ref : refs) {
        partETags.add(((S3PartRef) ref).getPartETag());
      }

      CompleteMultipartUploadRequest compRequest = new CompleteMultipartUploadRequest(
              bucket, objectKey, ((S3UploadID)uploadID).getS3MultipartID(),
              partETags);
      s3Client.completeMultipartUpload(compRequest);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Finalize multipart upload. Bucket: " + bucket +
                " Object Key: " + objectKey + " " + "Total Parts: " + partETags.size() +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in finalizeMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in finalizeMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public void abortMultipartUpload(String bucket, String objectKey, UploadID uploadID)
          throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      AbortMultipartUploadRequest req = new AbortMultipartUploadRequest(bucket, objectKey,
              ((S3UploadID)uploadID).getS3MultipartID());
      s3Client.abortMultipartUpload(req);

      if(LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Aborted multipart upload. Bucket: " + bucket +
                " Object Key: " + objectKey +
                " UploadID: "+((S3UploadID)uploadID).getS3MultipartID()+
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in abortMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in abortMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public List<ActiveMultipartUploads> listMultipartUploads(List<String> buckets, String prefix) throws IOException {
    List<ActiveMultipartUploads> uploads = new ArrayList();
    long startTime = System.currentTimeMillis();
    for (String bucket : buckets) {
      uploads.addAll(listMultipartUploadsForBucket(bucket, prefix));
    }

    if(LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. List multipart. Active Uploads " + uploads.size() +
              " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return uploads;
  }

  @Override
  public boolean restoreDeletedBlock(String bucket, String objectKey) throws IOException {
    ListVersionsRequest req = new ListVersionsRequest();
    req.setPrefix(objectKey);
    req.setBucketName(bucket);

    try {
      VersionListing versions = s3Client.listVersions(req);
      Iterator<S3VersionSummary> verItr = versions.getVersionSummaries().iterator();
      //remove the versions that have ETag set to null. These are delete markers
      //Stop when non delete version is surfaced
      while (verItr.hasNext()) {
        S3VersionSummary versionSummary = verItr.next();

        if (versionSummary.isDeleteMarker()) {
          if (LOG.isDebugEnabled()) {
            LOG.debug("HopsFS-Cloud. Recovering Block ID: " + versionSummary.getKey() + "  " +
                    "Deleting Version: " + versionSummary.getVersionId());
          }
          s3Client.deleteVersion(createDeleteVersionRequest(bucket, versionSummary));
        } else {
          // non delete version is surfaced
          return true;
        }
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in restoreDeletedBlock. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in restoreDeletedBlock. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }

    // unable to surface a non deleted version of the block
    return false;
  }

  @Override
  public boolean isVersioningSupported(String bucket) throws IOException {
    if (!versioningEnabled) {
      return false;
    }
    // check that the bucket is version enabled
    try {
      BucketVersioningConfiguration conf = s3Client.getBucketVersioningConfiguration(bucket);
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Versioning Status: " + conf.getStatus());
      }
      if (conf.getStatus().compareTo(BucketVersioningConfiguration.ENABLED) == 0) {
        LOG.debug("HopsFS-Cloud. Versioning is enabled");
        return true;
      } else {
        return false;
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in isVersioningSupported. Bucket: " +
              bucket + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in isVersioningSupported. Bucket: " +
              bucket + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  public List<S3VersionSummary> listAllVersions(String bucket, String objectKey) throws IOException {
    List<S3VersionSummary> versions = new ArrayList<S3VersionSummary>();
    ListVersionsRequest listVersionsRequest = new ListVersionsRequest();
    listVersionsRequest.setBucketName(bucket);
    listVersionsRequest.setPrefix(objectKey);
    VersionListing versionList = s3Client.listVersions(listVersionsRequest);
    while (true) {
      Iterator<S3VersionSummary> versionIter = versionList.getVersionSummaries().iterator();
      List<Future> futures = new ArrayList<>();
      while (versionIter.hasNext()) {
        S3VersionSummary vs = versionIter.next();
        versions.add(vs);
      }

      if (versionList.isTruncated()) {
        versionList = s3Client.listNextBatchOfVersions(versionList);
      } else {
        break;
      }
    }
    return versions;
  }

  @Override
  public void deleteAllVersions(String bucket, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      List<S3VersionSummary> versions = listAllVersions(bucket, objectKey);
      for (S3VersionSummary version : versions) {
        s3Client.deleteVersion(createDeleteVersionRequest(bucket, version));
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Deleted version " + version.getVersionId() + " of " +
                  "Object: " + objectKey);
        }
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Deleted all versions " + " of " + "Object: " + objectKey +
                " Time: " + (System.currentTimeMillis() - startTime) + " ms");
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in deleteAllVersions. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in deleteAllVersions. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  @Override
  public void deleteOldVersions(String bucket, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      List<S3VersionSummary> versions = listAllVersions(bucket, objectKey);
      for (S3VersionSummary version : versions) {
        if( !version.isLatest()) {
          s3Client.deleteVersion(createDeleteVersionRequest(bucket, version));
          if (LOG.isDebugEnabled()) {
            LOG.debug("HopsFS-Cloud. Deleted version " + version.getVersionId() + " of " +
                    "Object: " + objectKey);
          }
        }
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Deleted all old versions " + " of " + "Object: " + objectKey +
                " Time: " + (System.currentTimeMillis() - startTime) + " ms");
      }
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in deleteOldVersions. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in deleteOldVersions. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + up.getMessage());
      throw new IOException(up);
    }
  }

  private List<ActiveMultipartUploads> listMultipartUploadsForBucket(String bucket, String prefix)
          throws IOException {
    List<ActiveMultipartUploads> uploads = new ArrayList();
    try {
      ListMultipartUploadsRequest req = new ListMultipartUploadsRequest(bucket);
      req.setPrefix(prefix);
      MultipartUploadListing uploadListing =
              s3Client.listMultipartUploads(req);
      do {
        for (MultipartUpload upload : uploadListing.getMultipartUploads()) {
          uploads.add(new S3ActiveMultipartUploads(bucket, upload.getKey(),
                  upload.getInitiated().getTime(),
                  new S3UploadID(upload.getUploadId())));
        }
        ListMultipartUploadsRequest request = new ListMultipartUploadsRequest(bucket)
                .withUploadIdMarker(uploadListing.getNextUploadIdMarker())
                .withKeyMarker(uploadListing.getNextKeyMarker());
        uploadListing = s3Client.listMultipartUploads(request);
      } while (uploadListing.isTruncated());
    } catch (AmazonServiceException up) {
      LOG.info("HopsFS-Cloud: AmazonServiceException in listMultipartUploadsForBucket. Bucket: " +
              bucket + " Prefix: " + prefix + " Error: " + up.getMessage());
      throw new IOException(up);
    } catch (SdkClientException up) {
      LOG.info("HopsFS-Cloud: SdkClientException in listMultipartUploadsForBucket. Bucket: " +
              bucket + " Prefix: " + prefix + " Error: " + up.getMessage());
      throw new IOException(up);
    }
    return uploads;
  }

  public void enableVersioning(String bucket) {
    if (versioningEnabled) {
      BucketVersioningConfiguration configuration =
              new BucketVersioningConfiguration().withStatus("Enabled");
      LOG.info("HopsFS-Cloud. Enabling Versioning for the bucket: " + bucket);
      SetBucketVersioningConfigurationRequest setBucketVersioningConfigurationRequest =
              new SetBucketVersioningConfigurationRequest(bucket, configuration);
      s3Client.setBucketVersioningConfiguration(setBucketVersioningConfigurationRequest);
    }
  }

  public void enableBucketEncryption(String bucket) {
    if (sseEnabled) {
      if (sseType.compareToIgnoreCase(CloudS3Encryption.SSE_S3.toString()) == 0
              || sseType.compareToIgnoreCase(CloudS3Encryption.SSE_KMS.toString()) == 0) {

        ServerSideEncryptionByDefault serverSideEncryptionByDefault =
                new ServerSideEncryptionByDefault();
        ServerSideEncryptionRule rule = new ServerSideEncryptionRule()
                .withApplyServerSideEncryptionByDefault(serverSideEncryptionByDefault);

        if (sseType.compareToIgnoreCase(CloudS3Encryption.SSE_S3.toString()) == 0) {
          serverSideEncryptionByDefault.withSSEAlgorithm(SSEAlgorithm.AES256);
        } else {
          serverSideEncryptionByDefault.withSSEAlgorithm(SSEAlgorithm.KMS);
          rule.withBucketKeyEnabled(sseBucketKeyEnable);
        }

        ServerSideEncryptionConfiguration serverSideEncryptionConfiguration =
                new ServerSideEncryptionConfiguration().withRules(Collections.singleton(rule));
        SetBucketEncryptionRequest setBucketEncryptionRequest = new SetBucketEncryptionRequest()
                .withServerSideEncryptionConfiguration(serverSideEncryptionConfiguration)
                .withBucketName(bucket);
        s3Client.setBucketEncryption(setBucketEncryptionRequest);
      } else {
        throw new IllegalArgumentException("Encryption type (" + sseType + ") supported.");
      }
    }
  }

  @Override
  public Object getCloudClient() {
    return s3Client;
  }

  private void setUploadHeaders(Object req,
                                ObjectMetadata metadata) throws IOException {
    if (sseEnabled) {
      if (sseType.compareToIgnoreCase(CloudS3Encryption.SSE_S3.toString()) == 0) {
        metadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
      } else if (sseType.compareToIgnoreCase(CloudS3Encryption.SSE_KMS.toString()) == 0) {
        if (req instanceof InitiateMultipartUploadRequest) {
          ((InitiateMultipartUploadRequest) req).withSSEAwsKeyManagementParams(getSSEAwsKeyManagementParams());
          ((InitiateMultipartUploadRequest) req).setBucketKeyEnabled(sseBucketKeyEnable);
        } else if (req instanceof CopyObjectRequest) {
          ((CopyObjectRequest) req).withSSEAwsKeyManagementParams(getSSEAwsKeyManagementParams());
          ((CopyObjectRequest) req).setBucketKeyEnabled(sseBucketKeyEnable);
        } else if (req instanceof PutObjectRequest) {
          ((PutObjectRequest) req).withSSEAwsKeyManagementParams(getSSEAwsKeyManagementParams());
          ((PutObjectRequest) req).setBucketKeyEnabled(sseBucketKeyEnable);
        } else {
          throw new UnsupportedOperationException("Implement me");
        }
      } else {
        throw new IOException("Encryption type (" + sseType + ") supported.");
      }
    }

    if(bucketOwnerFullControll) {
      if (req instanceof InitiateMultipartUploadRequest) {
        ((InitiateMultipartUploadRequest) req).setCannedACL(CannedAccessControlList.BucketOwnerFullControl);
      } else if (req instanceof CopyObjectRequest) {
        ((CopyObjectRequest) req).setCannedAccessControlList(CannedAccessControlList.BucketOwnerFullControl);
      } else if (req instanceof PutObjectRequest) {
        ((PutObjectRequest) req).setCannedAcl(CannedAccessControlList.BucketOwnerFullControl);
      } else {
        throw new UnsupportedOperationException("Implement me");
      }

    }
  }

  private SSEAwsKeyManagementParams getSSEAwsKeyManagementParams() {
    if (sseKeyARN.isEmpty()) {
      return new SSEAwsKeyManagementParams();
    } else {
      return new SSEAwsKeyManagementParams(sseKeyARN);
    }
  }
}
