package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud;

import com.lc.repackaged.com.google.api.gax.paging.Page;
import com.lc.repackaged.com.google.api.gax.retrying.RetrySettings;
import com.lc.repackaged.com.google.cloud.storage.Blob;
import com.lc.repackaged.com.google.cloud.storage.BlobId;
import com.lc.repackaged.com.google.cloud.storage.BlobInfo;
import com.lc.repackaged.com.google.cloud.storage.Bucket;
import com.lc.repackaged.com.google.cloud.storage.BucketInfo;
import com.lc.repackaged.com.google.cloud.storage.Storage;
import com.lc.repackaged.com.google.cloud.storage.StorageException;
import com.lc.repackaged.com.google.cloud.storage.StorageOptions;
import com.lc.repackaged.com.google.common.annotations.VisibleForTesting;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;

public class CloudPersistenceProviderGCSImpl implements CloudPersistenceProvider {

  public static final Log LOG = LogFactory.getLog(CloudPersistenceProviderGCSImpl.class);

  private Storage storage;
  private final Configuration conf;
  private final int numBucketDeletionThreads;
  private final int prefixSize;
  private int numTransferThreads;
  private long partSize;
  private final String bucketLocation;
  private final boolean versioningEnabled;
  private final boolean requesterPays;
  private final String requesterPaysProject;

  public CloudPersistenceProviderGCSImpl(Configuration conf) {
    this.numBucketDeletionThreads =
            conf.getInt(DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_KEY,
                    DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_DEFAULT);
    this.prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    numTransferThreads = conf.getInt(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS,
            DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS_DEFAULT);
    if (numTransferThreads < 2) {
      LOG.warn(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS +
              " must be at least 2: forcing to 2.");
      numTransferThreads = 2;
    }

    partSize = conf.getLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE,
            DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE_DEFAULT);

    if (partSize < 5 * 1024 * 1024) {
      LOG.error(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE + " must be at least 5 MB");
      partSize = 5 * 1024 * 1024;
    }

    bucketLocation = conf.get(DFSConfigKeys.GCS_BUCKET_LOCATION_KEY,
            DFSConfigKeys.GCS_BUCKET_LOCATION_DEFAULT);
    versioningEnabled = conf.getBoolean(DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_KEY,
            DFSConfigKeys.GCS_BUCKET_ENABLE_VERSIONING_DEFAULT);
    requesterPays = conf.getBoolean(DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_KEY,
            DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_DEFAULT);
    requesterPaysProject = conf.get(DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_PROJECT_KEY,
            DFSConfigKeys.GCS_BUCKET_REQUESTER_PAYS_PROJECT_DEFAULT);
    if (requesterPays) {
      if (requesterPaysProject == "" || requesterPaysProject == null) {
        throw new IllegalArgumentException("Requster pays option is set. Please specify project " +
                "name for google cloud.");
      }
    }

    this.conf = conf;
    connect();
  }

  private void connect() {
    int retryCount = conf.getInt(DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_KEY,
            DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_DEFAULT);
    RetrySettings retrySettings = RetrySettings.newBuilder().setMaxAttempts(retryCount).build();
    storage = StorageOptions.newBuilder().setRetrySettings(retrySettings).build().
                    getDefaultInstance().getService();
  }

  @Override
  public void deleteAllBuckets(String prefix) throws IOException, ExecutionException, InterruptedException {
    long startTime = System.currentTimeMillis();
    ExecutorService tPool = Executors.newFixedThreadPool(numBucketDeletionThreads);
    List<Bucket> buckets = listAllBuckets();
    LOG.info("HopsFS-Cloud. Deleting all of the buckets with prefix \"" + prefix +
            "\" for this user. Number of deletion threads " + numBucketDeletionThreads);
    try {
      for (Bucket b : buckets) {
        if (b.getName().startsWith(prefix.toLowerCase())) {
          emptyBucket(b.getName(), false, tPool);
          deleteBucket(b.getName());
        }
      }
    } finally {
      tPool.shutdown();
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Deleted all buckets. "
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
  }

  @Override
  public boolean existsCID(String bucket) throws IOException {
    if (!bucketExists(bucket)) {
      return false;
    }

    return objectExists(bucket, CloudHelper.CID_FILE);
  }

  @Override
  public void setCID(String bucket, String cid) throws IOException {
    long startTime = System.currentTimeMillis();
      try {
        if (!bucketExists(bucket)) {
          throw new IOException("Bucket " + bucket + " does not exist");
        }

        BlobId blobId = BlobId.of(bucket, CloudHelper.CID_FILE);
        BlobInfo objectInfo = BlobInfo.newBuilder(blobId).build();
        storage.create(objectInfo, cid.getBytes(), getBlobTargetOptions());
      } catch (StorageException e) {
        LOG.info("HopsFS-Cloud: Exception in setCID, CID: " +
                cid + " Error: " + e.getMessage());
        throw new IOException(e);
      }
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  set CID. Bucket: " + bucket
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
  }

  @Override
  public String getCID(String bucket) throws IOException {
    long startTime = System.currentTimeMillis();
    String cid = null;
    try {
      if (!bucketExists(bucket)) {
        throw new IOException("Bucket " + bucket + " does not exist");
      }

      BlobId blobId = BlobId.of(bucket, CloudHelper.CID_FILE);
      byte[] cidBytes = storage.readAllBytes(blobId, getStorageBlobSourceOptions());
      cid = new String(cidBytes);
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in getCID. Bucket: " +
              bucket + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  get CID. "
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return cid;
  }

  @Override
  public boolean isEmpty(String bucket) throws IOException {
    if (!bucketExists(bucket)) {
      throw new IOException("Bucket "+bucket+" does not exist");
    }

    Page<Blob> blobs = storage.list(bucket, getBlobListOptions());
    if (blobs.iterateAll().iterator().hasNext()) {
      return false;
    }

    if(versioningEnabled) {
      blobs = storage.list(bucket, getBlobListOptions(Storage.BlobListOption.versions(true)));
      if (blobs.iterateAll().iterator().hasNext()) {
        return false;
      }
    }
    return true;
  }

  private List<Bucket> listAllBuckets() {
    List<Bucket> buckets = new ArrayList();
    Page<Bucket> bucketPage = storage.list(getBucketListOption());

    while (true) {
      Iterator<Bucket> bucketIterator = bucketPage.iterateAll().iterator();
      while (bucketIterator.hasNext()) {
        Bucket bucket = bucketIterator.next();
        buckets.add(bucket);
      }
      if (!bucketPage.hasNextPage()) {
        break;
      }
      bucketPage = bucketPage.getNextPage();
    }
    return buckets;
  }

  @Override
  public void format(List<String> buckets) throws IOException {

    System.out.println("HopsFS-Cloud. Deleting all of the buckets used by HopsFS.");
    ExecutorService tPool = Executors.newFixedThreadPool(numBucketDeletionThreads);
    try {
      for (String bucket : buckets) {

        // make sure that the bucket supports versioning if needed
        if (versioningEnabled && !isVersioningSupported(bucket)) {
          throw new IOException("Cannot format file system. Versioning is enabled. However the bucket does " +
                  "not support versioning");
        }

        emptyBucket(bucket, true, tPool);
      }

    } catch (InterruptedException | ExecutionException e) {
      LOG.warn(e);
    } finally {
      tPool.shutdown();
    }
  }

  private void deleteBucket(String bucketName) throws IOException {
    // delete bucket
      LOG.info("HopsFS-Cloud. Deleting bucket " + bucketName);
      boolean deleted = storage.delete(bucketName, getBucketSourceOptions());
      if (!deleted) {
        throw new IOException("Unable to delete the bucket: " + bucketName + " Bucket Not Found");
      }
  }

  private void emptyBucket(final String bucketName, boolean onlyDeleteHopsFSData,
                           ExecutorService tPool)
          throws IOException, ExecutionException, InterruptedException {
    final AtomicInteger deletedBlocks = new AtomicInteger(0);

    if (!bucketExists(bucketName)) {
      throw new IOException("Cannot format file system. The bucket does not exist");
    }

    try {
      // empty bucket
      Page<Blob> blobs = storage.list(bucketName, getBlobListOptions());
      while (true) {
        final List<Future> futures = new ArrayList<>();
        Iterator<Blob> blobIterator = blobs.iterateAll().iterator();
        while (blobIterator.hasNext()) {
          Blob blob = blobIterator.next();

          if (onlyDeleteHopsFSData && !blob.getName().startsWith(CloudHelper.PREFIX_STR)) {
            continue;
          }

          Callable task = new Callable<Object>() {
            @Override
            public Object call() throws Exception {
              blob.delete(getBlobSourceOptions());
              String msg = "\rDeleted Blocks: " + (deletedBlocks.incrementAndGet());
              System.out.print(msg);
              return null;
            }
          };
          futures.add(tPool.submit(task));
        }
        // wait for deletion of all blocks in this page
        for (Future future : futures) {
          future.get();
        }
        if (!blobs.hasNextPage()) {
          break;
        }
        blobs = blobs.getNextPage();
      }

      // delete all the versions
      blobs = storage.list(bucketName, getBlobListOptions(Storage.BlobListOption.versions(true)));
      while (true) {
        final List<Future> futures = new ArrayList<>();
        Iterator<Blob> blobIterator = blobs.iterateAll().iterator();
        while (blobIterator.hasNext()) {
          Blob blob = blobIterator.next();

          if (onlyDeleteHopsFSData && !blob.getName().startsWith(CloudHelper.PREFIX_STR)) {
            continue;
          }

          Callable task = new Callable<Object>() {
            @Override
            public Object call() throws Exception {
              blob.delete(getBlobSourceOptions());
              String msg = "\rDeleted Block versions: " + (deletedBlocks.incrementAndGet());
              System.out.print(msg);
              return null;
            }
          };
          futures.add(tPool.submit(task));
        }
        // wait for deletion of all blocks in this page
        for (Future future : futures) {
          future.get();
        }
        if (!blobs.hasNextPage()) {
          break;
        }
        blobs = blobs.getNextPage();
      }

      // delete CID file
      if (objectExists(bucketName, CloudHelper.CID_FILE)) {
        BlobId blobId = BlobId.of(bucketName, CloudHelper.CID_FILE);
        storage.delete(blobId, getStorageBlobSourceOptions());
      }

    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in emptyAndDeleteBucket. Error: " + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  public boolean bucketExists(String bucketName) throws IOException {
    try {
      Bucket bucket = storage.get(bucketName, getBucketGetOptions());
      if (bucket == null) {
        return false;
      }
      return bucket.exists();
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in bucketExists, Bucket: " +
              bucketName + " Error: " + e.getMessage());
      throw new IOException(e);
    }

  }

  public void createBucket(String bucketName) throws IOException {
    try {
      if (!bucketExists(bucketName)) {
        BucketInfo.Builder bucketInfoBuilder =
                BucketInfo.newBuilder(bucketName).setLocation(bucketLocation);
        if (requesterPays) {
          bucketInfoBuilder.setRequesterPays(requesterPays);
        }
        BucketInfo bucketInfo = bucketInfoBuilder.build();
        storage.create(bucketInfo, getBucketTargetOptions());

        enableVersioning(bucketName);
      } else {
        LOG.info("HopsFS-Cloud. Bucket already exists. Bucket Name: " + bucketName);
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in createBucket, Bucket: " +
              bucketName + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  public void enableVersioning(String bucketName) {
    if (versioningEnabled) {
      Bucket bucket = storage.get(bucketName, getBucketGetOptions());
      bucket.toBuilder().setVersioningEnabled(true).build().update();
    }
  }

  @Override
  public void checkAllBuckets(List<String> buckets) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      final int retry = 300;  // keep trying until the newly created container is available
      for (String bucketStr : buckets) {
        LOG.debug("Checking bucket: " + bucketStr);
        boolean exists = bucketExists(bucketStr);
        if (!exists) {
          throw new IllegalStateException("GCS Bucket " + bucketStr + " needed for the " +
                  "file system does not exists");
        } else {
          //check the container is writable
          UUID uuid = UUID.randomUUID();
          File file1 = new File("/tmp/" + uuid);
          File file2 = new File("/tmp/" + uuid + ".downloaded");
          try {
            String message = "hello! hello! testing! testing! testing 1 2  3!";
            FileWriter fw = new FileWriter(file1);
            fw.write(message);
            fw.close();


            BlobId blobId = BlobId.of(bucketStr, uuid.toString());
            Blob upBlob = storage.get(blobId, getBlobGetOptions());
            assert upBlob == null;
            BlobInfo objectInfo = BlobInfo.newBuilder(blobId).build();
            storage.create(objectInfo, Files.readAllBytes(Paths.get(file1.getAbsolutePath())),
                    getBlobTargetOptions());

            Blob downBlob = storage.get(blobId);
            downBlob.downloadTo(Paths.get(file2.getAbsolutePath()));

            //delete tmp file.
            storage.delete(BlobId.of(bucketStr, uuid.toString()), getStorageBlobSourceOptions());

            assert FileUtils.contentEquals(file1, file2) == true;
          } catch (Exception e) {
            throw new IllegalStateException("Write test for GCS bucket: " + bucketStr +
                    " failed. " + e);
          } finally {
            file1.delete();
            file2.delete();
          }
        }
      }

      LOG.info("HopsFS-Cloud. Check all buckets: " + Arrays.toString(buckets.toArray())
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in checkAllBuckets, Buckets: " +
              Arrays.toString(buckets.toArray()) + " Error: " + e.getMessage());
      throw new IOException(e);
    }

  }

  @Override
  public int getPrefixSize() {
    return prefixSize;
  }

  @Override
  public void uploadObject(String bucket, String objectKey, File object, Map<String, String> metadata) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      LOG.info("HopsFS-Cloud. Put Object. Bucket: " + bucket + " Object Key: " + objectKey +" " +
              " Object Size: " +objectKey.length());
      BlobId blobId = BlobId.of(bucket, objectKey);
      BlobInfo objectInfo = BlobInfo.newBuilder(blobId).build();
      Blob upBlob = storage.create(objectInfo,
              Files.readAllBytes(Paths.get(object.getAbsolutePath())), getBlobTargetOptions());

      if (metadata != null) {
        upBlob.toBuilder().setMetadata(metadata).build().update();
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in uploadObject. Bucket: " +
              bucket + " Key: " + objectKey + " File: " +
              object.getAbsolutePath() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  Upload object. Bucket: " + bucket + " Object Key: " + objectKey
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
  }

  @Override
  public boolean objectExists(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId blobId = BlobId.of(bucket, objectKey);
      Blob blob = storage.get(blobId, getBlobGetOptions());


      boolean objExists = false;
      if (blob != null) {
        objExists = blob.exists();
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Object Exists. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }

      return objExists;
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in objectExists. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }

  }

  @Override
  public Map<String, String> getUserMetaData(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId blobId = BlobId.of(bucket, objectKey);
      Blob blob = storage.get(blobId, getBlobGetOptions());
      if (blob == null) {
        throw new IOException("Object: " + objectKey + " not found in the bucket: " + bucket);

      } else {
        Map<String, String> metadata = blob.getMetadata();

        if (metadata == null) {
          metadata = new HashMap<>();
        }

        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Get Metadata. Bucket: " + bucket + " Object Key: " + objectKey
                  + " Time (ms): " + (System.currentTimeMillis() - startTime));
        }

        return metadata;
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in getUserMetaData. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public long getObjectSize(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId blobId = BlobId.of(bucket, objectKey);
      Blob blob = storage.get(blobId, getBlobGetOptions());

      if (blob == null) {
        throw new IOException("Object: " + objectKey + " not found in the bucket: " + bucket);
      } else {
        long objSize = blob.getSize();
        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Get object size. Bucket: " + bucket + " Object Key: " + objectKey
                  + " Time (ms): " + (System.currentTimeMillis() - startTime));
        }
        return objSize;
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in getObjectSize. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void downloadObject(String bucket, String objectKey, File path) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      if (path.exists()) {
        path.delete();
      } else {
        //make sure that all parent dirs exists
        if (!path.getParentFile().exists()) {
          path.getParentFile().mkdirs();
        }
      }

      Random rand = new Random(System.currentTimeMillis());
      File tmpFile = new File(path.getAbsolutePath() + "." + rand.nextLong() + ".downloading");
      if (tmpFile.exists()) {
        tmpFile.delete();
      }

      BlobId blobId = BlobId.of(bucket, objectKey);
      Blob blob = storage.get(blobId, getBlobGetOptions());
      if (blob == null) {
        throw new IOException("Object: " + objectKey + " not found in the bucket: " + bucket);
      } else {
        blob.downloadTo(Paths.get(tmpFile.getAbsolutePath()));
      }
      tmpFile.renameTo(path);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Download object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in downloadObject Bucket: " +
              bucket + " ObjKey: " + objectKey + " File: " + path.getAbsolutePath() +
              " Error: " + e.getMessage());
      throw new IOException(e);
    }

  }

  @Override
  public void deleteObject(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId blobId = BlobId.of(bucket, objectKey);
      if (!storage.delete(blobId, getStorageBlobSourceOptions())) {
        throw new IOException("Object: " + objectKey + " not found in the bucket: " + bucket);
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Delete Object. Bucket: " + bucket + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteObject. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @VisibleForTesting
  @Override
  public void renameObject(String srcBucket, String dstBucket, String srcKey, String dstKey) throws IOException {
    long startTime = System.currentTimeMillis();
    copyObject(srcBucket, dstBucket, srcKey, dstKey, null);
    try {
      BlobId srcBlobId = BlobId.of(srcBucket, srcKey);
      storage.delete(srcBlobId, getStorageBlobSourceOptions());

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Rename Object. Src Bucket: " + srcBucket + " Src Object Key: " + srcKey +
                " Dst Bucket: " + dstBucket + " Dst Object Key: " + dstKey +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in renameObject  Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void copyObject(String srcBucket, String dstBucket, String srcKey, String dstKey, Map<String, String> newObjMetadata) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId srcBlobId = BlobId.of(srcBucket, srcKey);
      Blob blob = storage.get(srcBlobId, getBlobGetOptions());
      if (blob == null) {
        throw new IOException("Object: " + srcKey + " not found in the bucket: " + srcBucket);
      } else {
        BlobId dstBlobId = BlobId.of(dstBucket, dstKey);
        blob.copyTo(dstBlobId);

        if (newObjMetadata != null) {
          // set the user metadata
          Blob newBlob = storage.get(dstBlobId, getBlobGetOptions());
          newBlob.toBuilder().setMetadata(newObjMetadata).build().update();
        }

        if (LOG.isDebugEnabled()) {
          LOG.debug("HopsFS-Cloud. Copy Object. Src Bucket: " + srcBucket + " Src Object Key: " + srcKey +
                  " Dst Bucket: " + dstBucket + " Dst Object Key: " + dstKey +
                  " Time (ms): " + (System.currentTimeMillis() - startTime));
        }
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in  copyObject  Src Bucket: " +
              srcBucket + " Dst Bucket: " + dstBucket + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public Map<BlockIDAndGSTuple, CloudBlock> getAll(String prefix, List<String> buckets) throws IOException {
    long startTime = System.currentTimeMillis();
    Map<BlockIDAndGSTuple, CloudBlock> allBlocks = new HashMap<>();
    try {
      for (String b : buckets) {
        getAllInt(allBlocks, b, prefix);
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in  getAll.  Prefix: " +
              prefix + " Error: " + e.getMessage());
      throw new IOException(e);
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get all objects. Buckets: " + Arrays.toString(buckets.toArray()) + " " +
              " Prefix:" + prefix + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return allBlocks;
  }

  private void getAllInt(Map<BlockIDAndGSTuple, CloudBlock> blocks, String bucketStr,
                         String prefix) throws IOException {
    try {

      Map<BlockIDAndGSTuple, CloudObject> blockObjs = new HashMap<>();
      Map<BlockIDAndGSTuple, CloudObject> metaObjs = new HashMap<>();

      Bucket bucket = storage.get(bucketStr, getBucketGetOptions());
      Page<Blob> page = bucket.list(getBlobListOptions(Storage.BlobListOption.prefix(prefix)));

      while (true) {
        Iterator<Blob> iterator = page.iterateAll().iterator();
        while (iterator.hasNext()) {
          Blob blob = iterator.next();
          processListBlob(blockObjs, metaObjs, blob);
        }
        if (!page.hasNextPage()) {
          break;
        }
        page = page.getNextPage();
      }

      CloudPersistenceProviderS3Impl.mergeMetaAndBlockObjects(metaObjs, blockObjs, blocks);
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in  getAllInt.  Prefix: " +
              prefix + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private void processListBlob(Map<BlockIDAndGSTuple, CloudObject> blockObjs,
                               Map<BlockIDAndGSTuple, CloudObject> metaObjs, Blob blob) {

    if (CloudHelper.isPartialGCSFile(blob.getName())) {
      return;
    }

    String key = blob.getName();
    CloudObject co = new CloudObject();
    co.setBucket(blob.getBucket());
    co.setKey(key);
    co.setSize(blob.getSize());
    co.setLastModifiedTime(blob.getUpdateTime());

    BlockIDAndGSTuple idAndGS = CloudHelper.getIDAndGSFromKey(key);
    if (idAndGS != null && CloudHelper.isBlockFilename(key)) {
      blockObjs.put(idAndGS, co);
    } else if (idAndGS != null || CloudHelper.isMetaFilename(key)) {
      metaObjs.put(idAndGS, co);
    } else {
      LOG.warn("HopsFS-Cloud. File system objects are tampered. The " + key + " is not HopsFS object.");
    }
  }

  @Override
  public List<String> getAllHopsFSDirectories(List<String> buckets) throws IOException {
    long startTime = System.currentTimeMillis();
    List<String> dirs = new ArrayList<>();
    try {
      for (String b : buckets) {
        getAllDirectoriesInt(b, dirs);
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in getAllDirectories Bucket: " +
              Arrays.toString(buckets.toArray()) + " Error: " + e.getMessage());
      throw new IOException(e);
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get all directories. Buckets: " + Arrays.toString(buckets.toArray()) + " " +
              " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return dirs;
  }


  private void getAllDirectoriesInt(String bucketStr, List<String> dirs) throws IOException {
    Bucket bucket = storage.get(bucketStr, getBucketGetOptions());
    Page<Blob> page = bucket.list(getBlobListOptions(Storage.BlobListOption.prefix(CloudHelper.ROOT_PREFIX),
            Storage.BlobListOption.currentDirectory()));
    while (true) {
      Iterator<Blob> iterator = page.iterateAll().iterator();
      while (iterator.hasNext()) {
        Blob blob = iterator.next();
        if (blob.getName().contains(CloudHelper.PREFIX_STR)) {
          dirs.add(blob.getName());
        }
      }
      if (!page.hasNextPage()) {
        break;
      }
      page = page.getNextPage();
    }
  }


  @Override
  public long getPartSize() {
    return partSize;
  }

  @Override
  public int getXferThreads() {
    return numTransferThreads;
  }

  @Override
  public UploadID startMultipartUpload(String bucket, String objectKey, Map<String, String> metadata) throws IOException {
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Starting Multipart Upload.");
    }
    return null;
  }

  @Override
  public PartRef uploadPart(String bucket, String objectKey, UploadID uploadID, int partNo, File file, long startPos, long endPos) throws IOException {
    long startTime = System.currentTimeMillis();
    String objectName = objectKey.substring(objectKey.lastIndexOf('/') + 1);
    String dir = objectKey.substring(0, objectKey.lastIndexOf('/'));
    String key = dir + "/" + CloudHelper.GCS_MULTI_PART_DIR + "/" +
            objectName + CloudHelper.GCS_MULTI_PART_SUFFIX + partNo;
    try {
      //data
      byte[] partData = new byte[(int) (endPos - startPos)];
      RandomAccessFile input = new RandomAccessFile(file, "r");
      input.seek(startPos);
      input.readFully(partData);
      input.close();

      BlobId blobId = BlobId.of(bucket, key);
      BlobInfo objectInfo = BlobInfo.newBuilder(blobId).build();
      Blob upBlob = storage.create(objectInfo, partData, getBlobTargetOptions());

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Upload object part. Bucket: " + bucket + " Object Key: " + key
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }

      return new GCSPartRef(partNo, key);
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in uploadPart. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " + uploadID.toString() +
              " Part No:" + partNo + " File: " + file.getAbsolutePath() +
              " Start Pos: " + startPos + " End Pos: " + endPos + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void finalizeMultipartUpload(String bucket, String objectKey, UploadID uploadID,
                                      List<PartRef> parts) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlobId blobId = BlobId.of(bucket, objectKey);
      BlobInfo blobInfo = BlobInfo.newBuilder(blobId).build();
      Storage.ComposeRequest.Builder requestBuilder = Storage.ComposeRequest.newBuilder();
      requestBuilder.setTarget(blobInfo);
      requestBuilder.setTargetOptions(getBlobTargetOptions());
      for (PartRef p : parts) {
        GCSPartRef part = (GCSPartRef) p;
        requestBuilder.addSource(part.getKey());
      }
      storage.compose(requestBuilder.build());

      //delete sub parts. Completely remove sub part including all versions
      if (versioningEnabled) {
        for (PartRef part : parts) {
          GCSPartRef gcsPart = (GCSPartRef) part;
          deleteAllVersions(bucket, gcsPart.getKey());
        }
      } else {
        List<BlobId> toDelete = new ArrayList();
        for (PartRef part : parts) {
          GCSPartRef gcsPart = (GCSPartRef) part;
          toDelete.add(BlobId.of(bucket, gcsPart.getKey()));
        }
        storage.delete(toDelete);
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Finalize multipart upload. Bucket: " + bucket + " " +
                "Object Key: " + objectKey + " Parts: " + Arrays.toString(parts.toArray())
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in finalizeMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void abortMultipartUpload(String bucket, String objectKey,
                                   UploadID uploadID /* not needed for GCS*/)
          throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      String objectName = objectKey.substring(objectKey.lastIndexOf('/') + 1);
      String dir = objectKey.substring(0, objectKey.lastIndexOf('/'));
      String prefix = dir + "/" + CloudHelper.GCS_MULTI_PART_DIR + "/" + objectName;
      StringBuilder deleteMsg = new StringBuilder();
      List<Blob> blobs = listPrefix(bucket, prefix);
      for (Blob b : blobs) {
        deleteObject(bucket, b.getName());
        deleteMsg.append(b.getName()).append(", ");
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Abort multipart upload. Bucket: " + bucket + " " +
                "Object Key: " + objectKey + " Parts Prefix: " + prefix +
                " Parts: [" + deleteMsg
                + "] Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in abortMultipartUpload. Bucket: " +
              bucket + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private Map<BlockIDAndGSTuple, List<Blob>> listPartialBlocks(List<String> buckets,
                                                               String prefix) throws IOException {
    Map<BlockIDAndGSTuple, List<Blob>> blobsMap = new HashMap<>();
    List<Blob> blobs = new ArrayList<>();
    for (String bucket : buckets) {
      blobs.addAll(listPrefix(bucket, prefix));
    }
    for (Blob b : blobs) {
      if (!CloudHelper.isPartialGCSFile(b.getName())) {
        LOG.warn("HopsFS-Cloud. Unrecognized object: " + b.getName() +
                " search prefix \"" + prefix + "\"");
        continue;
      }

      BlockIDAndGSTuple id = CloudHelper.getIDAndGSFromKey(b.getName());
      List<Blob> blobList = blobsMap.get(id);
      if (blobList == null) {
        blobList = new ArrayList<>();
      }
      blobList.add(b);
      blobsMap.put(id, blobList);
    }
    return blobsMap;
  }

  private List<Blob> listPrefix(String bucketStr, String prefix) throws IOException {
    List<Blob> blobs = new ArrayList<>();
    try {
      Bucket bucket = storage.get(bucketStr, getBucketGetOptions());
      Page<Blob> page = bucket.list(getBlobListOptions(Storage.BlobListOption.prefix(prefix)));

      while (true) {
        Iterator<Blob> iterator = page.iterateAll().iterator();
        while (iterator.hasNext()) {
          Blob blob = iterator.next();
          blobs.add(blob);
        }
        if (!page.hasNextPage()) {
          break;
        }
        page = page.getNextPage();
      }

    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in listPrefix. Bucket: " +
              bucketStr + " Prefix: " + prefix  + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    return blobs;
  }

  @Override
  public List<ActiveMultipartUploads> listMultipartUploads(List<String> buckets, String prefix) throws IOException {
    long startTime = System.currentTimeMillis();
    Map<BlockIDAndGSTuple, List<Blob>> blocksMap = listPartialBlocks(buckets, prefix);
    List<ActiveMultipartUploads> blocksList = new ArrayList<>();
    for (BlockIDAndGSTuple key : blocksMap.keySet()) {
      List<Blob> parts = blocksMap.get(key);
      blocksList.add(new GCSActiveMultipartUploads(key, parts));
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  List multipart uploads. Buckets: " + Arrays.toString(buckets.toArray()) +
              " " + "Prefix: " + prefix + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }

    return blocksList;
  }

  @Override
  public boolean restoreDeletedBlock(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      //list all versions and then recover
      List<Blob> versions = listAllVersions(bucket, objectKey);
      boolean restored = false;

      if (versions.size() == 0) {
        restored = false;
      } else {
        Blob latestVersion = versions.get(versions.size() - 1);
        if (latestVersion.getDeleteTime() == null) { // not a deleted version
          restored = false;
        } else {
          Storage.CopyRequest copyRequest =
                  Storage.CopyRequest.newBuilder()
                          .setSource(BlobId.of(bucket, objectKey, latestVersion.getGeneration()))
                          .setTarget(BlobId.of(bucket, objectKey), getBlobTargetOptions())
                          .setSourceOptions(getStorageBlobSourceOptions())
                          .build();
          storage.copy(copyRequest);
          latestVersion.delete();
          restored = true;
        }
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Restore deleted version. Bucket: " + bucket +
                " Key: " + objectKey + " Success: " + restored +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }

      return restored;
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in restoreDeletedBlock. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public boolean isVersioningSupported(String bucketStr) throws IOException {
    if (!versioningEnabled) {
      return false;
    }
    long startTime = System.currentTimeMillis();
    try {
      Bucket bucket = storage.get(bucketStr, getBucketGetOptions());
      Boolean versioning = bucket.versioningEnabled();
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Is versioning supported. Bucket: " + bucket +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      if (versioning != null) {
        return versioning;
      } else {
        return false;
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in isVersioningSupported. Bucket: " +
              bucketStr + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void deleteAllVersions(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      //list all versions and then delete them
      Page<Blob> page = storage.list(bucket, getBlobListOptions(Storage.BlobListOption.prefix(objectKey),
              Storage.BlobListOption.versions(true)));
      while (true) {
        Iterator<Blob> blobItr = page.iterateAll().iterator();
        while (blobItr.hasNext()) {
          Blob blob = blobItr.next();
          blob.delete(getBlobSourceOptions());
        }
        if (!page.hasNextPage()) {
          break;
        }
        page = page.getNextPage();
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud.  Delete all versions. Bucket: " + bucket +
                " " + "Object key: " + objectKey + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteAllVersions. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void deleteOldVersions(String bucket, String objectKey) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      //list all versions and then recover
      List<Blob> versions = listAllVersions(bucket, objectKey);

      if (versions.size() == 0) {
        throw new IOException("No versions found");
      } else {
        for (int i = 0; i < versions.size() - 1; i++) {
          Blob version = versions.get(i);
          version.delete(getBlobSourceOptions());
          if (LOG.isDebugEnabled()) {
            LOG.debug("HopsFS-Cloud. Deleted version " + version.getGeneration() + " of " +
                    "Object: " + objectKey);
          }
        }
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Deleted all old versions " + " of " + "Object: " + objectKey +
                " Time: " + (System.currentTimeMillis() - startTime) + " ms");
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteOldVersion. Bucket: " +
              bucket + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  public List<Blob> listAllVersions(String bucket, String prefix) throws IOException {
    // during testing with small pages sizes I have seen that some blob versions are
    // returned multiple times and there is order of the returned items
    Map<Long, Blob> versionsMap = new HashMap<>();
    try {
      Page<Blob> page = storage.list(bucket, getBlobListOptions(Storage.BlobListOption.prefix(prefix),
              Storage.BlobListOption.versions(true), Storage.BlobListOption.pageSize(5)));
      while (true) {
        Iterator<Blob> blobItr = page.iterateAll().iterator();
        while (blobItr.hasNext()) {
          Blob blob = blobItr.next();
          versionsMap.put(blob.getGeneration(), blob);
        }
        if (!page.hasNextPage()) {
          break;
        }
        page = page.getNextPage();
      }
    } catch (StorageException e) {
      LOG.info("HopsFS-Cloud: Exception in listAllVersions. Bucket: " +
              bucket + " ObjKey: " + prefix + " Error: " + e.getMessage());
      throw new IOException(e);
    }

    List<Blob> versions = new ArrayList<>(versionsMap.values());
    Collections.sort(versions, new Comparator<Blob>() {
      @Override
      public int compare(Blob o1, Blob o2) {
        return Long.compare(o1.getUpdateTime(), o2.getUpdateTime());
      }
    });
    return versions;
  }

  @Override
  public void shutdown() {
  }

  @Override
  public Object getCloudClient() {
    return storage;
  }

  private Storage.BucketListOption[] getBucketListOption(Storage.BucketListOption... options) {
    List<Storage.BucketListOption> optionsList = new ArrayList<>();
    for (Storage.BucketListOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BucketListOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BucketListOption[optionsList.size()]);
  }

  private Storage.BlobListOption[] getBlobListOptions(Storage.BlobListOption... options) {
    List<Storage.BlobListOption> optionsList = new ArrayList<>();
    for (Storage.BlobListOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BlobListOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BlobListOption[optionsList.size()]);
  }

  private Storage.BucketSourceOption[] getBucketSourceOptions(Storage.BucketSourceOption... options) {
    List<Storage.BucketSourceOption> optionsList = new ArrayList<>();
    for (Storage.BucketSourceOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BucketSourceOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BucketSourceOption[optionsList.size()]);
  }

  private Storage.BucketGetOption[] getBucketGetOptions(Storage.BucketGetOption... options) {
    List<Storage.BucketGetOption> optionsList = new ArrayList<>();
    for (Storage.BucketGetOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BucketGetOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BucketGetOption[optionsList.size()]);
  }

  private Storage.BucketTargetOption[] getBucketTargetOptions(Storage.BucketTargetOption... options) {
    List<Storage.BucketTargetOption> optionsList = new ArrayList<>();
    for (Storage.BucketTargetOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BucketTargetOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BucketTargetOption[optionsList.size()]);
  }

  private Storage.BlobGetOption[] getBlobGetOptions(Storage.BlobGetOption... options) {
    List<Storage.BlobGetOption> optionsList = new ArrayList<>();
    for (Storage.BlobGetOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BlobGetOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BlobGetOption[optionsList.size()]);
  }

  private Storage.BlobTargetOption[] getBlobTargetOptions(Storage.BlobTargetOption... options) {
    List<Storage.BlobTargetOption> optionsList = new ArrayList<>();
    for (Storage.BlobTargetOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BlobTargetOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BlobTargetOption[optionsList.size()]);
  }

  private Storage.BlobSourceOption[] getStorageBlobSourceOptions(Storage.BlobSourceOption... options) {

    List<Storage.BlobSourceOption> optionsList = new ArrayList<>();
    for (Storage.BlobSourceOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Storage.BlobSourceOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Storage.BlobSourceOption[optionsList.size()]);
  }

  private Blob.BlobSourceOption[] getBlobSourceOptions(Blob.BlobSourceOption... options) {

    List<Blob.BlobSourceOption> optionsList = new ArrayList<>();
    for (Blob.BlobSourceOption opt : options) {
      optionsList.add(opt);
    }
    if (requesterPays) {
      optionsList.add(Blob.BlobSourceOption.userProject(requesterPaysProject));
    }
    return optionsList.toArray(new Blob.BlobSourceOption[optionsList.size()]);
  }

}
