/*
 * Copyright (C) 2020 Logical Clocks AB.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.cloud;

import com.azure.core.exception.AzureException;
import com.azure.core.http.rest.PagedIterable;
import com.azure.core.http.rest.PagedResponse;
import com.azure.core.util.BinaryData;
import com.azure.identity.DefaultAzureCredentialBuilder;
import com.azure.storage.blob.BlobClient;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.BlobServiceClient;
import com.azure.storage.blob.BlobServiceClientBuilder;
import com.azure.storage.blob.models.*;
import com.azure.storage.blob.specialized.BlockBlobClient;
import com.azure.storage.common.policy.RequestRetryOptions;
import com.azure.storage.common.policy.RetryPolicyType;
import com.google.common.annotations.VisibleForTesting;
import io.hops.metadata.hdfs.BlockIDAndGSTuple;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSConfigKeys;
import org.apache.hadoop.hdfs.protocol.CloudBlock;
import org.apache.hadoop.hdfs.server.common.CloudHelper;

import java.util.concurrent.*;

import java.io.*;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;


public class CloudPersistenceProviderAzureImpl implements CloudPersistenceProvider {

  public static final Log LOG = LogFactory.getLog(CloudPersistenceProviderAzureImpl.class);

  BlobServiceClient blobClient;
  private final Configuration conf;
  private final int prefixSize;
  private int maxThreads;
  private long partSize;
  private final int bucketDeletionThreads;
  private final boolean softDeletesEnabled;

  public CloudPersistenceProviderAzureImpl(Configuration conf) throws IOException {
    this.conf = conf;

    String storageConnectionString = null;
    String storageName = null;
    String containerName = null;
    String clientID = null;

    if (System.getenv("AZURE_STORAGE_CONNECTION_STRING") != null) {
      storageConnectionString = System.getenv("AZURE_STORAGE_CONNECTION_STRING");
    } else {
      // Assuming access to the storage using managed identities
      containerName = conf.get(DFSConfigKeys.AZURE_CONTAINER_KEY,
              DFSConfigKeys.AZURE_CONTAINER_DEFAULT);
      storageName = conf.get(DFSConfigKeys.AZURE_STORAGE_KEY,
              DFSConfigKeys.AZURE_STORAGE_DEFAULT);
      clientID = conf.get(DFSConfigKeys.DFS_AZURE_MGM_IDENTITY_CLIENT_ID_KEY,
              DFSConfigKeys.DFS_AZURE_MGM_IDENTITY_CLIENT_ID_KEY_DEFAULT);

      if (containerName == null || containerName.compareTo("") == 0
              || storageName == null || storageName.compareTo("") == 0) {
        throw new IllegalArgumentException("Azure storage name or container name  is not set properly");
      }
    }

    this.prefixSize = conf.getInt(DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_KEY,
            DFSConfigKeys.DFS_CLOUD_PREFIX_SIZE_DEFAULT);
    maxThreads = conf.getInt(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS,
            DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS_DEFAULT);
    if (maxThreads < 2) {
      LOG.warn(DFSConfigKeys.DFS_DN_CLOUD_MAX_TRANSFER_THREADS +
              " must be at least 2: forcing to 2.");
      maxThreads = 2;
    }

    partSize = conf.getLong(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE,
            DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE_DEFAULT);

    if (partSize < 5 * 1024 * 1024) {
      LOG.error(DFSConfigKeys.DFS_CLOUD_MULTIPART_SIZE + " must be at least 5 MB");
      partSize = 5 * 1024 * 1024;
    }

    int retryCount = conf.getInt(DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_KEY,
            DFSConfigKeys.DFS_CLOUD_FAILED_OPS_RETRY_COUNT_DEFAULT);

    bucketDeletionThreads = conf.getInt(DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_KEY,
            DFSConfigKeys.DFS_NN_MAX_THREADS_FOR_FORMATTING_CLOUD_BUCKETS_DEFAULT);

    softDeletesEnabled = conf.getBoolean(DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_KEY,
            DFSConfigKeys.AZURE_ENABLE_SOFT_DELETES_DEFAULT);
    try {

      RequestRetryOptions retryOptions = new RequestRetryOptions(RetryPolicyType.EXPONENTIAL,
              retryCount, (Integer) null /*timeout*/, null/*retryDelayInMs*/,
              null/*maxRetryDelayInMs*/, null/*secondaryHost*/);
      BlobServiceClientBuilder builder = new BlobServiceClientBuilder();
      if (storageConnectionString != null) {
        LOG.info("HopsFS-Cloud. Connection connection string");
        builder.connectionString(storageConnectionString);
      } else {
        LOG.info("HopsFS-Cloud. Connection using managed identities");
        builder.endpoint("https://" + storageName + ".blob.core.windows.net/" + containerName);

        DefaultAzureCredentialBuilder credentialBuilder = new DefaultAzureCredentialBuilder();
        if (clientID != null && clientID.compareTo("") != 0) {
          LOG.info("Using managed identity with client ID: " + clientID);
          credentialBuilder.managedIdentityClientId(clientID);
        }

        builder.credential(credentialBuilder.build());
      }
      blobClient = builder.retryOptions(retryOptions).buildClient();
      LOG.info("Azure Connected ");
    } catch (AzureException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void deleteAllBuckets(String prefix) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      ListBlobContainersOptions options = new ListBlobContainersOptions();
      options.setPrefix(prefix.toLowerCase());
      for (BlobContainerItem cont : blobClient.listBlobContainers(options, null)) {
        LOG.info("Deleting container: " + cont.getName());
        BlobContainerClient bcc = blobClient.getBlobContainerClient(cont.getName());
        if (bcc.exists()) {
          bcc.delete();
        }
      }
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Delete all containers. Prefix: " + prefix
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteAllBuckets. Prefix: " +
              prefix + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public boolean existsCID(String container) throws IOException {
    BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
    if (!bcc.exists()) {
      return false;
    }

    return objectExists(container, CloudHelper.CID_FILE);
  }

  @Override
  public void setCID(String container, String cid) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      if (!bucketExists(container)) {
        throw new IOException("Container " + container + " does not exist");
      }

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(CloudHelper.CID_FILE);
      BinaryData binaryData = BinaryData.fromString(cid);
      bc.upload(binaryData);
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in setCID, CID: " +
              cid + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  set CID. Bucket: " + container
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
  }

  @Override
  public String getCID(String container) throws IOException {
    long startTime = System.currentTimeMillis();
    String cid = null;
    try {
      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      if (!bcc.exists()) {
        throw new IOException("Container " + container + " does not exist");
      }
      BlobClient bc = bcc.getBlobClient(CloudHelper.CID_FILE);
      BinaryData binaryData = bc.downloadContent();
      cid = new String(binaryData.toBytes());
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in getCID. Container: " +
              container + " Error: " + e.getMessage());
      throw new IOException(e);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud.  get CID. "
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return cid;
  }

  @Override
  public boolean isEmpty(String container) throws IOException {
    BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
    if (!bcc.exists()) {
      throw new IOException("Container " + container + " does not exist");
    }
    ListBlobsOptions lbo = new ListBlobsOptions();
    lbo.setPrefix("");

    if (bcc.listBlobs(lbo, null).iterator().hasNext()) {
      return false;
    }
    //no need to check versions as versioning is not supported
    return true;
  }

  @Override
  public boolean bucketExists(String container) throws IOException {
    BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
    return bcc.exists();
  }

  @Override
  public void format(List<String> containers) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      ExecutorService tPool = Executors.newFixedThreadPool(bucketDeletionThreads);

      try {
        //delete hopsfs data from
        for (String containerStr : containers) {
          // check for versioning
          if (softDeletesEnabled && !isVersioningSupported(containerStr)) {
            throw new IOException("Cannot format file system. Versioning is enabled. However " +
                    "the container does not have DeleteRetentionPolicy set.");
          }

          emptyBucket(containerStr, true, tPool);
        }
      } catch (InterruptedException | ExecutionException e) {
        LOG.warn(e);
      } finally {
        tPool.shutdown();
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Format containers: " + Arrays.toString(containers.toArray())
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void createBucket(String containerStr) throws IOException {

    //create container
    new CloudActionHandler() {
      @Override
      public Object task() throws IOException {
        LOG.info("Creating container: " + containerStr);
        BlobContainerClient bcc = blobClient.getBlobContainerClient(containerStr);
        if (!bcc.exists()) {
          bcc.create();
        }
        return null;
      }
    }.performTask();

    //enable soft deletes
    if (softDeletesEnabled) {
      new CloudActionHandler() {
        @Override
        public Object task() throws IOException {
          LOG.info("Enabling soft deletes for storage");
          enableSoftDeletes();
          return null;
        }
      }.performTask();
    }
  }

  private void emptyBucket(final String container,
                           boolean onlyDeleteHopsFSData, ExecutorService tPool)
          throws IOException, ExecutionException, InterruptedException {
    final AtomicInteger deletedBlocks = new AtomicInteger(0);

    try {

      if (!bucketExists(container)) {
        throw new IOException("Cannot format file system. The container does not exist");
      }

      System.out.println("HopsFS-Cloud. Emptying container: " + container);

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      ListBlobsOptions lbo = new ListBlobsOptions();
      Iterable<PagedResponse<BlobItem>> blobPages = bcc.listBlobs(lbo, null).iterableByPage();

      for (PagedResponse<BlobItem> page : blobPages) {
        final List<Future> futures = new ArrayList<>();
        List<BlobItem> blobs = page.getValue();
        for (BlobItem item : blobs) {
          final String objectkey = item.getName();

          if (onlyDeleteHopsFSData && !objectkey.startsWith(CloudHelper.PREFIX_STR)) {
            continue;
          }

          Callable task = new Callable<Object>() {
            @Override
            public Object call() throws Exception {
              deleteObject(container, objectkey);
              String msg = "\rDeleted Blocks: " + (deletedBlocks.incrementAndGet());
              System.out.print(msg);
              return null;
            }
          };
          futures.add(tPool.submit(task));
        }
        for (Future future : futures) {
          future.get();
        }
      }

      // delete CID file
      BlobClient bc = bcc.getBlobClient(CloudHelper.CID_FILE);
      if (bc.exists()) {
        bc.delete();
      }

      System.out.println("");
    } catch (AzureException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void checkAllBuckets(List<String> containers) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      final int retry = 300;  // keep trying until the newly created container is available
      for (String contStr : containers) {
        LOG.debug("Checking container: " + contStr);
        boolean exists = false;
        BlobContainerClient bcc = null;
        for (int j = 0; j < retry; j++) {
          bcc = blobClient.getBlobContainerClient(contStr);
          if (!bcc.exists()) {
            //wait for a sec and retry
            try {
              Thread.sleep(1000);
            } catch (InterruptedException e) {
            }
            continue;
          } else {
            exists = true;
            break;
          }
        }

        if (!exists) {
          throw new IllegalStateException("Azure Container " + contStr + " needed for the " +
                  "file system does not exists");
        } else {
          //check the container is writable
          UUID uuid = UUID.randomUUID();
          File file1 = new File("/tmp/" + uuid);
          File file2 = new File("/tmp/" + uuid + ".downloaded");
          try {
            BlobClient bc = bcc.getBlobClient(uuid.toString()/*key*/);
            String message = "hello! hello! testing! testing! testing 1 2  3!";
            FileWriter fw = new FileWriter(file1);
            fw.write(message);
            fw.close();
            bc.uploadFromFile(file1.getAbsolutePath());
            bc.downloadToFile(file2.getAbsolutePath());
            bc.delete();
            assert FileUtils.contentEquals(file1, file2) == true;
          } catch (Exception e) {
            throw new IllegalStateException("Write test for Azure container: " + contStr +
                    " failed. " + e);
          } finally {
            file1.delete();
            file2.delete();
          }
        }
      }

      LOG.info("HopsFS-Cloud. Check all containers: " + Arrays.toString(containers.toArray())
              + " Time (ms): " + (System.currentTimeMillis() - startTime));
    } catch (AzureException e) {
      throw new IOException(e);
    }
  }

  @Override
  public int getPrefixSize() {
    return prefixSize;
  }

  @Override
  public void uploadObject(String container, String objectKey, File file,
                           Map<String, String> metadata) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      LOG.info("HopsFS-Cloud. Put Object. Bucket: " + container + " Object Key: " + objectKey +
              " Object Size: " + objectKey.length());

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);
      bc.uploadFromFile(file.getAbsolutePath(), true);
      bc.setMetadata(metadata);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Put Object. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in uploadObject. Container: " +
              container + " Key: " + objectKey + " File: " +
              file.getAbsolutePath() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public boolean objectExists(String container, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Obj Exists? Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return bc.exists();
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in objectExists. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public Map<String, String> getUserMetaData(String container, String objectKey)
          throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Get metadata. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return bc.getProperties().getMetadata();
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in getUserMetaData. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public long getObjectSize(String container, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Get obj size. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return bc.getProperties().getBlobSize();
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in getObjectSize. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void downloadObject(String container, String objectKey, File path) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      if (path.exists()) {
        path.delete();
      } else {
        //make sure that all parent dirs exists
        if (!path.getParentFile().exists()) {
          path.getParentFile().mkdirs();
        }
      }

      Random rand = new Random(System.currentTimeMillis());
      File tmpFile = new File(path.getAbsolutePath() + "." + rand.nextLong() + ".downloading");
      if (tmpFile.exists()) {
        tmpFile.delete();
      }

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);
      BlobProperties props = bc.downloadToFile(tmpFile.getAbsolutePath());

      tmpFile.renameTo(path);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Download obj. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (BlobStorageException e) {
      if (e.getErrorCode() == BlobErrorCode.BLOB_ARCHIVED &&
              e.getStatusCode() == 409) {
        String message = " The block has moved to ARCHIVE storage. Please restore the block to read the file";
        throw new BlockMovedToColdStorageException(message);
      }
      throw e;

    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in downloadObject Container: " +
              container + " ObjKey: " + objectKey + " File: " + path.getAbsolutePath() +
              " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  /*
   * only for testing
   */
  @VisibleForTesting
  @Override
  public Map<BlockIDAndGSTuple, CloudBlock> getAll(String prefix, List<String> containers) throws IOException {
    long startTime = System.currentTimeMillis();
    Map<BlockIDAndGSTuple, CloudBlock> blocks = new HashMap<>();
    for (String contStr : containers) {
      listContainer(contStr, prefix, blocks);
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Get all blocks. Containers: " + Arrays.toString(containers.toArray()) +
              " Prefix: " + prefix +
              " Total Blocks: " + blocks.size() +
              " Time (ms): " + (System.currentTimeMillis() - startTime));
    }
    return blocks;
  }

  @Override
  public List<String> getAllHopsFSDirectories(List<String> containers) throws IOException {
    try {
      long startTime = System.currentTimeMillis();
      List<String> dirs = new ArrayList<>();
      for (String container : containers) {
        PagedIterable<BlobItem> items =
                blobClient.getBlobContainerClient(container).listBlobsByHierarchy("");

        Iterator<BlobItem> itr = items.iterator();
        while (itr.hasNext()) {
          BlobItem item = itr.next();
          String key = item.getName();
          if (key.contains(CloudHelper.PREFIX_STR)) {
            dirs.add(key);
          } else {
            LOG.info("HopsFS-Cloud. Ignoring " + key + " directory. It is not HopsFS directory");
          }
        }
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Get all top level dirs. Containers: " + Arrays.toString(containers.toArray()) +
                " Total Dirs: " + dirs.size() +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return dirs;
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in getAllDirectories Container: " +
              Arrays.toString(containers.toArray()) + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private void listContainer(String container, String prefix, Map<BlockIDAndGSTuple, CloudBlock> result)
          throws IOException {
    try {
      Map<BlockIDAndGSTuple, CloudObject> blockObjs = new HashMap<>();
      Map<BlockIDAndGSTuple, CloudObject> metaObjs = new HashMap<>();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      ListBlobsOptions lbo = new ListBlobsOptions();
      lbo.setPrefix(prefix);

      Iterable<PagedResponse<BlobItem>> blobPages = bcc.listBlobs(lbo, null).iterableByPage();
      for (PagedResponse<BlobItem> page : blobPages) {
        List<BlobItem> blobs =  page.getValue();
        for(BlobItem item : blobs){
          String key = item.getName();

          CloudObject co = new CloudObject();
          co.setBucket(container);
          co.setKey(item.getName());
          co.setSize(item.getProperties().getContentLength());
          co.setLastModifiedTime(item.getProperties().getLastModified().toEpochSecond());

          BlockIDAndGSTuple idAndGS = CloudHelper.getIDAndGSFromKey(key);
          if (idAndGS != null && CloudHelper.isBlockFilename(key)) {
            blockObjs.put(idAndGS, co);
          } else if (idAndGS != null || CloudHelper.isMetaFilename(key)) {
            metaObjs.put(idAndGS, co);
          } else {
            LOG.warn("HopsFS-Cloud. File system objects are tampered. The " + key + " is not HopsFS object.");
          }
        }
      }

      CloudPersistenceProviderS3Impl.mergeMetaAndBlockObjects(metaObjs, blockObjs, result);
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in listContainer. Container: " +
              container + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void deleteObject(String container, String objectKey) throws IOException {
    try {
      deleteObjectInternal(container, objectKey);
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteObject. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  public void deleteObjectInternal(String container, String objectKey) throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);
      bc.delete();

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Delete Object. Container: " + container + " Object Key: " + objectKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (BlobStorageException e) {
      if (e.getErrorCode() == BlobErrorCode.SNAPSHOTS_PRESENT && e.getStatusCode() == 409) {
        LOG.warn("Unable to delete the object key: " + objectKey + " as it has snapshot. Retrying " +
                "delete includeing snapshots");
        deleteIncludingSnapshots(container, objectKey);
        return;
      }
      throw new IOException(e);
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteObjectInternal. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private void deleteIncludingSnapshots(String container, String objectKey) throws IOException {

    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient bcc = blobClient.getBlobContainerClient(container);
      BlobClient bc = bcc.getBlobClient(objectKey);
      bc.deleteWithResponse(DeleteSnapshotsOptionType.INCLUDE, new BlobRequestConditions(), null, null);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Delete Object including all snaphots. Container: " + container +
                " Object Key: " + objectKey + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteIncludingSnapshots. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  /*
  only for testing
   */
  @VisibleForTesting
  @Override
  public void renameObject(String srcContainer, String dstContainer, String srcKey, String dstKey)
          throws IOException {
    try {
      long startTime = System.currentTimeMillis();

      BlobContainerClient srcBcc = blobClient.getBlobContainerClient(srcContainer);
      BlobClient srcBc = srcBcc.getBlobClient(srcKey);
      UUID uuid = UUID.randomUUID();
      File file = new File("/tmp/" + uuid);
      srcBc.downloadToFile(file.getAbsolutePath());
      srcBc.delete();

      BlobContainerClient dstBcc = blobClient.getBlobContainerClient(dstContainer);
      BlobClient dstBc = dstBcc.getBlobClient(dstKey);
      dstBc.uploadFromFile(file.getAbsolutePath());
      file.delete();

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Rename Object. Src Container: " + srcContainer + " Src Object Key: " + srcKey +
                " Dst Container: " + dstContainer + " Dst Object Key: " + dstKey
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in renameObject  Src Container: " +
              srcContainer + " Dst Container: " + dstContainer + " SrcKey: " +
              srcKey + " DstKey: " + dstKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void copyObject(String srcContainer, String dstContainer, String srcKey, String dstKey,
                         Map<String, String> newObjMetadata)
          throws IOException {
    // For azure re-upload the block with new name to create a copy of the block
    // Azure supports BlobClientBase.copyFromUrl function to create copies but it has some
    // limitations. For example, the source must be a block blob no larger than 256MB.
    // The source must also be either public or have a sas token attached. The URL must be URL encoded.
    // In HopsFS the block size may be more than 256 MB.
    throw new UnsupportedOperationException("Azure does not suppor copy or rename operation");
  }

  @Override
  public long getPartSize() {
    return partSize;
  }

  @Override
  public int getXferThreads() {
    return maxThreads;
  }

  @Override
  public UploadID startMultipartUpload(String container, String objectKey,
                                       Map<String, String> metadata)
          throws IOException {
    if (LOG.isDebugEnabled()) {
      LOG.debug("HopsFS-Cloud. Starting Multipart Upload.");
    }
    return null;
  }

  @Override
  public PartRef uploadPart(String container, String objectKey, UploadID uploadID, int partNo,
                            File file, long startPos, long endPos) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      BlockBlobClient bbc = blobClient.getBlobContainerClient(container).
              getBlobClient(objectKey).getBlockBlobClient();

      String id64 = base64ID(partNo);
      FileInputStream fis = new FileInputStream(file);
      fis.skip(startPos);
      int len = (int) (endPos - startPos);
      byte[] bytes = new byte[len];
      int read = fis.read(bytes, 0, len);

      bbc.stageBlock(id64, new ByteArrayInputStream(bytes), read);
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Uploaded Part.  Container: " + container + " Object Key: " + objectKey
                + " PartID: " + id64 + " Part Size: " + len
                + " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
      return new AzurePartRef(id64);

    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in uploadPart. Container: " +
              container + " ObjKey: " + objectKey + " UploadID: " + uploadID.toString() +
              " Part No:" + partNo + " File: " + file.getAbsolutePath() +
              " Start Pos: " + startPos + " End Pos: " + endPos + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  private String base64ID(int id) {
    String id64 = new String(Base64.getEncoder().encode(String.format("%09d", id).getBytes()));
    return id64;
  }

  @Override
  public void finalizeMultipartUpload(String container, String objectKey, UploadID uploadID,
                                      List<PartRef> refs) throws IOException {
    long startTime = System.currentTimeMillis();
    try {
      List<String> ids = new ArrayList<String>();
      for (PartRef ref : refs) {
        ids.add(((AzurePartRef) ref).getId64());
      }

      BlockBlobClient bbc = blobClient.getBlobContainerClient(container).
              getBlobClient(objectKey).getBlockBlobClient();
      bbc.commitBlockList(ids);

      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Finalize Multipart Upload. Container: " + container +
                " Object Key: " + objectKey + " Parts: " + ids.size() +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in finalizeMultipartUpload. Container: " +
              container + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void abortMultipartUpload(String container, String objectKey,
                                   UploadID uploadID /*not needed for Azure*/)
          throws IOException {
    //https://docs.microsoft.com/en-us/dotnet/api/azure.storage.blobs.specialized.blockblobclient?view=azure-dotnet
    //azure will remove uncommitted blocks after one week.

    long startTime = System.currentTimeMillis();
    try {

      BlockBlobClient bbc = blobClient.getBlobContainerClient(container).
              getBlobClient(objectKey).getBlockBlobClient();
      bbc.delete();

    } catch (BlobStorageException e) {

      if (e.getMessage().contains("The specified blob does not exist")) {
        LOG.debug("HopsFS-Cloud. Multipart upload alreay aborted.");
        return;
      } else {
        throw e;
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in abortMultipartUpload. Container: " +
              container + " ObjKey: " + objectKey + " UploadID: " +
              uploadID.toString() + " Error: " + e.getMessage());
      throw new IOException(e);
    } finally {
      if (LOG.isDebugEnabled()) {
        LOG.debug("HopsFS-Cloud. Abort multipart upload. Container: " + container +
                " Object Key: " + objectKey +
                " Time (ms): " + (System.currentTimeMillis() - startTime));
      }

    }
  }

  @Override
  public List<ActiveMultipartUploads> listMultipartUploads(List<String> container, String prefix)
          throws IOException {
    throw new UnsupportedOperationException("Operation not supported for azure");
  }

  @Override
  public boolean restoreDeletedBlock(String container, String objectKey) throws IOException {
    try {
      blobClient.getBlobContainerClient(container).getBlobClient(objectKey).undelete();
      return true;
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in listMultipartUploads. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public boolean isVersioningSupported(String container) throws IOException {
    try {
      BlobServiceProperties props = blobClient.getProperties();
      return props.getDeleteRetentionPolicy().isEnabled();
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in isVersioningSupported. Container: " +
              container + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  public void enableSoftDeletes() throws IOException {
    try {

      int days = conf.getInt(DFSConfigKeys.AZURE_SOFT_DELETES_RETENTION_DAYS_KEY,
              DFSConfigKeys.AZURE_SOFT_DELETES_RETENTION_DAYS_DEFAULT);

      if (days < 7) {
        throw new RuntimeException("Number of retention days for soft deleted blocks must be >= 7");
      }

      BlobRetentionPolicy blobRetentionPolicy = new BlobRetentionPolicy();
      blobRetentionPolicy.setEnabled(true).setDays(days);
      BlobServiceProperties blobServiceProperties = new BlobServiceProperties();
      blobServiceProperties.setDeleteRetentionPolicy(blobRetentionPolicy);
      blobClient.setProperties(blobServiceProperties);
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in enableSoftDeletes." +
              " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void deleteAllVersions(String container, String objectKey) throws IOException {
    try {
      BlobClient bc =
              blobClient.getBlobContainerClient(container).getBlobClient(objectKey);
      if (bc.exists()) {
        bc.delete();
      }
    } catch (AzureException e) {
      LOG.info("HopsFS-Cloud: Exception in deleteAllVersions. Container: " +
              container + " ObjKey: " + objectKey + " Error: " + e.getMessage());
      throw new IOException(e);
    }
  }

  @Override
  public void deleteOldVersions(String bucket, String objectKey) throws IOException {
    // do nothing in case of azure as in azure we do not use versioning.
  }

  @Override
  public void shutdown() {
  }

  public BlobServiceClient getBlobClient() {
    return blobClient;
  }

  private abstract class CloudActionHandler {
    public abstract Object task() throws IOException;

    public Object performTask() throws IOException {
      List<Exception> exceptions = new ArrayList<>();
      long sleepTime = 500;
      for (int i = 0; i < 10; i++) {

        try {
          if (i != 0) {
            sleepTime = sleepTime * 2;
            LOG.info("HopsFS-Cloud. Operation Failed. Cause: " + exceptions.get(exceptions.size() - 1)
                    + " Retrying operation after " + sleepTime + " ms. Retry Count: " + i);
            Thread.sleep(sleepTime);
          }
        } catch (InterruptedException e) {
        }

        try {
          Object obj = task();
          return obj;
        } catch (AzureException e) {
          exceptions.add(e);
        }
      }

      for (Exception e : exceptions) {
        LOG.info("Supressed Exception", e);
      }
      throw new IOException(exceptions.get(0));
    }
  }

  @Override
  public Object getCloudClient() {
    return blobClient;
  }
}
